;*****************************************************************************        
;
;   Module:     1-wire.asm
;               
;   Author:     Mike Hibbett, mike.hibbett@gmail.com
;                                                                  
;   Version:    1.0 10/09/07                                                  
;
;               Provides various 1-wire functions
;
;*****************************************************************************


#include P16F917.inc

	extern  delay5us

	UDATA

; These are the internal variables
; and are not publically visible
result		RES 1
bitCount   	RES 1
txByte		RES 1
rxByte		RES 1
	
	CODE
	


;*****************************************************************************        
;
;   Function :  init1wireHw
;               Initialises the I/O port pins used by the hardware interface.
;               There is on one pin used, so not much work to do!
;               
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
	
init1wireHw	
	global init1wireHw
	; PortB.1 is our 1-wire interface.
	; Set it to it's default state: input.
	bsf		STATUS,RP0
	bcf		STATUS,RP1
	bsf		TRISB, 1
	bcf		STATUS,RP0
	
	return	
	
	
;*****************************************************************************        
;
;   Function :  reset1wire
; 				Perform a reset on the bus and return, in the W register
; 				an indication of whether one or more 1-wire devices
; 				are on the bus. A 0 indicates none, non zero indicates one 
; 				or more found.
;               
;   Input:      None
;
;   Output:     0/non zero in W
;
;*****************************************************************************        
reset1wire
	global reset1wire
	
	; drive wire low
	bsf		STATUS,RP0
	bcf		STATUS,RP1
	bcf		TRISB, 1
	bcf		STATUS,RP0
	bcf		PORTB, 1
	
	; delay for 480us
	movlw	D'480' / 5
	call	delay5us

	; release the wire
	bsf		STATUS,RP0
	bcf		STATUS,RP1
	bsf		TRISB, 1
	bcf		STATUS,RP0

	; result will hold the status of the device 
	; detected signal.
	clrf 	result
	
	; We should look for a presence pulse from a device
	; delay for 70us, the earliest a pulse could start
	movlw	D'70' / 5
	call	delay5us
	
	btfss	PORTB, 1
	incf	result, F
	
	; We delay a little longer and look again
	movlw	D'50' / 5
	call	delay5us
	
	btfss	PORTB, 1
	incf	result, F

	; Final delay to complete the 480us period during
	; which devices may be acknowledging their presence
	movlw	D'370' / 5
	call	delay5us
	
	; swap result and store it in W
	; Now, a zero in the W register indicate no device found
	swapf	result, W
	
	return 


;*****************************************************************************        
;
;   Function :  write0bit
; 				This is the lowest level software for communicating on the 
;               bus. It sets the I/O line low.
;               
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
write0bit	
	bsf		STATUS,RP0
	bcf		TRISB, 1
	bcf		STATUS,RP0
	bcf		PORTB, 1
	
	return


;*****************************************************************************        
;
;   Function :  write1bit
; 				This is the lowest level software for communicating on the 
;               bus. It sets the I/O line High.
;               
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
write1bit
	bsf		STATUS,RP0
	bsf		TRISB, 1
	bcf		STATUS,RP0
	
	return
	
	
;*****************************************************************************        
;
;   Function :  read1wire
; 				This is the lowest level software for communicating on the 
;               bus. It reads a data bit from a remote device.
;               
;   Input:      None
;
;   Output:     Data bit in LSB of the result variable
;
;*****************************************************************************        
read1wire
	call	write0bit
	movlw	0x01
	call	delay5us
	call	write1bit
	
	clrf 	result
	btfsc	PORTB, 1
	bsf		result, 0
	movlw	0x01
	call	delay5us
	btfsc	PORTB, 1
	bsf		result, 0
	
	return		


;*****************************************************************************        
;
;   Function :  write8wire
; 				This is the basic routine, available externally to other 
;               routines, that writes a byte to the bus.
;               
;   Input:      byte to send in W
;
;   Output:     None.
;
;*****************************************************************************        
write8wire
	global write8wire
	
	movwf	txByte
	movlw	0x08
	movwf	bitCount
	
w8w001	
	; Send tx bit slot marker
	call	write0bit
	movlw	0x01
	call	delay5us
	

	rrf		txByte, F
	btfsc	STATUS, C
	call	write1bit

	; Delay for rest of timeslot
	movlw	0x17
	call	delay5us
	
	; Restore bus to normal
	call	write1bit
	
	decfsz	bitCount, F
	goto	w8w001
	
	return
	
	
	
;*****************************************************************************        
;
;   Function :  read8wire
; 				This is the basic routine, available externally to other 
;               routines, that reads a byte from a remote device.
;               
;   Input:      None.
;
;   Output:     Byte read in the W register
;
;*****************************************************************************        
read8wire
	global read8wire
	
	movlw	0x08
	movwf	bitCount
	
r8w001	
	; Request a single bit from the remote device
	call	read1wire
	
	; Store the databit in the rxByte buffer
	rrf		result, W
	rrf		rxByte, F
	
	; Loop until all 8 bits received.
	decfsz	bitCount, F
	goto	r8w001
	
	; return the byte in the W register
	movfw	rxByte	
		
	return	
	
		
	END