/****************************************************************************
*
*   Module:     vidgen_drv.cpp
*   Author:     Mike Hibbett
*   Date:       20/4/12
*   Purpose:    Implementation of the video driver code for the Uno32. 
*               This code will drive an RBG monitor at a resolution of 800x600
*               and provide a monochrome text based interface to it. Pixel level 
*               addressing is not supported due to limited RAM and processing power.
*
*               Limited graphics support, such as rendering a small monochrome 
*               image may be possible through additional software
*
****************************************************************************/

/**** INCLUDES *************************************************************/

#define OPT_SYSTEM_INTERNAL
#define OPT_BOARD_INTERNAL
#include "vidgen_drv.h"
#include <p32xxxx.h>
#include <plib.h>
#include <WProgram.h> 
#include <inttypes.h>
#include <pins_arduino.h>
#include "font\lcd_font.h"

/**** CONSTANTS ************************************************************/

// Could improve on this by having tmr3 expire on the falling edge of the horizintal sync pulse. Then, I'd only need to do 88 + 800 + 1 pixels ( last one to ensure level is low )
#define NUM_VID_LONGS 14  //  We tx 896 bits. The line is 1056 pixels long. therefore we have 4.6us before next line
#define NUM_LINES 8  // Number of lines in each buffer. 8, to match the font size.

#define SCREEN_HEIGHT_INTERNAL  VIDGEN_SCREEN_DEPTH
#define SCREEN_WIDTH_INTERNAL   (VIDGEN_SCREEN_WIDTH+4)    // Four extra bytes reserved for the video blanking period

#define TEXT_BUFFER_WIDTH_START 2   // The first visible character in the text buffer ( First two are reserved )

/**** GLOBAL VARIABLES *****************************************************/

// Frame buffers. A is the first buffer used, then B.

volatile unsigned char textBuffer[SCREEN_HEIGHT_INTERNAL][SCREEN_WIDTH_INTERNAL];
volatile unsigned long vid_line0[NUM_VID_LONGS+1];  // one extra to allow for simple buffer filling loop
volatile unsigned long vid_line1[NUM_VID_LONGS+1];  // one extra to allow for simple buffer filling loop

volatile unsigned char buffInUse = 0;   // Which of the two frame buffers we are using.
volatile unsigned char buffLong = 0;    // the long word we are referencing in the buffer

volatile unsigned long *vid_line = &vid_line0[0];
volatile unsigned int  cnt = 0;
volatile unsigned char frameDone = 0;


/****************************************************************************
*
*   Function:   fillBuff
*   inputs:     buff        Which of the two buffers to fill
*               textline    Which line of text to transfer
*               scanline    Which line of font data to transfer
*
*   returns:    Nothing
*   Purpose:    For a given line of the text buffer, and for a given index into the
*               character font, create a display line of pixel data.
*               This is the most critical function as it must convert a line of text
*               into a "strip" of pixels very quickly, before the next line of the
*               display is to be generated. This explains the rather odd repeating
*               sequence of code. Doing this in a loop would be too slow.
*               This routine is an ideal candidate for hand optimising in assembly.
*
****************************************************************************/
void fillBuff( unsigned char buff, unsigned char textline, unsigned char scanline )
{
  unsigned char lp;
  unsigned char fval;
  volatile unsigned char *tptr;
  volatile unsigned long *vptr;
  const unsigned char *fptr;
  
    // Get a pointer to the current line of text to be converted.
    tptr = textBuffer[textline];
    // Get a pointer to the current row of the line's pixels to be processed
    fptr = font6x8[scanline];

    // We have two pixel buffers; one is currently being drawn to screen, the other
    // we populate in advance. Point to the inactive one    
    if ( buff ) 
      vptr = &vid_line0[1];
    else
      vptr = &vid_line1[1];
      
    // Now for the fun bit. vid_line stores a row of pixels in 32 bit chunks. Each text character is 6 pixels wide,
    // So we need to extra those six pixels and pack them into the current 32 bit word. 6 into 32 doesn't fit well,
    // hence the four blocks of similar (but different) code.
    for (lp=1; lp<13; lp+=3 ) {
      fval = fptr[*tptr++];
      *vptr = fval<<24;
      fval = fptr[*tptr++];
      *vptr |= fval<<18;
      fval = fptr[*tptr++];
      *vptr |= fval<<12;
      fval = fptr[*tptr++];
      *vptr |= fval<<6;
      fval = fptr[*tptr++];
      *vptr |= fval;
      fval = fptr[*tptr++];
      *vptr++ |= fval>>6;

      *vptr = fval<<26;
      fval = fptr[*tptr++];
      *vptr |= fval<<20;
      fval = fptr[*tptr++];
      *vptr |= fval<<14;
      fval = fptr[*tptr++];
      *vptr |= fval<<8;
      fval = fptr[*tptr++];
      *vptr |= fval<<2;
      fval = fptr[*tptr++];
      *vptr++ |= fval>>4;

      *vptr = fval<<28;
      fval = fptr[*tptr++];
      *vptr |= fval<<22;
      fval = fptr[*tptr++];
      *vptr |= fval<<16;
      fval = fptr[*tptr++];
      *vptr |= fval<<10;
      fval = fptr[*tptr++];
      *vptr |= fval<<4;
      fval = fptr[*tptr++];
      *vptr++ |= fval>>2;    
    }   
    
    fval = fptr[*tptr++];
    *vptr = fval<<24;
    fval = fptr[*tptr++];
    *vptr |= fval<<18;
    fval = fptr[*tptr++];
    *vptr |= fval<<12;
    fval = fptr[*tptr++];
    *vptr |= fval<<6;
}


/****************************************************************************
*
*   Function:   T3_IntHandler
*   inputs:     None
*   returns:    Nothing
*   Purpose:    Interrupt routine for the Timer3 module. 
*               This occurs once for each scan line ( horizontal sync pulse,)
*               and we use it to determine when a new frame is starting, to generate
*               the slow frame sync pulse and to trigger the write of displat pixels              
*
****************************************************************************/
extern "C"
{
void __ISR(_TIMER_3_VECTOR, _T3_IPL_ISR) T3_IntHandler (void)
{
  int lp;
  unsigned char fval;
  volatile unsigned char *tptr;
  
  cnt++;
  
  IFS0CLR = 0x1000; // Clear timer interrupt status flag

  if ( cnt == 1 ) {
    LATD |= 0x4;
    
  } else if ( cnt == 5 ) {
    LATD &= ~0x4;  
    
    // Beginning of a frame, so set up the first vid_line.
    fillBuff(0,0,0);
  } else if ( cnt == 620 ) {
     // Indicate to the main application that the display text buffer is free to be updated
     // as we have finished transferring the text buffer data to the display 
     frameDone = 1; 
  }else if ( cnt == 629 ) {
    
    // Starting a new frame
    cnt = buffLong = buffInUse = 0;    
  } else if ( (cnt >=28 )&&(cnt <=619 ) ) {
    
    // start the spi transfer when in the display range
    SPI1BUF = vid_line[buffLong];
    buffLong++;
    
    fillBuff(buffInUse,(cnt-28)>>4,((cnt-28)>>1) & 0x07);    
  }
}


/****************************************************************************
*
*   Function:   __SPI1Interrupt
*   inputs:     None
*   returns:    Nothing
*   Purpose:    Interrupt routine for the SPI module. 
*               This occurs when the SPI module is ready to receive a new word for
*               transmission. This interrupt routine 'feeds' the SPI module, 
*               ensuring that it always has data to send.              
*
****************************************************************************/
void __ISR(_SPI_1_VECTOR, ipl7) __SPI1Interrupt(void)
{ 
  IFS0CLR = 0x03800000;
  // Transmit the rest of the line
  if ( buffLong < NUM_VID_LONGS ) {
   
    SPI1BUF = vid_line[buffLong];
    buffLong++;
  } else {
    
    if ( ((cnt & 0x03) == 1) || ((cnt & 0x03) == 3) ) {
      if ( buffInUse )
        vid_line = &vid_line0[0];
      else
        vid_line = &vid_line1[0];
      buffInUse = ~buffInUse;
    }
    buffLong = 0;    
  }
}
} // End of extern C


/****************************************************************************
*
*   Function:   vidgen_cls
*   inputs:     None
*   returns:    Nothing
*   Purpose:    Clears the display
*
****************************************************************************/
void vidgen_cls( void )
{
    memset((void *)textBuffer, ' ', sizeof(textBuffer));
}


/****************************************************************************
*
*   Function:   vidgen_putc
*   inputs:     ch  character to print
*               x   column position
*               y   row position
*   returns:    Nothing
*   Purpose:    displays the character ch at position x,y. Top left corner origin
*
****************************************************************************/
void vidgen_putc( char ch, unsigned char x, unsigned char y )
{
    if ( (x < SCREEN_WIDTH) && (y < SCREEN_DEPTH) )
        textBuffer[y][TEXT_BUFFER_WIDTH_START + x] = ch;   
}


/****************************************************************************
*
*   Function:   vidgen_getc
*   inputs:     x   column position
*               y   row position
*   returns:    The character at that location
*   Purpose:    Returns the character in the text buffer at the specified position
*
****************************************************************************/
char vidgen_getc( unsigned char x, unsigned char y )
{
    char ret_val;
    
    if ( (x < SCREEN_WIDTH) && (y < SCREEN_DEPTH) )
        ret_val = textBuffer[y][TEXT_BUFFER_WIDTH_START + x];   
    else 
        ret_val = 0;  // Invalid position - return NULL character
        
    return ret_val;
}


/****************************************************************************
*
*   Function:   vidgen_puts
*   inputs:     str string to display
*               x   column position
*               y   row position
*   returns:    Nothing
*   Purpose:    displays the string str at position x,y. Top left corner origin
*               If the string extends beyound the end of the line, those
*               extra characters are not displayed
*
****************************************************************************/
void vidgen_puts( char *str, unsigned char x, unsigned char y )
{
    while (str && *str) {
        if ( (x < SCREEN_WIDTH) && (y < SCREEN_DEPTH) ) {
            textBuffer[y][TEXT_BUFFER_WIDTH_START + x] = *str;   
        }
        x++;
        str++;
    }
}


/****************************************************************************
*
*   Function:   vidgen_setup
*   inputs:     None
*   returns:    Nothing
*   Purpose:    The setup routine called by the MPIDE 'sketch' to start the video              
*               features.
*
****************************************************************************/
void vidgen_setup( void )
{
  unsigned int rData; // Used to read data from SPI peripheral
  
  
  // Clear the display buffer and scan line buffers
  memset((void *)textBuffer, ' ', sizeof(textBuffer));
  memset((void *)vid_line0, 0x0, sizeof(vid_line0)); 
  memset((void *)vid_line1, 0x0, sizeof(vid_line1)); 
  
  // Configure the SPI peripheral for 32 bit word and a 20MHz clock rate.
  // This is our pixel data generator
  U1MODECLR =0x8000;
  SPI1CON = 0;        // Stops and resets the SPI1.
  TRISBSET = 0x4;     // Set RB2 as a digital input
  TRISDCLR = 0x04;    // Vertical Sync pin
  TRISFCLR = 0x08;     // Set pin RF3 as a digital output For SPI output (pixel data)
  AD1PCFGSET = 0x4;   // Analog input pin in digital mode
  rData=SPI1BUF;      // clears the receive buffer
  IFS0CLR=0x03800000; // clear any existing event
  IPC5CLR=0x1f000000; // clear the priority
  IPC5SET=0x1f000000; // Set IPL=3, subpriority 1
  IEC0SET=0x01000000;  // enable tx ints
  SPI1BRG=0x1; // divide pb clock down by 4 ( default is 2 )
  SPI1STATCLR=0x40;   // clear the Overflow
  SPI1CON= 0xc0000C20; // 11100000 00000010 10000000 00100000
  SPI1CONSET= 0x8000; // turn on module

  
  // Configure Output Compare peripheral OC1 
  // Connects to pin 3 on chipKit Uno32 header J5
  OC1CON = 0x0d;    // Setup is 16 bit, using Timer3, +ve edge
  OC1R =   0;       // Positive edge at count 0
  OC1RS =  256;     // Falling edge 256 counts later
  
  // Timer 3 setup, to support OC1
  PR3 = 2111;       // Timer will restart after 2111 counts
  T3CON = 0;        // Disable timer while configuring it
  TMR3 = 0;
  
  // Enable an interrupt to occur on each line. We will
  // use this later to help generate the vertical sync
  IFS0CLR = 0x1000; // Clear timer interrupt status flag
  IEC0SET = 0x1000; // Set interrupt enable on Tmr3
  
  // Specify the interrupt priority
  IPC3CLR = 0x1F;   
  IPC3SET = (_T3_IPL_IPC << 2) | _T3_SPL_IPC;
  
  // Start the timer and output compare peripheral
  T3CONSET =  0x8000;
  OC1CONSET = 0x8000;  
}
