///////////////////////////////////////////////////////////////////
//
//	Project: 	Motor2
//	File:  		main.c
//	Author:		Darren Wenn
//	
//
///////////////////////////////////////////////////////////////////
/******************************************************************
CODE OWNERSHIP AND DISCLAIMER OF LIABILITY

Microchip Technology Incorporated ("Microchip") retains all ownership and 
intellectual property rights in the code accompanying this message and in all 
derivatives hereto.  You may use this code, and any derivatives created by any 
person or entity by or on your behalf, exclusively with Microchips proprietary 
products.  Your acceptance and/or use of this code constitutes agreement to the 
terms and conditions of this notice.

CODE ACCOMPANYING THIS MESSAGE IS SUPPLIED BY MICROCHIP 
"AS IS".  NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR 
STATUTORY, INCLUDING, BUT NOT LIMITED TO, IMPLIED 
WARRANTIES OF NON-INFRINGEMENT, MERCHANTABILITY 
AND FITNESS FOR A PARTICULAR PURPOSE APPLY TO THIS CODE, 
ITS INTERACTION WITH MICROCHIPS PRODUCTS, COMBINATION 
WITH ANY OTHER PRODUCTS, OR USE IN ANY APPLICATION. 

YOU ACKNOWLEDGE AND AGREE THAT, IN NO EVENT, SHALL 
MICROCHIP BE LIABLE, WHETHER IN CONTRACT, WARRANTY, 
TORT (INCLUDING NEGLIGENCE OR BREACH OF STATUTORY DUTY), 
STRICT LIABILITY, INDEMNITY, CONTRIBUTION, OR OTHERWISE, 
FOR ANY INDIRECT, SPECIAL, PUNITIVE, EXEMPLARY, INCIDENTAL
 OR CONSEQUENTIAL LOSS, DAMAGE, FOR COST OR EXPENSE OF 
ANY KIND WHATSOEVER RELATED TO THE CODE, HOWSOEVER 
CAUSED.  

You agree that you are solely responsible for testing the code and determining its 
suitability. Microchip has no obligation to modify, test, certify, or support the code.
******************************************************************/

#include <p30fxxxx.h>
#include <dsp.h>
#include <stdio.h>


// primary oscillator with FRC 16X PLL
_FOSC(CSW_FSCM_OFF & FRC_PLL8);
_FWDT(WDT_OFF);
_FBORPOR(PBOR_OFF & MCLR_EN & PWMxH_ACT_HI & PWMxL_ACT_HI);
_FGS(CODE_PROT_OFF & GWRP_OFF);

// define the processor and PWM frequency
#define FCY (7372800L * 2L)		// xtal = 7.3728MHz with 8X PLL
#define FREQ_PWM 20000

// PWM period 1474 counts
#define PWM_PERIOD 		((FCY / FREQ_PWM) - 1)

// baud rate for comms
#define BAUD_RATE ((FCY / (16L * 57600L)) - 1)

// macros to allow the module to be rapidly turned on and off
// using only channels 1 and 2
#define DISABLE_FIRING (PWMCON1 = 0x0000)
#define ENABLE_FIRING  (PWMCON1 = 0x0111)

// sampling interval to measure the back EMF ~175Hz
#define SAMPLE_INTERVAL 1000

// current values of the Back EMF and Pot
volatile fractional BackEMF;
volatile fractional vPot;

// flag to indicate new sample pair are ready for processing
volatile unsigned int bSampleReady;

// filter 107 tap kaiser, 175Hz ts 4Hz PB, 10Hz SB, 1db PB 60db SB
extern FIRStruct lpfFilter;
fractional lpfOutput;

// PID controller data structure
fractional abcCoefficient[3] __attribute__ ((section (".xbss, bss, xmemory")));
fractional controlHistory[3] __attribute__ ((section (".ybss, bss, ymemory")));
tPID motorPID;
fractional coeffPID[3];

// this symbol is defined in stdio and allows us to select
// UART2 as the default output for printf
extern int __C30_UART;

///////////////////////////////////////////////////////////////////
//
//	void InitPWM(void)
//
//	Initialise the PWM module to drive the motor
//
///////////////////////////////////////////////////////////////////
void InitPWM(void)
{
	// ensure the module is turned off and accept defaults for
	PTCON = 0x0000;	
	// set the timebase period
	PTPER = PWM_PERIOD;	
	PWMCON1 = 0x0000;	
	// set PWMCON2 for synchronized updates
	PWMCON2 = 0x0000;
	
	// turn on PWM1L and PWM modulate PWM1H
	OVDCON = 0x0201;
	// set the initial duty cycle
	PDC1 = 0;	
	// turn PWM time base on
	PTCONbits.PTEN = 1;
}

///////////////////////////////////////////////////////////////////
//
//	void InitADC(void)
//
//	Initialise the ADC module to sample the Back EMF and Pot
//
///////////////////////////////////////////////////////////////////
void InitADC(void)
{
	// set up the ADC to sample the output signal
	ADCON1bits.SSRC = 0; 	// trigger manually
	ADCON1bits.SIMSAM = 1; 	// enable simultaneous sampling
	ADCON1bits.ASAM = 1;
	ADCON1bits.FORM = 0; 	// output data in raw format
	
	ADCON2bits.CSCNA = 0;	// do not scan inputs
	ADCON2bits.CHPS = 1;	// 2 S/H channels per sample
	ADCON2bits.SMPI = 0;	// interrupt after sample sequence
	
	// Sampling rate is:
	// Tcy = 33.908ns
	// Tad = Tcy(ADCS + 1) / 2 = 356ns
	// Total time = (12 + 18) * Tad = 10.68us
	// sampling rate = 93624 Hz 
	// 2 channels so max sample rate is = 46812 Hz
	ADCON3bits.SAMC = 18;  
	ADCON3bits.ADCS = 20;  
	
	// CH0 = Back EMF, CH1 = Pot
	ADCHSbits.CH123SA = 0; // AN0 fed into CH1
	ADCHSbits.CH0SA = 3; // CH0 = AN3
	
	ADPCFG = 0xFFF0;	// AN0-1 analogue inputs
	ADCSSL = 0x0000;
	
	// enable ADC interrupts
	IFS0bits.ADIF = 0;	
	IEC0bits.ADIE = 1;
	// turn the module on
	ADCON1bits.ADON = 1;	
}


int main(void)
{
	long dutyCycle;
	
	// set the TRIS register for the MCPWM module
	TRISE = 0xFFF8;	
	
	// set up the UART for 57600 8-N-1 using UART 2
	__C30_UART = 2;
	TRISFbits.TRISF5 = 0;
	U2BRG = BAUD_RATE;
	U2MODEbits.UARTEN = 1;
	U2STAbits.UTXEN = 1;
	
	// clear the FIR filter delay buffer
	FIRDelayInit(&lpfFilter);
	// intialise the PID within the locations of the coefficients and history
    motorPID.abcCoefficients = &abcCoefficient[0];  
    motorPID.controlHistory = &controlHistory[0]; 
    // clear the history buffer 
	PIDInit(&motorPID);	
	coeffPID[0] = Q15(0.65);
	coeffPID[1] = Q15(0.05);
	coeffPID[2] = Q15(0.0);

	// precalculate the PID coefficients
	PIDCoeffCalc(coeffPID, &motorPID);
	
	// set up TMR2 to interrupt and start a measurement sequence
	T2CONbits.TCKPS = 2; // 1/64
	PR2 = SAMPLE_INTERVAL;
	IFS0bits.T2IF = 0;
	IEC0bits.T2IE = 1;
	T2CONbits.TON = 1;
	
	// T3 is used for the speed control loop
	T3CONbits.TCKPS = 3;
	PR3 = 10000;
	IFS0bits.T3IF = 0;
	T3CONbits.TON = 1;

	InitPWM();
	InitADC();
	PWMCON1 = 0x0111;
		
	printf("\r\nProgram Started\r\n");	
	while (1) {
		bSampleReady = 0;
		// wait for a new sample to come in
		while (!bSampleReady);
		// filter the incoming data to provide a noise free signal
		FIR(1, &lpfOutput, (fractional*) &BackEMF, &lpfFilter);
		
		// the control loop must operate slower than the sampling loop
		// typically 100Hz should be fine
		if (IFS0bits.T3IF == 1) {
			// we take the last sample as the output of the system (the motor)
			motorPID.measuredOutput = lpfOutput;
			// update the reference value from the pot
			motorPID.controlReference = vPot;
			// calculate the new control output
			PID(&motorPID);
			
			dutyCycle = (long) motorPID.controlOutput;
			dutyCycle *= 1480L;
			dutyCycle /= 1023L;
			
			// apply some limits to prevent negative values being written
			// and also prevent integral wind-up.
			if (dutyCycle > 1600)
				dutyCycle = 1600;
			
			if ((motorPID.controlReference == 0) || (dutyCycle <= 0))
				PDC1 = 0;
			else
				PDC1 = (int) dutyCycle;
			
			// for diagnostic purposes print out the control points			
			printf("%8d %8d %8d\r\n", 
				motorPID.controlReference, 
				motorPID.controlOutput,
				motorPID.measuredOutput);
			
			IFS0bits.T3IF = 0;	
		}
	}	
	
	return 1;	
}

///////////////////////////////////////////////////////////////////
//
//	void _ADCIntterupt
//
//	Process the ADC interrupt 
//
///////////////////////////////////////////////////////////////////
void __attribute__((__interrupt__, auto_psv)) _ADCInterrupt(void)
{	
	long bEMF;
	
	// stop ADC sampling
	ADCON1bits.SSRC = 0;
	// get the BackEMF values and convert to fractionals applying scaling
	bEMF = ADCBUF0;
	bEMF *= 1023L;
	bEMF /= 850L;
	BackEMF = bEMF;	
	// only take one Pot value reading
	vPot = ADCBUF1;
	
	// indicate a sample is ready
	bSampleReady = 1;	
	// turn off the indicator led
	LATEbits.LATE2 = 0;
	// clear the interrupt
	IFS0bits.ADIF = 0;	
}

///////////////////////////////////////////////////////////////////
//
//	void _T2Intterupt
//
//	Process the Timer 2 interrupt and manage Back-EMF sampling
//
///////////////////////////////////////////////////////////////////
void __attribute__((__interrupt__, auto_psv)) _T2Interrupt(void)
{	
	static enum {
		IDLE,
		HOLD_OFF,
		MEASURE	
	} tState = IDLE;
	
	switch (tState) {
		case IDLE:	
			LATEbits.LATE2 = 0;
			PR2 = 250;			// delay while BEMF settles
			OVDCON = 0x0001;
			tState = HOLD_OFF;
			break;
		case HOLD_OFF:
			LATEbits.LATE2 = 1;
			PR2 = 40;			// allow time for ADC
			OVDCON = 0x0001;
			ADCON1bits.SSRC = 7;
			tState = MEASURE;
			break;
		case MEASURE:
			tState = IDLE;
			OVDCON = 0x0201;
			PR2 = SAMPLE_INTERVAL;	// return to normal
			break;	
	}
	
	// clear the TMR2 interrupt
	IFS0bits.T2IF = 0;
}

