/*******************************************************************************
  Company:
    Microchip Technology Inc.

  File Name:
    main_fft_example.c

  Summary:
    This file is used to call the FFT related functions.

  Description:
    This file includes the function calls for FFTComplexIP, BitReverseComplex,
    SquareMagnitudeCplx functions that are central to the computation of the FFT
    of an input signal and then the peak frequency is calculated.
*******************************************************************************/
/*******************************************************************************
Copyright (c) 2012 released Microchip Technology Inc.  All rights reserved.

Microchip licenses to you the right to use, modify, copy and distribute
Software only when embedded on a Microchip microcontroller or digital signal
controller that is integrated into your product or third party product
(pursuant to the sublicense terms in the accompanying license agreement).

You should refer to the license agreement accompanying this Software for
additional information regarding your rights and obligations.

SOFTWARE AND DOCUMENTATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
EITHER EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION, ANY WARRANTY OF
MERCHANTABILITY, TITLE, NON-INFRINGEMENT AND FITNESS FOR A PARTICULAR PURPOSE.
IN NO EVENT SHALL MICROCHIP OR ITS LICENSORS BE LIABLE OR OBLIGATED UNDER
CONTRACT, NEGLIGENCE, STRICT LIABILITY, CONTRIBUTION, BREACH OF WARRANTY, OR
OTHER LEGAL EQUITABLE THEORY ANY DIRECT OR INDIRECT DAMAGES OR EXPENSES
INCLUDING BUT NOT LIMITED TO ANY INCIDENTAL, SPECIAL, INDIRECT, PUNITIVE OR
CONSEQUENTIAL DAMAGES, LOST PROFITS OR LOST DATA, COST OF PROCUREMENT OF
SUBSTITUTE GOODS, TECHNOLOGY, SERVICES, OR ANY CLAIMS BY THIRD PARTIES
(INCLUDING BUT NOT LIMITED TO ANY DEFENSE THEREOF), OR OTHER SIMILAR COSTS.
*******************************************************************************/
// *****************************************************************************
// *****************************************************************************
// Section: Included Files
// *****************************************************************************
// *****************************************************************************

#include <xc.h>
#include <stdint.h>
#include <dsp.h>
#include "fft.h"
#include "ili9341.h"
#include <string.h>
#include <stdio.h>

#if __XC16_VERSION < 1011
    #warning "Please upgrade to XC16 v1.11 or newer."
#endif

// *****************************************************************************
// *****************************************************************************
// Section: Macros for Configuration Fuse Registers:
// *****************************************************************************
// *****************************************************************************
// Invoke macros to set up  device configuration fuse registers.The fuses will
//   select the oscillator source, power-up timers, watch-dog timers etc. The
//   macros are defined within the device header files. The configuration fuse
//   registers reside in Flash memory.
 

// DSPIC33EP256GP506 Configuration Bit Settings

// FICD
#pragma config ICS = PGD1               // ICD Communication Channel Select bits (Communicate on PGEC1 and PGED1)
#pragma config JTAGEN = OFF             // JTAG Enable bit (JTAG is disabled)

// FPOR
#pragma config ALTI2C1 = OFF            // Alternate I2C1 pins (I2C1 mapped to SDA1/SCL1 pins)
#pragma config ALTI2C2 = OFF            // Alternate I2C2 pins (I2C2 mapped to SDA2/SCL2 pins)
#pragma config WDTWIN = WIN25           // Watchdog Window Select bits (WDT Window is 25% of WDT period)

// FWDT
#pragma config WDTPOST = PS32768        // Watchdog Timer Postscaler bits (1:32,768)
#pragma config WDTPRE = PR128           // Watchdog Timer Prescaler bit (1:128)
#pragma config PLLKEN = ON              // PLL Lock Enable bit (Clock switch to PLL source will wait until the PLL lock signal is valid.)
#pragma config WINDIS = OFF             // Watchdog Timer Window Enable bit (Watchdog Timer in Non-Window mode)
#pragma config FWDTEN = OFF             // Watchdog Timer Enable bit (Watchdog timer enabled/disabled by user software)

// FOSC
#pragma config POSCMD = NONE            // Primary Oscillator Mode Select bits (Not used)
#pragma config OSCIOFNC = ON            // OSC2 Pin Function bit (OSC1&2 are digital I/O)
#pragma config IOL1WAY = OFF            // Peripheral pin select configuration (Allow multiple reconfigurations)
#pragma config FCKSM = CSDCMD           // Both Clock switching and Fail-safe Clock Monitor are disabled

// FOSCSEL
#pragma config FNOSC = FRCPLL           // Fast RC Oscillator with divide-by-N with PLL module (FRCPLL)
#pragma config IESO = OFF               // Two-speed Oscillator Start-up Enable bit (Start up with user-selected oscillator source)

// FGS
#pragma config GWRP = OFF               // General Segment Write-Protect bit (General Segment may be written)
#pragma config GCP = OFF                // General Segment Code-Protect bit (General Segment Code protect is Disabled)

// *****************************************************************************
// *****************************************************************************
// Section: /* Extern definitions */
// *****************************************************************************
// *****************************************************************************
 // Typically, the input to an FFT  is a complex array containing samples
 // of an input signal. For this example, we will provide the input signal in an
 // array declared in Y-data space.
 fractcomplex sigCmpx[FFT_BLOCK_LENGTH] __attribute__ ((eds, space(ymemory), aligned (FFT_BLOCK_LENGTH * 2 *2)));

 /* Declare Twiddle Factor array in X-space*/
 #ifndef FFTTWIDCOEFFS_IN_PROGMEM
 fractcomplex twiddleFactors[FFT_BLOCK_LENGTH/2] 	
 __attribute__ ((section (".xbss, bss, xmemory"), aligned (FFT_BLOCK_LENGTH*2)));
 #else
 extern const fractcomplex twiddleFactors[FFT_BLOCK_LENGTH/2]	/* Twiddle Factor array in Program memory */
 __attribute__ ((space(prog), aligned (FFT_BLOCK_LENGTH*2)));
 #endif

// *****************************************************************************
// *****************************************************************************
// Section: File Scope or Global Constants
// *****************************************************************************
// *****************************************************************************
fractional output[FFT_BLOCK_LENGTH/2];
int16_t	peakFrequencyBin = 0; // Declare post-FFT variables to compute the
uint32_t peakFrequency = 0; // frequency of the largest spectral component


volatile fractcomplex adcBuff[FFT_BLOCK_LENGTH];
volatile uint16_t bufferIndex = 0;
volatile uint16_t bufferFull = 0;
uint32_t sampleRate = 3;
volatile uint8_t HOLDPressed = 0;
volatile uint8_t SPANPressed = 0;
volatile uint8_t HOLDActive = 0;


typedef struct {
    uint16_t PR1Val;
    uint16_t rate;
    char *text;
} samp;


#define NUM_SAMPLERATES 6

// TODO: we could add another parameter, setting a prescaler, to allow us to go to lower frequencies
samp samplerates[NUM_SAMPLERATES] = {
    {64400, 1000,   "500Hz"},
    {32200, 2000,   " 1KHz"},
    {16100, 4000,   " 2KHz"},
    {6440,  10000,  " 5KHz"},
    {3220,  20000,  "10KHz"},
    {1610,  40000,  "20KHz"}
};


// Interrupt routine for the ADC conversion complete interrupt
void __attribute__((interrupt(auto_psv))) _AD1Interrupt(void)
{
    fractional val;
    
    val = ADC1BUF0;
    
    if (!bufferFull) {
        adcBuff[bufferIndex].real = val >> 1;
        adcBuff[bufferIndex++].imag = 0;

        if (bufferIndex == FFT_BLOCK_LENGTH) {
            bufferIndex = 0;
            bufferFull = 1;
            // Stop any further ADC conversions.
            IEC0bits.T1IE = 0; 
            // Copy the buffer of readings into the FFT buffer
            // TODO: Don't need a separate buffer now.
            memcpy(sigCmpx, (const void *)adcBuff, sizeof(sigCmpx));
        }
    }
    IFS0bits.AD1IF = 0;
}



// Interrupt routine button press detection, either buttons
void __attribute__((__interrupt__, no_auto_psv)) _CNInterrupt(void)
{
	IFS1bits.CNIF = 0;

    // Check both buttons, start the timer if the button is not already pressed
    
    if ((PORTBbits.RB5 == 0) && (!HOLDPressed)) {
        // HOLD button pressed - start Timer2
        PR2 = 13670; // 50ms timer
        TMR2 = 0;
        T2CON = 0x8030;
        IFS0bits.T2IF = 0;
        IEC0bits.T2IE = 1;
    }

    if ((PORTBbits.RB6 == 0) && (!SPANPressed)) {
        // SPAN button pressed - start Timer3
        PR3 = 13670; // 50ms timer
        TMR3 = 0;
        T3CON = 0x8030;
        IFS0bits.T3IF = 0;
        IEC0bits.T3IE = 1;
    }
}


// Interrupt routine button press detection, either buttons
void __attribute__((__interrupt__, no_auto_psv)) _T2Interrupt(void)
{
    // Clear the flag and turn the time off
	IFS0bits.T2IF = 0;
    T2CON = 0;

    // Check button RB5 (HOLD_BUTTON), take action if low
    HOLDPressed = (PORTBbits.RB5 == 0);
    
    // We can do the required action in this interrupt, as it is quick
    if (HOLDPressed) {
        HOLDActive = ~HOLDActive;
        HOLDPressed = 0;
    }
    
}


// Interrupt routine button press detection, either buttons
void __attribute__((__interrupt__, no_auto_psv)) _T3Interrupt(void)
{
    // Clear the flag and turn the time off
	IFS0bits.T3IF = 0;
    T3CON = 0;

    // Check button RB6 (SPAN_BUTTON), take action if low
    SPANPressed = (PORTBbits.RB6 == 0);   
    
    // We can do the required action in this interrupt, as it is quick
    if (SPANPressed) {
        sampleRate++;
        if (sampleRate == NUM_SAMPLERATES)
            sampleRate = 0;
            // Configure the sample rate setting
            PR1 = samplerates[sampleRate].PR1Val;        
        SPANPressed = 0;
    }
}


// Interrupt routine for the Timer1 ADC sampling interval interrupt
void __attribute__((__interrupt__, no_auto_psv)) _T1Interrupt(void)
{
	IFS0bits.T1IF = 0;
	AD1CON1bits.SAMP = 1;
}


/******************************************************************************
 * Function:        int main(void)
 *
 * PreCondition:    None
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Sets CPU clock and initializes Ports and timer.
 *****************************************************************************/

int main(void)
{
    uint8_t lp;
    char str[16];
    double peakKHz;
    
    PLLFBDbits.PLLDIV = 68; // M 
    CLKDIVbits.PLLPOST = 0; // N2 = 2
    CLKDIVbits.PLLPRE = 0; // N1 = 2
    
    // No ADC inputs used, so turn them off
    ANSELB = 0;

    // Configure the GPIO pins shared with the op-amps signals as inputs
    TRISB = 0x0161;
    TRISA = 0x0003;
    LATB = 0;
    LATA = 0;

    // Configure the reference voltage CVREF to be mid-rail, relative to the analog supply rail
    // and put that voltage on pin CVREF10
    CVRCON = 0x00EC;
    
    // Configure the op-amp, turning it on
    CM2CON = 0x8400;
    // Auto convert, 12 bit

    // Fractional data
    AD1CON1 = 0x07E0;
    AD1CON2 = 0x0000;
    // Internal ADC osc
    // fast sample time // mid-range sample time
    AD1CON3 = 0x8F00; // 0x8F00;
    // DMA disabled
    AD1CON4 = 0x0000;
    // Use Op-amp2
    AD1CHS0 = 0x1919;
    AD1CHS123 = 0x0000;
    // No scanning
    AD1CSSH = 0x0000;
    AD1CSSL = 0x0000;
    
    // Configure the LCD 
    ili9341Init();
    
    displayArrayInit();
    
    sprintf(str, "PEAK");
    drawString(0,100,str, 255, 0, 2);    
    sprintf(str, "SPAN");
    drawString(0,50,str, 255, 0, 2);    
    
    // Enable interrupt on change for our buttons
    // on RB5 (HOLD_BUTTON) & RB6 (SPAN_BUTTON)
    // Signal goes low when button pressed
    CNENBbits.CNIEB5 = 1;
    CNENBbits.CNIEB6 = 1;
    // Clear the change of state interrupt flag
    IFS1bits.CNIF = 0;
    // Enable the interrupt
    IEC1bits.CNIE = 1;

    // Turn on interrupts
    IFS0bits.AD1IF = 0;
    IEC0bits.AD1IE = 1;
    INTCON2bits.GIE = 1;

    // Start the ADC
    AD1CON1bits.ADON = 1;
    AD1CON1bits.SAMP = 1;
    
    // Configure the sample rate setting
    PR1 = samplerates[sampleRate].PR1Val;
    IFS0bits.T1IF = 0; 
    IEC0bits.T1IE = 1; 
    T1CONbits.TON = 1;
    
    do {   
        do {
            Idle();
        } while (!bufferFull);

        if (!HOLDActive) {        
            /* Perform FFT operation */

            FFTComplexIP (LOG2_BLOCK_LENGTH, &sigCmpx[0], (fractcomplex *) __builtin_psvoffset(&twiddleFactors[0]), (int) __builtin_psvpage(&twiddleFactors[0]));

            /* Store output samples in bit-reversed order of their addresses */
            BitReverseComplex (LOG2_BLOCK_LENGTH, &sigCmpx[0]);

            /* Compute the square magnitude of the complex FFT output array so we have a Real output vector */
            SquareMagnitudeCplx(FFT_BLOCK_LENGTH/2, &sigCmpx[0], output);

            // Zero the first 10 'bins', to remove the false signals generated
            // by the DC offset of our audio signal.
            // This means we are not displaying signals below 50Hz
            for (lp = 0; lp < 10; lp++)
                output[lp] = 0;

            /* Find the frequency Bin ( = index into the SigCmpx[] array) that has the largest energy*/
            /* i.e., the largest spectral component */
            VectorMax(FFT_BLOCK_LENGTH/2, output, &peakFrequencyBin);

            /* Compute the frequency (in Hz) of the largest spectral component */
            peakFrequency = ((uint32_t)peakFrequencyBin * samplerates[sampleRate].rate) / FFT_BLOCK_LENGTH;
            peakKHz = (double)peakFrequency / 1000.0;

            // Display the full audio spectrum as a bargraph
            displayArray((uint16_t *)output);

            if (peakKHz < 1)
                sprintf(str, "%4d ", (int)(peakKHz*1000.0));
            else
                sprintf(str, "%4.1fK", peakKHz);
            drawString(0,80,str, 255, 0, 2);    
        }
        
        sprintf(str, samplerates[sampleRate].text);
        drawString(0,30,str, 255, 0, 2);
        
        if (HOLDActive) 
            drawString(0,5,"HOLD", 255, 0, 2);
        else
            drawString(0,5,"    ", 255, 0, 2);

        // Allow conversion results to start again
        // Make FFT calc & display faster by disabling ADC interrupts 
        // during those times
        bufferFull = 0;
        // Re-enable ADC conversions
        IEC0bits.T1IE = 1; 
        
    } while (1);
}
/*******************************************************************************
 End of File
*/