/****************************************************************************
*
*   Module:     pic24video.c
*   Author:     Mike Hibbett
*   Date:       22/1/09
*   Purpose:    Main source file for the pic24 video library demonstration.
*                
*               This must be compiled for the PIC24HJ128GP202, the processor
*               supported by the video library.
*               The processor clock must run at 80Mhz with an Fcy of 40MHz
*
****************************************************************************/

/**** INCLUDES *************************************************************/
#include "p24HJ128GP202.h"
#include <string.h>

/* Video lirary includes */
#include "pic24video.h"
#include "font5x7.h"


/**** CONSTANTS ************************************************************/
#define LINE_TIME_US	64	    // period of each display line - PAL

// Timer2 setup: 1:1 prescale, run from FCY directly, tmr off
#define T2CON_SETUP     0b0000000000000000 
#define T2CON_BIT_TON   0b1000000000000000 // ON/OFF bit
#define PR2_64US        2560    // reload value to get 64us

// OC3 will be our sync pulse output signal
// OC3 setup: // TMR2 src, continuous pulses
#define OC3CON_SETUP    0b0000000000000101
#define OC3RS_64US      2560    // reload value to get 64us line
#define OC3RS_32US      1280    // reload value to get 32us line
#define OC3R_2US        80      // reload value to get 2us pulse
#define OC3R_4US        188     // reload value to get 4.7us pulse
#define OC3R_30US       1200    // reload value to get 30us pulse
#define OC3_RP_MAP      RPOR2bits.RP5R // Pin on which to place OC3 - RP5
#define RP_OC3          0b10100 // OC3 peripheral value for mapping

// OC1 Setup. We will use it to generate audio beeps.
// Timer3 setup: 1:1 prescale, run from FCY directly, tmr off
#define T3CON_SETUP     0b0000000000000000 
#define T3CON_BIT_TON   0b1000000000000000 // ON/OFF bit
#define OC1CON_SETUP    0b0000000000001101
#define OC1_RP_MAP      RPOR5bits.RP11R // Pin on which to place OC1 - RP11
#define RP_OC1          0b10010 // OC1 peripheral value for mapping

// SPI1 definitions. SPI1 is used for video data output
#define SDO1_RP_MAP     RPOR2bits.RP4R    // Pin on which to place SDO1
#define RP_SDO1         0b00111           // SDO1 peripheral value

// Define the number of 16 bit words required to hold the video image
#define FB_SIZE_WORDS   ((VID_WIDTH_PIXELS/16) * VID_DEPTH_PIXELS)



/**** VARIABLES ************************************************************/
static volatile unsigned int T2State = 0;
static volatile unsigned int syncCount;
unsigned short frameBuffer[FB_SIZE_WORDS];
static volatile unsigned short *fbptr;
static volatile unsigned char frameFlag;
static volatile unsigned char soundTimer=0;


/**** FORWARD DECLARATIONS *************************************************/



/**** CODE *****************************************************************/


/****************************************************************************
*
*   Function:   PIC24VideoDelayms
*   inputs:     unsigned int : number of millseconds to delay
*   returns:    void
*   Purpose:    To provide a delay routine for the foreground task,
*               which can be interrupted but still give an accurate delay.
*
****************************************************************************/
void PIC24VideoDelayms( unsigned int delayTime )
{
    // This routine uses timer one in timer mode, 1:8 on FCY (40MHz)
    // which gives a 0.2us count
    unsigned int val = delayTime * 156.25; // convert required # ms to timer period
    TMR1 = 0;
    T1CON = 0b1000000000110000;
    while (TMR1 < val ) {
        ;   
    }
}



/****************************************************************************
*
*   Function:   PIC24VideoDelays
*   inputs:     unsigned int : number of seconds to delay
*   returns:    void
*   Purpose:    to provide a delay routine for the foreground task,
*               which can be interrupted but still give an accurate delay
*
****************************************************************************/
void PIC24VideoDelays( unsigned int delayTime )
{
    delayTime *= 10;
    while (delayTime) {
        delayTime--;
        PIC24VideoDelayms(100);
    }
}



/****************************************************************************
*
*   Function:   PIC24VideoPlayNote
*   inputs:     unsigned int : frequency of note to play (not in any units)
*   returns:    void
*   Purpose:    To start playing an audio signal out of the TV speaker
*
****************************************************************************/
void PIC24VideoPlayNote( unsigned int freq, unsigned char duration )
{
    TMR3   = 0;
    T3CON  = T3CON_SETUP;
    PR3    = freq;
    T3CON |= T3CON_BIT_TON;

    OC1RS  = freq;
    OC1R   = freq/2;
    OC1CON = OC1CON_SETUP;
    
    // Place the OC1 signal onto an output pin
    OC1_RP_MAP = RP_OC1;
    
    soundTimer = duration;
}



/****************************************************************************
*
*   Function:   PIC24VideoStopNote
*   inputs:     void
*   returns:    void
*   Purpose:    Stops audio output
*
****************************************************************************/
void PIC24VideoStopNote( void )
{
    OC1CON = 0;
}



/****************************************************************************
*
*   Function:   PIC24VideoPutChar
*   inputs:     x,y - pixel location for top left corner of character.
*               ch - ASCII character to display
*   returns:    none
*   Purpose:    Displays an ascii character
*
****************************************************************************/
void PIC24VideoPutChar(unsigned char x, unsigned char y, unsigned char ch)
{
    unsigned char xp, yp;
    
    // we only store the characters starting from the space character, so 
    // translate the requested ascii code into a value in our table
    if ( (ch < ' ') || (ch > 'z'))
        ch = 0;
    else
        ch -= ' ';
    
    for ( yp = 0; yp < 7; yp++ ) {
        for ( xp = 0; xp < 5; xp++ ) {
            if ( font5x7[ch*5 + xp ] & (0x01 << yp) )
                PIC24VIDEO_SETPIX(x+xp,y+yp);
            else
               PIC24VIDEO_CLRPIX(x+xp,y+yp);
        }
    }
}



/****************************************************************************
*
*   Function:   PIC24VideoPutStr
*   inputs:     x,y - pixel location for top left corner of string.
*               str - ASCII character string to display 
*   returns:    none
*   Purpose:    Displays an ascii character
*
****************************************************************************/
void PIC24VideoPutStr(unsigned char x, unsigned char y, char *str) 
{
    do {
        PIC24VideoPutChar(x, y, *str);
        str++;
        x += 6;
    } while ( *str );
    
}


/****************************************************************************
*
*   Function:   PIC24VideoWaitFrame
*   inputs:     none
*   returns:    none
*   Purpose:    Waits for the frame end signal. This is set when all the
*               frame buffer has been displayed, and a period of low CPU
*               activity starts - an ideal time to be updating the display.
*               So if you want your code to run for a short time but as 
*               quick as possible, call this routine. When it exits, your code
*               will run uninterrupted. 
*
****************************************************************************/
void PIC24VideoWaitFrame(void)
{
    do {
        ; // nothing
    } while ( !frameFlag );
    
    frameFlag = 0;
}



/****************************************************************************
*
*   Function:   PIC24VideoCls
*   inputs:     none
*   returns:    none
*   Purpose:    Clears the framebuffer, setting the screen to black.
*
****************************************************************************/
void PIC24VideoCls(void)
{
    memset(frameBuffer, 0, sizeof(frameBuffer));
}



/****************************************************************************
*
*   Function:   PIC24VideoDrawRect
*   inputs:     top left & bottom right coordinates of box to fill
*   returns:    void
*   Purpose:    draws a filled rectangle. For speed, no clipping is done.
*               NOTE: This routine is not optimal and could be made quicker.
*
****************************************************************************/
void PIC24VideoDrawRect( unsigned char xtl, unsigned char ytl, unsigned char xbr, unsigned char ybr )
{
    unsigned char x,y;
    
    for ( x=xtl; x <= xbr; x++ ) {
        for ( y = ytl; y <= ybr; y++ )
            PIC24VIDEO_SETPIX(x,y);
    }
}



/****************************************************************************
*
*   Function:   PIC24VideoClearRect
*   inputs:     top left & bottom right coordinates of box to clear
*   returns:    void
*   Purpose:    clears a filled rectangle. for speed, no clipping is done.
*               NOTE: This routine is not optimal and could be made quicker.
*
****************************************************************************/
void PIC24VideoClearRect( unsigned char xtl, unsigned char ytl, unsigned char xbr, unsigned char ybr )
{
    unsigned char x,y;
    
    for ( x=xtl; x <= xbr; x++ ) {
        for ( y = ytl; y <= ybr; y++ )
            PIC24VIDEO_CLRPIX(x,y);
    }
}



/****************************************************************************
*
*   Function:   PIC24VideoInit
*   inputs:     none
*   returns:    none
*   Purpose:    Setup the I/O used by the video system, clear the framebuffer 
*               and start the video signals.
*
****************************************************************************/
void PIC24VideoInit(void)
{
    // Continuous pulse train
    // Set TMR2 to allow us a 1us resolution, but better than 65us range,
    // Timer reset by PR2
    // Set PR2 to 64us
    // Set OC3RS to 64us
    // Set OC3R to trigger at 2us ( say )
    // Set OC3 mde to continuous pulse
    // PRobably need to map OC3 pin to an I/O pin.
    // Enable TMR2
    // LAter, add interrupt on timer2, so we can play with OC3 pin
    
    // Configure the source tmr for OC3. 
    TMR2   = 0;
    T2CON  = T2CON_SETUP;
    PR2    = PR2_64US;
    
    // Configure the sync pulse signal OC3
    OC3RS  = OC3RS_64US;
    OC3R   = OC3R_4US;
    OC3CON = OC3CON_SETUP;
    
    // Place the OC3 signal onto an output pin
    OC3_RP_MAP = RP_OC3;
    
    // Place the SDO1 signal onto an output pin
    SDO1_RP_MAP = RP_SDO1;

    // Setup the SPI1 peripheral
    SPI1CON2 = 0b0000000000000000;
                  
    ODCB = 0;
    
    // Setup the DMA channel. The manual writing of data to SPIBUF
    // will cause a tx complete interrupt, which the DMA module 
    // will pick up.
    
    // Start the timer, and therefore the pulse train
    T2CON |= T2CON_BIT_TON;
    
    frameFlag = 0;
    
    // Enable timer interrupts, to get video going.
    IFS0bits.T2IF = 0;  // Clear the flag, in case of spurious sets
    IEC0bits.T2IE = 1;  // Enable TMR2 interrupts
        
    PIC24VideoCls();
}



/****************************************************************************
*
*   Function:   _T2Interrupt
*   inputs:     none, its an interrupt
*   returns:    none, its an interrupt
*   Purpose:    The interrupt for timer2 - triggered when OC3 goes low
*               This is a very simple state machine that outputs
*               sync pulses of various lengths to give us a 
*               non interlaced frame of graphics. The sequence is 6-5-5 of
*               short-long-short ( all themselves half length syncs ), 
*               followed by 296 standard length lines.
*
****************************************************************************/

void __attribute__((interrupt, auto_psv)) _T2Interrupt(void)
{
    int x;
    
    switch (T2State) {
        case 0:    
            // setup for 6 short, half length syncs.
            syncCount = 6;
            PR2       = OC3RS_32US;
            OC3RS     = OC3RS_32US;
            OC3R      = OC3R_2US;
            T2State++;
            // drop through to next state
            
        case 1:
            // Output the short sync. Nothing to do here.
            if ( --syncCount == 0 ) T2State++;
            break;
        
        case 2:
            // setup for 5 long, half length syncs.
            syncCount = 5;
            OC3R      = OC3R_30US;
            T2State++;
            // drop through to next state
            
        case 3:
            // Output the long sync. Nothing to do here.
            if ( --syncCount == 0 ) T2State++;
            break;
        
        case 4:
            // setup for 5 short, half length syncs.
            syncCount = 5;
            OC3R      = OC3R_2US;
            T2State++;
            // drop through to next state
            
        case 5:
            // Output the short sync. Nothing to do here.
            if ( --syncCount == 0 ) T2State++;
            break;
        
        case 6:
            // Setup for normal lines

            syncCount = 296;
            PR2       = OC3RS_64US;
            OC3RS     = OC3RS_64US;
            OC3R      = OC3R_4US;
            
            fbptr = &frameBuffer[0];

            T2State++;
            // drop through to next state
            
        case 7:     
            // Output the short sync. 
            
            if ( --syncCount == 0 ) 
                T2State = 0;
            else {
                
                if ( (syncCount > 20) && (syncCount < 261) ) {
                    
                    // Enable the SPI
                    SPI1CON1 = 0b0000010100111010;
                    SPI1STAT = 0b1000000000000000;

                    // get past the back porch
                    for ( x = 0; x < 65; x++ ) 
                        Nop();
                    
                    
                    SPI1BUF = 0;

                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();

                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    SPI1BUF = *fbptr++; 
                    for ( x = 0; x < 18; x++ ) 
                        Nop();
                    
                    SPI1BUF = 0;
                    
                } else if ( syncCount == 261 ) {
                    // indicate that the video update for this frame has finished
                    frameFlag = 1;    
                    
                    // update the simple sound duration timer
                    if (soundTimer) {
                        soundTimer--;
                        if (soundTimer == 0) {
                            PIC24VideoStopNote();   
                        }   
                    }
                }
            }               
            break;         
    }
    
    IFS0bits.T2IF = 0;  // Clear flag - otherwise we come straight back here
}
