/*
    Module: serialDebug.c

    Author: Mike Hibbett

    Date: 21/11/09

    Function: Implementation of the serial interface debugging module for PIC processors.
              Provides a quick means of adding debugging features into your application.
              It gives a nice interactive menu interface.
              Options are provided to have the debug output generated in the receive 
              interrupt ( not great, but necessary if your application does not have a 
              main loop ) or as a routine called within your main loop, which adds very little
              overhead to the interrupt processing times.
              

              Refer to serialDebug.h for more details.

*/

#include <usart.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "serialDebug.h"
#include "sdTable.h"


/*****************************************************************************
    Global Data
*****************************************************************************/

static char rxBuff[DEBUG_CMD_STR_LENGTH+1];	/* Buffer to hold the serial message */
static unsigned char rxIndex = 0; 			/* Indicates position in the rxBuffer */
static unsigned char cmdReceived = 0;		/* Indicates when a full command has been received */


/*****************************************************************************
    Function: serialDebug

    Purpose:  Main application entry to the debugger.

    Returns: 
*****************************************************************************/
void serialDebug(void)
{
	cmdList_t *sdp = cmdTable;
	unsigned char cmdLen;
	char *ptr;
	static char startupFlag = 0;
	
	
	if ( startupFlag == 0 ) {
    	
        startupFlag = 1;
        putrsUSART((const far rom char *)"\r\n> Debug terminal available.\r\n> ");	
    }   
    
	// todo - exit if no work to do, determined by the cmdReceived flag
	if ( cmdReceived == 1 ) {
		
	    ptr = rxBuff;

		/* Strip off leading whitespace */
		while ( !isalnum(*ptr) ) {
    		
			ptr++;
		}

		/* Find end of the string, and terminate it */			
		while ( isalnum(*ptr) ) {
    		
			ptr++;
		}
		
		*ptr = NULL;
		ptr++;  /* We will use ptr later to pass paramaters to the handler */

		/* See if we can find the command requested */

		cmdLen = strlen(rxBuff);

        if ( (rxBuff[0] == '\r') || (rxBuff[0] == '\n') ) {
        
            /* Looks like just the enter pressed. Show prompt */
            putrsUSART((const far rom char *)"\r\n> ");
        } else {
        
            /* Has a valid command name been entered? */
    		while ( 
    		        (sdp->name != NULL) && 
    		        (strncmpram2pgm((far rom char *)sdp->name, rxBuff, cmdLen) ) 
    		       ) {
        		       
    			sdp++;
    		}
    
    		/* If we have, call the handler, otherwise display error msg */
    		if ( sdp->name != NULL ) {
    			putrsUSART((const far rom char *)"\r\n");
    			sdp->funcptr(ptr);
    			putrsUSART((const far rom char *)"\r\n> ");
    		} else {
    			putrsUSART((const far rom char *)"\r\nInvalid command. Try 'help'\r\n> ");
    		}		
        }
        
		/* Allow the debugger to receive further commands */
		cmdReceived = 0;	
	}
}



/*****************************************************************************
    Function: sdRxChar

    Purpose:  called by the application to pass a character to the debugger

    Returns: 
*****************************************************************************/
void sdRxChar( char rxCh )
{
	/* Ignore character if we are currently processing a command */
	if ( cmdReceived == 1 ) 
		return;
		
	/* Add character to rxcmd */
	rxBuff[rxIndex++] = rxCh;
	/* echo it back */
	while (BusyUSART());
	putcUSART(rxCh);
	
	/* A CR terminates the command entry */
	if ( (rxCh == 0x0D) || (rxIndex == DEBUG_CMD_STR_LENGTH) ) {
		cmdReceived = 1;
		rxBuff[rxIndex] = 0x00;
		rxIndex = 0;
	}
}

