/*********************************************************************
 *
 * This program uses subroutines from Microchip USB C18 Firmware Version 1.0
 *
 *********************************************************************
 * FileName:        Usb.c
 * Dependencies:    See INCLUDES section below
 * Processor:       PIC18
 * Compiler:        C18 2.30.01+
 * Author:          Robert B. Lang 
 * This software works with the hardware described in the article
 * Digital Geiger Counter by Robert Lang and subroutines from MICROCHIP 
 * Communication Device Class (CDC) Firmware Revision 1 at 
 * http://www.microchip.com/stellent/idcplg?IdcService=SS_GET_PAGE&nodeId=2124&param=en022625
/** I N C L U D E S **********************************************************/
#include <p18cxxx.h>
#include "typedefs.h"                        // Required
#include "usb.h"                         // Required
#include "xxlcd.h"
#include <stdio.h>
/** U S B ***********************************************************/
#define usb_bus_sense       1
#define self_power          1
extern volatile far unsigned char NTRANSMIT;
extern volatile far unsigned char NRECEIVE;
extern volatile far unsigned int MON;
extern volatile far unsigned int DAY;
extern volatile far unsigned int YR;
extern volatile far unsigned int HR;
extern volatile far unsigned int MIN;
extern volatile far unsigned int SEC;
extern volatile far unsigned char DATETIMESTRING[20];
extern volatile far unsigned char COUNTSTRING[20];
extern unsigned char DATAPOINT; // DATA POINT NUMBER
/** V A R I A B L E S ********************************************************/
#pragma udata
static unsigned char usb_input_buffer[64];  // USER USB WORKING BUFFERS
static unsigned char usb_output_buffer[64];
#pragma udata

/** P R I V A T E  P R O T O T Y P E S ***************************************/
unsigned char USB_Input_handler(void);
void USB_Output_handler(void);
void Home(unsigned char LINE);
void USBResponse(unsigned char MYCHAR);
void EraseAllData(void);
void USBStatus(void);
void USBTasks(void);
void UpdateDateTime(void);
void MemcpyFlash2Ram(rom far unsigned int *src,  ram far unsigned int *dst, int nBytes);
#pragma code

//*****************************************************************************	
// void DataTransfer(void) Handles USB data transfers
//*****************************************************************************
void DataTransfer(void)
{	unsigned char RET;
   USBTasks(); //Queue data received from PC via USB for input
   USBStatus(); // Show USB status on LCD
  if((usb_device_state < CONFIGURED_STATE)||(UCONbits.SUSPND==1)) return;
    RET=USB_Input_handler(); // Process data received from USB and prepare response 
   	if (RET) // IF THERE WAS INPUT THEN THERE MAY BE OUTPUT
   	{
   	 USB_Output_handler();   // queue response for output to USB
   	 USBTasks(); // Send Queued data to PC via USB
   	}  
}  //end DataTransfer	

// THE FOLLOWING ROUTINES RESPOND TO VARIOUS COMMANDS RECEIVED VIA USB	
void RespondErase(void)
{ 
  XputrsXLCD("ERASE RECEIVED     ");
  EraseAllData(); // ERASE ALL FLASH DATA
  NTRANSMIT=1;// SET TRANSMIT
  NRECEIVE=0; //RESET RECEIVE
  usb_output_buffer[0]='R';  // SEND BACK READY	
}	
void RespondQuery(void)
{ unsigned char NPTS;
	XputrsXLCD("QUERY RECEIVED     ");
  NTRANSMIT=2;
  NPTS=DATAPOINT-1;
  usb_output_buffer[0]=NPTS;
  usb_output_buffer[1]='R';  // SEND BACK READY	
	}
void RespondSetclock(void)
{ 
	XputrsXLCD("SET TIME RECEIVED  ");
  NTRANSMIT=1;
  Home(3);
  if (NRECEIVE>6)
  { // PROPER AMOUNT OF DATA RECEIVED
  // GET REST OF TIME INFORMATION
  MON=(unsigned int) usb_input_buffer[1];
  DAY=(unsigned int)usb_input_buffer[2];
  YR=(unsigned int)usb_input_buffer[3];
  HR=(unsigned int)usb_input_buffer[4];
  MIN=(unsigned int)usb_input_buffer[5];
  SEC=(unsigned int)usb_input_buffer[6];
  Home(4);
 UpdateDateTime(); 
  XputsXLCD(DATETIMESTRING); // display the date time
 usb_output_buffer[0]='R';  // SEND BACK READY
  }
// WAIT FOR MORE DATA
}	
void RespondUpload(void)
{ unsigned int REQUESTED_DATAPOINT;
  unsigned int ADDRESS;
  XputrsXLCD("UPLOAD REQ RECEIVED ");
  Home(3);
  if (NRECEIVE>1)
  {
  REQUESTED_DATAPOINT=(unsigned int) usb_input_buffer[1];
  usb_output_buffer[0]=usb_input_buffer[1]; // LOAD DATAPOINT # INTO SEND BUFFER
  ADDRESS=DATASTART + (REQUESTED_DATAPOINT-1)*32;
  MemcpyFlash2Ram(ADDRESS, usb_output_buffer+1, 32); // PUT FLASH DATA INTO SEND BUFFER LOCATIONS 1 TO 32
  usb_output_buffer[33]='R';  // END SEND BLOCK WITH R
  NTRANSMIT=34;  // SENDING 34 BYTES
  Home(4);
  sprintf(COUNTSTRING,"DATA#%3u UPLOADED   ",REQUESTED_DATAPOINT);
  XputsXLCD(COUNTSTRING); // REUSE COUNTSTRING TO DISPLAY MESSAGE
  }
  // OR WAIT FOR MORE DATA
}

//************************************************************************
// unsigned char USB_Input_handler(void)
// This routine takes data from the USB and loads in buffer.
//************************************************************************
unsigned char USB_Input_handler(void)
// returns 1 data is received successfully
// returns 0 if no data 
{	unsigned char index;
 if (getsUSBUSART(usb_input_buffer,CDC_BULK_OUT_EP_SIZE)) // 0 if no data ready
        {
          NRECEIVE=mCDCGetRxLength();
          USBResponse(usb_input_buffer[0]);
	return 1; // data processed
	}
	return 0; // no data  
}// USB_Input_handler

//**************************************************
// USB_Output_handler(void) This routine takes data 
// and sends to USB 
//**************************************************
void USB_Output_handler(void)
{   unsigned char index;
    unsigned char mychar;
// Is there data ready to send and is USB input stream available?
// NTRANSMIT is the number of characters to be  transmitted to usb
	if (NTRANSMIT>0)
	{
	if (mUSBUSARTIsTxTrfReady()) // IF NOT READY WAIT TILL NEXT TIME
	  {	
         index=NTRANSMIT;
         mUSBUSARTTxRam((byte *) usb_output_buffer,index); //queue usb buffer to send
         NTRANSMIT=0;
      }
    } 
}  // END OF USB_Output_handler

//***********************************************************
// void USBResponse(unsigned char MYCHAR)  CHOOSE PROPER RESPONSE FOR COMMAND
//***********************************************************
 void USBResponse(unsigned char MYCHAR)
 {unsigned int MYCHARINT;
	char ERRORSTRING[20]; 
	MYCHARINT=MYCHAR;
	Home(4); 
//ERASE ANYTHING REMAINING ON LINE 4 FROM UPLOAD OR SETCLOCK COMMAND
	XputrsXLCD("                   ");
	Home(3); // OUTPUT ANY MESSAGES ON LINE 3 
 switch (MYCHAR)
  {	 
  case 'Q':
  RespondQuery();
  break;	 
  case 'U':
  RespondUpload();
  break;	 	 
  case 'E':
  RespondErase();
  break;	 
  case 'S':
  RespondSetclock();
  break;	 	 
  default:
  // FORMAT ERRROR STRING FOR LCD 
  sprintf(ERRORSTRING,"INVALID COMMAND=[%1c]",MYCHARINT);
  XputsXLCD(ERRORSTRING);
  NTRANSMIT=1;// SET TRANSMIT
  NRECEIVE=0;
 usb_output_buffer[0]='N';  // SEND BACK NAK
  }	 
}

//***************************************************************************
//   USBStatus(void) displays the USB device state on Line 2 of LCD.
//***************************************************************************
void USBStatus(void)
{
 static word led_count=0;
    if(led_count == 0)led_count = 10000U;
    led_count--;

    if(UCONbits.SUSPND == 1)
     { if(led_count==0)
       { RED_LED=!RED_LED;  // SIMULTANEOUS BLINK
         GREEN_LED = RED_LED; 
         //XputrsXLCD("SUSPENDED "); MESSAGE REMOVED SINCE IT APPEARED OF CAUSE ENUMERATION TO FAIL
       }
    }       
    else
    {
            if(usb_device_state == POWERED_STATE)
             { 	RED_LED=1; // TURN ON HV LED (RED)
                GREEN_LED=0; // TURN OFF AUX LED (GREEN) 
                //XputrsXLCD("RESET   "); MESSAGE REMOVED SINCE IT APPEARED OF CAUSE ENUMERATION TO FAIL
             }                  
    else if(usb_device_state == ADDRESS_STATE)
             {  RED_LED=0; // TURN OFF HV LED (RED)
                GREEN_LED=1; // TURN ON AUX LED (GREEN) 
                //XputrsXLCD("ADDRESSED ");  MESSAGE REMOVED SINCE IT APPEARED OF CAUSE ENUMERATION TO FAIL
             }          
    else if(usb_device_state == CONFIGURED_STATE)
             {   if(led_count==0)
                 {
                 RED_LED=!RED_LED;
                  GREEN_LED = !RED_LED;    // Alternate blink 
                  // XputrsXLCD("CONFIGURED"); MESSAGE REMOVED SINCE IT APPEARED OF CAUSE ENUMERATION TO FAIL
                 }
             }
    }//end if(UCONbits.SUSPND...)
}//end USBStatus

void USBTasks(void)
{
// SERVICE USB HARDWARE
    USBCheckBusStatus();   // Must use polling method in usbdrv.c
    if(UCFGbits.UTEYE!=1)
    USBDriverService(); // Interrupt or polling method in usbdrv.c
    CDCTxService();  // this handles all USB CDC traffic  
}

