/*********************************************************************
 * GMCOUNTER1 - GEIGER MUELLER COUNTER
 *********************************************************************
 * FileName:        gm_counter.c
 * Dependencies:    See INCLUDES section below
 * Processor:       18F2550 (can be used with 18F2455 with changes int XXLCD.H)
 * Compiler:        C18 2.30.01+
 * Author:          Robert B. Lang 
 * This software works with the hardware described in the article
 * PICking Up Radiation
 * by Robert Lang and Steve Thompson.  
 * Caution: This circuit/program can generate >600 volts DC. 
 * 20 Mhz CRYSTAL driving 18F2455 PIC at 48 Mhz. 
 *
 * THIS SOFTWARE IS PROVIDED IN AN AS IS CONDITION. NO WARRANTIES,
 * WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT NOT LIMITED
 * TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. THE AUTHOR SHALL NOT,
 * IN ANY CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL OR
 * CONSEQUENTIAL DAMAGES, FOR ANY REASON WHATSOEVER.
 *
 *  VERSION    MODIFICATION
 *   1.0.0     INITIAL RELEASE
 *   1.0.1     SWITCH LOGIC CHANGED FROM GOUND=ACTIVE TO 5V=ACTIVE.
 *             DEBOUNCEINPROGRESS VARIABLE TO INDICATE THAT SWITCH 
 *             INTERRRUPT HAS OCCURRED AND SWITCH DEBOUNCE IS IN PROGRESS. 
 *   1.0.2     S1DBRUNNING AND S2DBRUNNING REPLACE DEBOUNCEINPROGRESS
 *             WRITE 0XFFFF IN EACH MEMORY LOCATION TO ERASE MEMORY
 *   1.0.3     ADDED HV CONTROL SO HV IS ONLY ON WHEN NEEDED. HVflag AND HV(ONOFF)   
 *   1.0.4     ADDED COUNT TIME TO DATA STORE
 *   1.0.5     SWITCHES MOVED TO B2 AND B3, RTC INTERFACE ADDED ON B0 AND B1.
 *             MENU CODING ADDED TO SET RTC AND READSWITCHES MODIFIED.
 *   1.0.6     CODING ADDED TO DECODE AND ENCODE TIME FROM REAL TIME CLOCK. I2C
 *             ROUTINES ADDED TO TALK TO RTC.  CONSTANTS ADJUSTED TO USE 32MHZ CLOCK.
 *   1.0.7     PWM logic changed to generate constant 100Hz square wave. 
 *             MULTI feature added for multiple measurements.
 *   1.0.8     ADDED DELAYS TO ALLOW HV TO STABILIZE.  ADDED USB FRAMEWORK.  RA1 IS INPUT.
 *   1.0.9     REPROGRAMMED TO USE INDUCTOR RATHER THAN TRANSFORMER FOR HV GENRATION
 *   2.0.1     ADDED DUMMY RTC ROUTINES AND ELIMINATED EXTERNAL RTC 
 *   3.0.1     SWITCHED LEDS FROM PIN 6 AND 12 TO PIN 17 AND 18, ADDED LCD POWER CONTROL,
 *             ADDED WATCH CRYSTAL EXTERNAL OSCILLATOR FOR REAL TIME CLOCK,
 *             TURN OFF HEARTBEAT WHILE SLEEPING.
 * CHANGE CUR_VER BELOW...*/
char VERSION[20]="   VERSION 3.0.1   "; 
// COMMENT FOLLOWING LINE FOR 18F2550 (SEE XXLCD.H FOR DETAILS)
#define IS_18F2455 1

/** INCLUDE HEADER FILES******/
#include <timers.h>
#include <adc.h>
#include <delays.h>
#include <stdio.h>
#include <portb.h>
#include <pwm.h>
#include "typedefs.h"                        
#include "usb.h"    
#include "xxlcd.h"

// CONGIGURATION BITS FOR MICROPROCESSOR
// TIMER1 OSCILLATOR CONFIGURED FOR LOW POWER
#pragma config PLLDIV=5, CPUDIV=OSC1_PLL2, USBDIV=2
#pragma config IESO=OFF, FCMEM=OFF, FOSC=HSPLL_HS 
#pragma config PWRT=OFF, BOR=ON, BORV=28, VREGEN=ON, MCLRE=ON
#pragma config PBADEN=OFF, STVREN=ON, LVP=OFF, ICPRT=OFF 
#pragma config XINST=OFF, DEBUG=OFF, WDT=OFF, LPT1OSC=ON

// DEFINE FUNCTION PROTOTYPES
void ADCGet(void);
void timer_handler(void);
void ReadSwitches(void);
void Rsi(void);
void UpdateCount(void);
void UpdateVoltages(void);
void UpdateOnTime(void);
void SaveData(void);
void SleepLCD(void);
void StopClock(void);
void StartClock(void);
void StartCount(void);
void UpdateClock(void);
void EncodeTime(void);
void UpdateDateTime(void);
void DataTransfer(void);
void DelayPORXLCD(void);
void UpdateDateTime(void);
void MemcpyFlash2Ram(rom far unsigned int *src,  ram far unsigned int *dst, int nBytes);	
// following routines manipulate flash program memory
void EraseFlash(unsigned int dst);
void MemcpyRam2Flash(rom far unsigned int *dst, ram far unsigned int *src);	
void MemcpyFlash2Ram(rom far unsigned int *src,  ram far unsigned int *dst, int nBytes);
void Tic(void);
void HV(unsigned char);	

//  DATA TYPE DEFINITIONS 
	typedef union {
	unsigned int word ;
	struct {
		unsigned char low;
		unsigned char high;
	};
} UWORDbytes;

/** VARIABLES ********************************************************/
#pragma udata
unsigned char HEARTBEAT=0; // COUNTER FOR LED HEARTBEAT
unsigned char NRECEIVE=0; // NUMBER OF CAHRACTERS RECEIVED FROM USB
unsigned char NTRANSMIT=0; // NUMBER OF CHARACTERS TO BE TRANSMITTED VIA USB
unsigned char MENUSTATE; // MACHINE STATE OF MENU
unsigned char SCROLL=0;  // INITIALIZE SCROLL SWITCH OFF
unsigned char EXECUTE=0; // INITIALIZE EXECUTE SWITCH OFF
unsigned char SLEEPING=0; // MICROPROCESSOR SLEEPING FLAG
unsigned char COUNTING=0; //FLAG THAT INDICATES GM TUBE PULSES ARE TO BE COUNTED
unsigned char HVflag=0; // FLAG THAT INDICATES HV IS BEING GENERATED FOR GM TUBE
unsigned char RB2PREV;  //PREVIOUS VALUE OF SCROLL SWITCH 
unsigned char RB3PREV;  //PREVIOUS VALUE OF EXECUTE SWITCH
unsigned int TIME;  // NUMBER OF SECONDS TO COUNT (max 65k)
unsigned int TIME_TARGET; // NUMBER OF SECOND INCREMENTS TO COUNT(DEFAULT 15 SECONDS) 
unsigned int TIME_ELAPSED; // IN SECOND INCREMENTS
unsigned char DATAPOINT; // DATA POINT NUMBER
unsigned int COUNT; // CURRENT GM COUNT
unsigned char MULTI;// FLAG THAT INDICATES CONTINUOUS MULTIPLE MEASUREMENTS
unsigned int HR;  //CURRENT TIME (HOURS)
unsigned int MIN; // CURRENT TIME (MINUTES)
unsigned int SEC; // CURRENT TIME (SECONDS)
unsigned int MON; // CURRENT TIME (MONTH)
unsigned int DAY; // CURRENT TIME (DAY OF MONTH)
unsigned int YR;  // CURRENT TIME (YEAR)
unsigned long PERIOD_TIME_UNIT=1332; // time in nsec corresponding to one unit in period (PR2 register)
// 20.8 nanoseconds* Timer2 prescaler (16)* 4 =~1332
unsigned int ON_TIME;
char DATETIMESTRING[20]; // CURRENT DATE AND TIME [MON 01/31/06 13:59  ]
char TIMESTRING[20]; // TIME CONVERTED TO STRING FOR DISPLAY [600 SECONDS]
char VOLTAGESTRING[20]; // VOLTAGE CONVERTED TO STRING FOR DISPLAY [GM TUBE BIAS=500.1 V]
char COUNTSTRING[20]; // COUNT CONVERTED TO STRING FOR DISPLAY [COUNT #002=000312]
char ONTIMESTRING[20]; // ON TIME CONVERTED TO STRING FOR DISPLAY [ON TIME=200 MSEC]
static volatile unsigned long VOUT=0;
unsigned long MAXVOLTS=600;  // maximum of 600 volts

#pragma code int_vector=0x08   // SET INTERRUPT VECTOR LOCATION
void int_vector (void)
{
  _asm goto timer_handler _endasm
}
#pragma code
#pragma interrupt timer_handler
//**************************************************************************
// Timer interrupts are handled by this routine 
//**************************************************************************
void timer_handler(void)
{  
// IS THIS AN RTC INTERRUPT
  if (PIR1bits.TMR1IF==1)
{  PIR1bits.TMR1IF=0; // CLEAR INTERRUPT FLAG
  WriteTimer1(0x8000); // preload TIMER1 for 1 second
// FLASH LED ONE OUT OF TEN SECONDS TO INDICATE RTC RUNNING 
  if ((!SLEEPING)&& (PORTAbits.RA1==0)) //IF NOT SLEEPING AND NOT USB CONNECTED THEN DISPLAY HEARTBEAT
{ HEARTBEAT=HEARTBEAT+1;
  if (HEARTBEAT==10)
     {
     GREEN_LED=1;
     HEARTBEAT=0;
     }
  else
     GREEN_LED=0;
}

// UPDATE TIME AND COUNT WHEN COUNTING 
    if (COUNTING)
{
    TIME_ELAPSED=TIME_ELAPSED+1; // COUNT SECONDS
// IS COUNT TIME UP?
   if (TIME_ELAPSED>=TIME_TARGET)
   { MENUSTATE=60;      // DISPLAY FINAL COUNT MENU 
     HV(0); //STOP COUNT AND TURN OFF HV  
     COUNTING=0; //CLEAR COUNTING FLAG, IE STOP COUNTING
   }
} 
  
// VOLTAGE ONLY MEASURED WHEN HV ON
 if (HVflag) 
    ADCGet(); // Measure VOUT 

  Tic(); // INCREMENT TIME
  Rsi();  //CHECK THE SWITCHES

  if (EXECUTE)  // STAY AWAKE IF EXECUTE SWITCH PRESSED
      SLEEPING=0; // AND RETURN TO MENUSTATE=3
// CHECK USB
 
  if (PORTAbits.RA1==1)  // IF WE HAVE POWER FROM USB 
    SLEEPING=0; // AND RETURN TO MENUSTATE=3

  if (SLEEPING) // GO BACK TO SLEEP
      {
      RED_LED=0; // MAKE SURE RED LED OFF BEFORE GOING TO SLEEP
      GREEN_LED=0; // MAKE SURE HEARTBEAT IS OFF BEFORE GOING TO SLEEP
      Sleep(); // GO BACK TO SLEEP
      } 
 }
 }  

#pragma code
void ADCGet(void)
//****************************************************************************
//   Perform Analog to Digital Conversion for VDES and VOUT
//****************************************************************************
{  SetChanADC(ADC_CH0);  // Select channel 0 (VOUT)
  ConvertADC();      // Start A/D conversion
  while(BusyADC());  // Wait for conversion to complete
  VOUT=ReadADC();    // get converted value
// 0-5 volts=0 to 1023 counts
 // SetChanADC(ADC_CH1);  // Select channel 1 (VDES)
  //ConvertADC();      // Start A/D conversion
 //while(BusyADC());  // Wait for conversion to complete
 // VDES=ReadADC();    // get converted value
}
	
void ADCInit(void)
//****************************************************************************
//   Initialize Analog to Digital Conversion
//****************************************************************************
{ unsigned char config;
  unsigned char config2;
  unsigned char portconfig;
  #define TRIS_AN0  DDRAbits.RA0    /* TRIS for CHANNEL 0 */ 
  // MAKE SURE AN0 IS INPUT FOR A/D TO WORK
   TRIS_AN0=1;
// config=select ADC clock source, justification and conversion time
// config2=select channel and interrupt
// portconfig= contents of ADCON1 register=14 enables AN0 as analog.	
   config=ADC_FOSC_64  & ADC_RIGHT_JUST & ADC_20_TAD;
   config2=ADC_CH0 & ADC_VREFPLUS_VDD & ADC_VREFMINUS_VSS & ADC_INT_OFF;
   portconfig=14;  
   OpenADC(config, config2, portconfig);
 }

// SET UP DELAYS USED BY LCD LIBRARY FUNCTIONS
void  DelayFor18TCY(void)
  {
Delay10TCYx(20); // delay 200 instruction cycles 200*125nsec=2.5usec
  } 
void  DelayXLCD(void)
  {
Delay10KTCYx(4); // delay 40,000 instruction cycles (5 msec)
  }
void  DelayPORXLCD(void)
  {
Delay10KTCYx(12); // delay 120,000 instruction cycles (15 msec)
  }  

void EraseAllData(void)
// THIS ROUTINE ERASES ALL FLASH MEMORY
// USED FOR DATA STORAGE (ABOVE DATASTART)
{ unsigned int MEMORY_LOC;
  unsigned int MAXMEM=DATASTART+(MAXPOINTS*32)-1;
for (MEMORY_LOC=DATASTART;MEMORY_LOC<= MAXMEM;MEMORY_LOC=MEMORY_LOC+64) {	 
EraseFlash(MEMORY_LOC); // ERASE A BLOCK OF 64 BYTES STARTING AT MEMORY_LOC	
}
DATAPOINT=1;  // reset datapoint to 1
}

void EraseFlash(unsigned int dst)
// THIS ROUTINE ERASES A 64 BYTE BLOCK OF PROGRAM MEMORY
// DST IS THE ADDRESS OF THE MEMORY BLOCK TO BE ERASED.   
// LOWER 6 BITS ARE IGNORED.
{
	unsigned short long int dst_1;
dst_1 = ((unsigned short long int)dst);  
		TBLPTR = dst_1;
		EECON1bits.EEPGD=1; //POINT TO FLASH MEMORY
		EECON1bits.CFGS=0;  //ACCESS FLASH MEMORY
		EECON1bits.WREN=1;	//ENABLE WRITE TO MEMORY	
		EECON1bits.FREE=1;  //ENABLE ROW ERASE	
		INTCONbits.GIE=0; // DISABLE INTERRUPTS
		
		_asm 			// ASSEMBLY MUST NOT BE CHANGED
		MOVLW 0x55 
		MOVWF EECON2, 0 
		MOVLW 0xAA 
		MOVWF EECON2, 0 
		BSF EECON1, 1, 0 
		_endasm 
		
		EECON1bits.WREN=0;	//DISABLE WRITE TO MEMORY
		INTCONbits.GIE=1; // ENABLE INTERRUPTS
}

void Home(unsigned char LINE)
{ // Moves the cursor to home position on specified line
  // of a 20x4 line LCD display.
// THESE ARE THE START ADDRESSES FOR THE FOUR LINES OF THE DISPLAY
#define LINE1 0x0 // STRANGE BUT TRUE
#define LINE2 0x40
#define LINE3 0x14
#define LINE4 0x54
switch (LINE)
{ 
case 1:	
XSetDDRamAddr(LINE1); // Line 1
break;
case 2:
XSetDDRamAddr(LINE2); // Line 2
break;
case 3:
XSetDDRamAddr(LINE3); // Line 3
break;
case 4:
XSetDDRamAddr(LINE4); // Line 4
}
}

void HV(unsigned char ONOFF)
// THIS ROUTINE CONTROLS THE HIGH VOLTAGE TO THE GM TUBE
// RED LED INDICATES HIGH VOLTAGE
{
if (ONOFF) 
{
//============================================
// TIMER0 COUNTS GM PULSES   
   OpenTimer0(TIMER_INT_OFF & T0_16BIT &  T0_SOURCE_EXT & T0_EDGE_FALL & T0_PS_1_1 ); 
//============================================
// TIMER2 IS USED IN PWM
   OpenTimer2(TIMER_INT_OFF & T2_PS_1_16 & T2_POST_1_1);
// set HV flag so ON_TIME will be updated in periodic interrupt
HVflag=1;
RED_LED=1; // TURN On LED
// SET PERIOD FOR 250usec (4000Hz) WITH 48MHZ PROCESSOR SPEED
// PWMPERIOD= (PR2+1) * 4* TOSC * TMR2
// 250E-6 =(PR2+1) * 4 * 1/48E-6 * 16
// PR2= 250*48/(16*4) - 1
// PR2=186
OpenPWM1(186);
// ONTIME= (PR2+1) * TOSC * TMR2
// 125E-6 =(CCPR) * 1/48E-6 * 16
// CCPR= 125*48/(16) 
// CCPR=372
ON_TIME=372;
SetDCPWM1(ON_TIME); // 125usec ON TIME
}
else
{HVflag=0;
ClosePWM1();
CloseTimer0();
CloseTimer2();		
HV_MOSFET=0; // MAKE SURE INPUT TO MOSFET IS TURNED OFF, NO DROP ACROSS COIL
RED_LED=0; // TURN OFF LED
	
} 	
}

void InitializeDataStore(void)
// THIS ROUTINE WILL DETERMINE HOW MUCH DATA IS STORED IN EEPROM AND WILL
// UPDATE DATAPOINT TO POINT TO THE NEXT LOCATION
{  unsigned int MEASUREMENT[16];
   unsigned int ADDRESS;
   ADDRESS=MAXPOINTS;  // REMOVE AFETER TEST	
	for (DATAPOINT=1;DATAPOINT<= MAXPOINTS;DATAPOINT++) 
{
// COMPUTE ADDRESS OF DATAPOINT IN FLASH PROGRAM MEMORY		
   ADDRESS=DATAPOINT;	
   ADDRESS=DATASTART + (ADDRESS-1)*32;
// READ FIRST TWO BYTES OF DATAPOINT (TIME) INTO FLAG   
   MemcpyFlash2Ram(ADDRESS,  &MEASUREMENT, 16);
       if(MEASUREMENT[0]==0xffff) // IF TIME EQUAL TO 0xffff then it must unused data point
              goto EXITLOOP;
}
// DATAPOINT NOW CONTAINS THE NEXT DATAPOINT TO BE STORED
// IF DATAPOINT=MAXPOINTS+1 THEN MEMORY IS FULL
EXITLOOP: if(DATAPOINT== MAXPOINTS+2) 
          DATAPOINT= MAXPOINTS+1;
} 

void InitializeLCD(void)
// THIS ROUTINE IS CALLED WHENEVER EXITING FROM SLEEP MODE DUE TO SWITCH PRESS.
// IT TURNS POWER ON TO LCD MODULE AND INITIALIZES THE LCD AND DISPLAYS THE START MENU
{
    LCD_CONTROL=1; // TURN POWER ON TO LCD MODULE
    XOpenXLCD(); // INITIALIZE AND ERASE LCD
    MENUSTATE=200; // SET UP FOR WELCOME SCREEN
}

void LCDErase(void)
// The erase function erases the LCD
{
while(XBusyXLCD());
XWriteCmdXLCD(0x01);  // erase command
Delay10KTCYx(4); // delay 40,000 instruction cycles (5 msec)
while(XBusyXLCD());
}

void MemcpyFlash2Ram(rom far unsigned int *src,  ram far unsigned int *dst, int nBytes)	
//  COPIES nBytes FROM SOURCE ADDRESS IN FLASH PROGRAM MEMORY TO DESTINATION ADDRESS
//  IN RAM MEMORY.  
{
	char ram far* p = dst;
	char byte;
	unsigned int x, y;
	TBLPTR = (unsigned short long int )src; // promote src to 24 bits
	y = nBytes >>1;
	for(x=0; x<y; x++)
	{
		_asm  // read into TABLAT
		TBLRD
		_endasm
		TBLPTR++;  // INCREMENT TABLPTR
		*p = TABLAT;
		p++;
		_asm
		TBLRD
		_endasm
		TBLPTR++;
		*p = TABLAT;
		p++;
	}
}
	
void MemcpyRam2Flash(rom far unsigned int *dst, ram far unsigned int *src)	
//  COPIES 32 Bytes FROM SOURCE ADDRESS IN RAM TO DESTINATION ADDRESS
//  IN FLASH PROGRAM MEMORY.  DESTINATION ADDRESS MUST BE MULTIPLE OF
//  32 BYTES AND VALID FLASH ADDRESS.
//  MICROCHIP PIC18F2550 MANUAL SAYS THAT DATA MUST BE WRITTEN IN BLOCKS OF 64 BYTES,
//  BUT THIS IS INCORRECT.  BLOCKS OF 32 ARE USED.
{
	int x, y, blocks;
	unsigned short long int dst_1; // create a 24 bit variable
	static unsigned char k ; 
	static unsigned char m ; 
	static unsigned char l ;

	
     TBLPTR = ((unsigned short long int)dst);   // RESET TABLE POINTER
 	for(m=0; m<16; m++)	//  integer (ROM word) indexing.
		{
			x = *(src++);
			TABLAT = (x & 0xFF);//  store lower byte to holding registers.
			_asm
			TBLWT
			_endasm
			TBLPTR++;
			TABLAT = ((x & 0xFF00)>>8);	//store upper byte in holding registers.
			_asm
			TBLWT
			_endasm
			TBLPTR++;
		}
		TBLPTR=TBLPTR-32;
		EECON1bits.EEPGD=1; //POINT TO FLASH MEMORY
		EECON1bits.CFGS=0;  //ACCESS FLASH MEMORY
		EECON1bits.WREN=1;	//ENABLE WRITE TO MEMORY	
		INTCONbits.GIE=0; // DISABLE INTERRUPTS
		_asm  // WRITE THE 32 BYTES TO PROGRAM MEMORY, ASSEMBLY MUST NOT BE CHANGED
		MOVLW 0x55 
		MOVWF EECON2, 0 
		MOVLW 0xAA 
		MOVWF EECON2, 0 
		BSF EECON1, 1, 0 
		_endasm 
		EECON1bits.WREN=0;	//DISABLE WRITE TO MEMORY
		INTCONbits.GIE=1; // ENABLE INTERRUPTS
		TBLPTR=TBLPTR+32;
}

void Menu(void)
// This routine displays the menu based on MENUSTATE
{ char SETTIME[20];
  unsigned int TIMELEFT;
switch (MENUSTATE) {
	
//=======================USB MENU========================	
case 300:
Home(1);   
XputrsXLCD("USB STATUS      300");
    mInitializeUSBDriver();         // See usbdrv.h
    while(PORTAbits.RA1==1 ) // JUST LOOP HERE WHILE CONNECTED TO USB
{
	DataTransfer(); // Queue and send data transfers
}
//EXECUTE FOLLOWING WHEN UNPLUGGED FROM USB
   USBCheckBusStatus(); // SHUT DOWN USB MODULE
   SleepLCD(); // PUT MICROPROCESSOR TO SLEEP
// PROGRAM STARTS UP WITH INTERRUPT AND RETURNS HERE
   InitializeLCD(); // TURN ON POWER TO LCD AND INITIALIZE AND ERASE LCD AND SET MENUSTATE=200
break;  //exit switch
   	
//=======================MENU 1==========================
case 1:
MULTI=0; // MAKE SURE MULTI FLAG IS CLEARED
Home(1);
UpdateDateTime();
XputsXLCD(DATETIMESTRING);   
Home(2);
XputrsXLCD(" GEIGER COUNTER OFF ");
Home(3);
XputrsXLCD(" MEASUREMENT MENU");
Home(4);
XputrsXLCD(" UTILITY MENU");
Home(3); // select line 3
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=2; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH	   
   MENUSTATE=10; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

case 2:
Home(1);
UpdateDateTime();
XputsXLCD(DATETIMESTRING);
Home(4); // select line 4
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=3; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH	   
   MENUSTATE=70; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

case 3:
Home(1);
UpdateDateTime();
XputsXLCD(DATETIMESTRING);
Home(2); // select line 2
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=1; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   SleepLCD(); // PUT MICROPROCESSOR TO SLEEP
// PROGRAM STARTS UP WITH INTERRUPT AND RETURNS HERE
   InitializeLCD(); // TURN ON POWER TO LCD AND INITIALIZE AND ERASE LCD AND SET MENUSTATE=200
}
break;  //exit switch

//============================MENU 2 =======================
case 10:
Home(1);
XputrsXLCD("SAMPLE TIME UNITS  2");
Home(2);
XputrsXLCD(" SECONDS");
Home(3);
XputrsXLCD(" MINUTES");
Home(4);
XputrsXLCD(" GO BACK");
Home(2);
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=11; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH	   
   MENUSTATE=20; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

case 11:
Home(3); // select line 3
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=12; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH	   
   MENUSTATE=30; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

case 12:
Home(4); // go back selected
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=10; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH	   
   MENUSTATE=1; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

//===============================MENU 3=======================
case 20:
Home(1);
XputrsXLCD(" 15 SECONDS        3");
Home(2);
XputrsXLCD(" 30 SECONDS");
Home(3);
XputrsXLCD(" 45 SECONDS");
Home(4);
XputrsXLCD(" GO BACK");
Home(1);
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=21; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   TIME=15;	  //  15 SECONDS 
   MENUSTATE=40; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

case 21:
Home(2); // 30 sec selected
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=22; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   TIME=30;	   
   MENUSTATE=40; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch   
   
case 22:
Home(3); //select line 3
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=23; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   TIME=45;	   
   MENUSTATE=40; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch   
   
case 23:
Home(4); // select line 4
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=20; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH	   
   MENUSTATE=10; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

//=============================MENU 4========================
case 30:
Home(1);
XputrsXLCD(" 1 MINUTE          4");
Home(2);
XputrsXLCD(" 5 MINUTES");
Home(3);
XputrsXLCD(" 10 MINUTES");
Home(4);
XputrsXLCD(" GO BACK");
Home(1); // select line 1
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=31; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   TIME=60;		   
   MENUSTATE=40; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch   
   
case 31:
Home(2); // select line 2
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=32; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   TIME=300;		   
   MENUSTATE=40; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

case 32:
Home(3); // select line 3
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=33; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   TIME=600;  		   
   MENUSTATE=40; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

case 33:
Home(4); // select line 4
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=30; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH	   
   MENUSTATE=10; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

//===================MENU 5===============================
// TIMESTRING IN THE FOLLOWING SCREEN WILL BE SOMETHING
// LIKE TIMESTRING="600 SECONDS"
case 40:
sprintf(TIMESTRING,"TIME=%3u SECONDS",TIME); // FORMAT TIME STRING FOR LCD 
// CHECK TO SEE IF MEMORY FULL
if (DATAPOINT== MAXPOINTS+1)
      sprintf(TIMESTRING,"DATA MEMORY FULL!   ");
Home(1);
XputrsXLCD(" SINGLE SAMPLE     5");
Home(2);
XputrsXLCD(" MULTIPLE SAMPLE");
Home(3);
XputrsXLCD(" GO BACK");
Home(4);
XputsXLCD(TIMESTRING);
Home(1); //select 1
if (MULTI)
    MENUSTATE=41;
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=41; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1) // SINGLE MEASUREMENT
{
   EXECUTE=0;    // RESET SWITCH
   HV(1);
   Home(1);
   XputrsXLCD("PLEASE WAIT...     6");
// WAIT FOR HIGH VOLTAGE TO STABILIZE BEFORE SINGLE MEASUREMENT  
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES 
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES    
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES    
 
   StartCount(); 	   
   MENUSTATE=50; // SELECT NEW MENUSTATE
}
break;  //exit switch

case 41:
if (MULTI)
    EXECUTE=1;  // MULTI MEASUREMENTS
Home(2);  // select line 2
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=42; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
// CONTINUOUS MEASUREMENTS
   EXECUTE=0;    // RESET SWITCH
   MULTI=1;
   HV(1);
   Home(2);
   XputrsXLCD("PLEASE WAIT...      ");
// WAIT FOR HIGH VOLTAGE TO STABILIZE  
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES 
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES    
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES    
   StartCount(); 	   
   MENUSTATE=50; // SELECT NEW MENUSTATE
}
break;  //exit switch

case 42:
Home(3);  // select line 3
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=40; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH	   
   MENUSTATE=10; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch
//======================MENU 6============================
// COUNTSTRING IN THE FOLLOWING DISPLAY
// WILL BE SOMETHING LIKE COUNTSTRING="COUNT #002=000312"
// VOLTAGESTRING WILL BE LIKE VOLTAGESTRING="GM TUBE BIAS=500 V "
case 50:
UpdateVoltages();
UpdateCount();
TIMELEFT=TIME_TARGET - TIME_ELAPSED; // CALCULATE REMAINING TIME
Home(1);
sprintf(SETTIME,"PLEASE WAIT...%03u  6",TIMELEFT); // FORMAT TIME STRING FOR LCD
XputsXLCD(SETTIME);
Home(2);
XputsXLCD(COUNTSTRING);
Home(3);
XputsXLCD(VOLTAGESTRING);
Home(4);
XputrsXLCD(" ABORT MEASUREMENT");
Home(4); // select line 4
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   MULTI=0;  // KILL MULTI IF SET
   HV(0); // TURN OFF HIGH VOLTAGE
   COUNTING=0;
   MENUSTATE=1; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

//===============================MENU 7 =======================
case 60:
LCDErase();
if (MULTI)  // SAVE DATA AND MAKE ANOTHER MEASUREMENT
    EXECUTE=1;
Home(1);
XputrsXLCD("SAMPLING COMPLETE! 7");
Home(2);
XputsXLCD(COUNTSTRING);
Home(3);
XputrsXLCD(" SAVE DATA");
Home(4);
XputrsXLCD(" DISCARD DATA");
Home(3); // select line 3
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=61; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   Home(3);
   XputrsXLCD(" DATA SAVED!!!!");
   Home(3);
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES 
   SaveData();
   MENUSTATE=40; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

case 61:
Home(4); // select line 4
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=60; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   // DO NOTHING, DATA NOT SAVED	   
   MENUSTATE=1; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

//==========================MENU 8======================= 
case 70:            // UTILITY MENU
Home(1);
XputrsXLCD(" ERASE ALL DATA    8");
Home(2);
XputrsXLCD(" HV SCREEN");
Home(3);
XputrsXLCD(" SET CLOCK");
Home(4);
XputrsXLCD(" MAIN MENU");
Home(1); // select line 1
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=71; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   Home(1);
   XputrsXLCD(" ALL DATA ERASED!!!!");
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES 
   EraseAllData();  
   MENUSTATE=70; // SELECT UTILITY MENU
   LCDErase();
}
break;  //exit switch

case 71:
Home(2); // select line 2
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=72; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)  // CALIBRATION SCREEN SELECTED
{
   EXECUTE=0;    // RESET SWITCH	   
   MENUSTATE=80; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

case 72:
Home(3); // select line 3
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=73; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)  // CALIBRATION SCREEN SELECTED
{
   EXECUTE=0;    // RESET SWITCH	   
   MENUSTATE=90; // SELECT NEW MENUSTATE
   LCDErase();
}
break;  //exit switch

case 73:
Home(4); // select line 4
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=70; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH	   
   MENUSTATE=1; // MAIN MENU SELECTED
   LCDErase();
}
break;  //exit switch

//==========================MENU 9======================= 
case 80:
HV(1); //TURN ON HV FOR CALIBRATION
UpdateVoltages();
UpdateOnTime();
Home(1);
XputrsXLCD("HV SCREEN         9");
Home(2);
XputsXLCD(VOLTAGESTRING);
Home(3);
XputsXLCD(ONTIMESTRING);
Home(4);
XputrsXLCD(" UTILITY MENU");
Home(4);
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   MENUSTATE=70; // SELECT UTILITY MENU
   HV(0); // TURN OFF HIGH VOLTAGE
   LCDErase();
}
break;  //exit switch


//=====================MENU 10=======================
case 90:  // SET MONTH
StopClock(); // stop the clock while setting
sprintf(SETTIME,"%2u",MON); // FORMAT TIME STRING FOR LCD 
Home(1);
XputrsXLCD("SET MONTH       10");
Home(2);
XputsXLCD(SETTIME);
Home(3);
XputrsXLCD(" INCREMENT MONTH");
Home(4);
XputrsXLCD(" NEXT");
Home(3); // SELECT LINE 3
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=91; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
  MON=MON+1;
   if (MON>12)  // DO NOT LET MONTH EXCEED 12
        MON=1;
}
break;  //exit switch

case 91:
Home(4); // SELECT LINE 4
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=90; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   MENUSTATE=100;
   LCDErase();
}
break;  //exit switch


//=====================MENU 11=======================
case 100:  // SET DAY
sprintf(SETTIME,"%2u",DAY); // FORMAT TIME STRING FOR LCD 
Home(1);
XputrsXLCD("SET DAY OF MONTH  11");
Home(2);
XputsXLCD(SETTIME);
Home(3);
XputrsXLCD(" INCREMENT DAY");
Home(4);
XputrsXLCD(" NEXT");
Home(3);
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=101; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
  DAY=DAY+1;
   if (DAY>31)  // DO NOT LET MONTH EXCEED 31
        DAY=1;
}
break;  //exit switch

case 101:
Home(4); // SELECT LINE 4
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=100; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   MENUSTATE=110;
   LCDErase();
}
break;  //exit switch

//=====================MENU 12=======================
case 110:  // SET YEAR
sprintf(SETTIME,"%2u",YR); // FORMAT TIME STRING FOR LCD 
Home(1);
XputrsXLCD("SET YEAR         12");
Home(2);
XputsXLCD(SETTIME);
Home(3);
XputrsXLCD(" INCREMENT YEAR");
Home(4);
XputrsXLCD(" NEXT");
Home(3); // SELECT LINE 3
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=111; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
  YR=YR+1;
   if (YR>9)  // DO NOT LET YEAR EXCEED 2010
        YR=1;
}
break;  //exit switch

case 111:
Home(4); // SELECT LINE 4
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=110; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   MENUSTATE=120;
   LCDErase();
}
break;  //exit switch

//===================MENU 13============================
case 120:  // SET HOURS
sprintf(SETTIME,"%2u",HR); // FORMAT TIME STRING FOR LCD 
Home(1);
XputrsXLCD("SET HOURS         13");
Home(2);
XputsXLCD(SETTIME);
Home(3);
XputrsXLCD(" INCREMENT HOURS");
Home(4);
XputrsXLCD(" NEXT");
Home(3); // SELECT LINE 3
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=121; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
  HR=HR+1;
   if (HR>23)  // DO NOT LET HOURS EXCEED 24
         HR=0;
}
break;  //exit switch

case 121:
Home(4); // SELECT LINE 4
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=120; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   MENUSTATE=130;
   LCDErase;
}
break;  //exit switch

//=====================MENU 14=========================
case 130:  // SET MINUTES
sprintf(SETTIME,"%2u",MIN); // FORMAT TIME STRING FOR LCD 
Home(1);
XputrsXLCD("SET MINUTES       14");
Home(2);
XputsXLCD(SETTIME);
Home(3);
XputrsXLCD(" INCREMENT MINUTES");
Home(4);
XputrsXLCD(" UTILITY MENU");
Home(3); // SELECT LINE 3
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=131; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
  MIN=MIN+1;
   if (MIN>59)  // DO NOT LET MINUTES EXCEED 59
         MIN=0;
}
break;  //exit switch

case 131:
Home(4); // SELECT LINE 4
if (SCROLL==1)
   {
   SCROLL=0;    // RESET SWITCH	   
   MENUSTATE=130; // SELECT NEW MENUSTATE
   }
if (EXECUTE==1)
{
   EXECUTE=0;    // RESET SWITCH
   MENUSTATE=70;
   Home(2);
   UpdateDateTime(); //GET THE CURRENT TIME
   XputsXLCD(DATETIMESTRING);
   Home(3);
   StartClock(); // RESTART THE RTC
   XputrsXLCD(" CLOCK UPDATED!!!  ");
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES
   Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES 
   LCDErase();
}
break;  //exit switch


//===============WELCOME SCREEN======================
case 200:
Home(1);
XputrsXLCD(" LANG AND THOMPSON ");
Home(2);
XputrsXLCD("  GEIGER COUNTER");
Home(3);
XputsXLCD(VERSION);
Home(4);
UpdateDateTime();
XputsXLCD(DATETIMESTRING);
Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES
Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES
Delay10KTCYx(0); // delay ~2.6 MILLION CYCLES
MENUSTATE=1;
if (PORTAbits.RA1==1)  // IF WE HAVE POWER FROM USB 
    MENUSTATE=300;  // GO TO USB MENU
LCDErase();
break;  //exit switch

//==============DEFAULT ERROR CATCHER===============
// PROGRAM SHOULD NEVER GET HERE
default:  // default MENUSTATE
LCDErase();
Home(1);
XputrsXLCD("INVALID MENU OPTION ");
Home(2);
XputrsXLCD("SYSTEM SCREWUP!!!   ");
Home(3);
XputrsXLCD("CYCLE POWER TO CLEAR");
break;  //exit switch
}}

 void ReadSwitches(void)
// THIS ROUTINE CHECKS THE STATUS OF SWITCHES ON TWO CONSECUTIVE
// CALLS. IF THE STATUS HAS CHANGED AND THE NEW STATUS IS PRESSED
// THEN THE SCROLL OR EXECUTE FLAG IS SET.
{
    if (PORTBbits.RB2!=RB2PREV) 
    {
  	if (PORTBbits.RB2==1)  // scroll switch still depressed
        SCROLL=1;       // mark switch as closed
	else
	    SCROLL=0;
    }
   if (PORTBbits.RB3!=RB3PREV)
   { 
   if (PORTBbits.RB3==1) // execute switch still depressed 
    	   EXECUTE=1;
    	else
	       EXECUTE=0;
   }
   RB2PREV=PORTBbits.RB2;  // REMEMBER LAST SETTING
   RB3PREV=PORTBbits.RB3;
}

void Rsi(void)
// THIS ROUTINE IS SIMILAR TO READSWITCHES BUT IS CALLED FROM
// THE INTERRUPT ROUTINE
{
    if (PORTBbits.RB2!=RB2PREV) 
    {
  	if (PORTBbits.RB2==1)  // scroll switch still depressed
        SCROLL=1;       // mark switch as closed
	else
	    SCROLL=0;
    }
   if (PORTBbits.RB3!=RB3PREV)
   { 
   if (PORTBbits.RB3==1) // execute switch still depressed 
    	   EXECUTE=1;
    	else
	       EXECUTE=0;
   }
   RB2PREV=PORTBbits.RB2;  // REMEMBER LAST SETTING
   RB3PREV=PORTBbits.RB3;
}
	
void SaveData(void)
// THIS ROUTINE STORES THE CURRENT MEASUREMENT IN PROGRAM MEMORY
{   unsigned int MEASUREMENT[16];
	unsigned char m;
	unsigned int ADDRESS;
// first calculate address in program memory
   ADDRESS=DATAPOINT;	
   ADDRESS=DATASTART + (ADDRESS-1)*32;
// zero array   
   for(m=0; m<16; m++)
   {MEASUREMENT[m]=0;}
// assign data
   MEASUREMENT[0]=TIME;
   MEASUREMENT[1]=COUNT;
   MEASUREMENT[2]=MON;
   MEASUREMENT[3]=DAY;
   MEASUREMENT[4]=YR;
   MEASUREMENT[5]=HR;
   MEASUREMENT[6]=MIN;
   MEASUREMENT[7]=SEC; 
// WRITE THE DATA TO PROGRAM MEMORY       
   MemcpyRam2Flash(ADDRESS, MEASUREMENT);   	
//	EEpromWrite(DATAPOINT,COUNT,TIME); //STORE THE COUNT, COUNTTIME,DATE AND TIME OF COUNT
	DATAPOINT=DATAPOINT+1;
 if (DATAPOINT== MAXPOINTS+1) //DO NOT ALLOW MORE THAN  MAXPOINTS DATA POINTS TO BE STORED
     DATAPOINT= MAXPOINTS; 	
}

void SleepLCD(void)
// THIS ROUTINE IS CALLED TO
// TURNS POWER OFF TO LCD MODULE AND ENTER SLEEP MODE
{   
    LCDErase(); // ERASE THE LCD BEFORE GOING TO SLEEP
    // TURN OFF DISPLAY AND CURSOR
   while(XBusyXLCD());          // Wait if LCD busy
    XWriteCmdXLCD(0b00001000);   // Display OFF/Cursor OFF/BLINK OFF
    RED_LED=0; // MAKE SURE RED LED OFF
    GREEN_LED=0; // MAKE SURE GREEN LED OFF
    LCD_CONTROL=0; // TURN POWER OFF TO LCD MODULE
    SLEEPING=1; // SET SLEEPING FLAG
    OSCCONbits.IDLEN=0; // GO TO SLEEP NOT IDLE
    Sleep();  // GO TO SLEEP LITTLE MICROPROCESSOR
}

void StartCount(void)
{ TIME_TARGET=TIME; // SAVE DESIRED SECONDS
WriteTimer0(0);// ZERO THE PULSE COUNT
TIME_ELAPSED=0; // ZERO THE ELAPSED TIME IN SECOND INTERVALS
COUNTING=1; // SET COUNTING FLAG
}

void StartClock(void)
// THIS ROUTINE SETS UP AND STARTS INTERRUPTS FOR REAL TIME CLOCK
{
// I used the TIMER1 external oscillator described in AN580 with 32k crystal for real time clock
   DDRCbits.RC1=1;
  DDRCbits.RC0=1;
  OpenTimer1(TIMER_INT_ON & T1_8BIT_RW & T1_SOURCE_EXT & T1_PS_1_1 & T1_OSC1EN_ON & T1_SYNC_EXT_OFF);  
  INTCONbits.PEIE=1; //ENABLE PERIPHERAL INTERRUPTS  
  INTCONbits.GIE=1; //GLOBAL INTERRUPT ENABLE
}

void StopClock(void)
// THIS ROUTINE STOPS INTERRUPTS FOR REAL TIME CLOCK
{
  CloseTimer1(); // 
}


void Tic(void)
// THIS ROUTINE IS CALLED ONCE PER SECOND TO INCREMENT THE TIME
{
 SEC=SEC+1;
// INCREMENT MINUTE?
 if (SEC>59)
{ MIN=MIN+1;
  SEC=0;
}
// INCREMENT HOUR
  if (MIN>59)
{ HR=HR+1;
  MIN=0;
}
// INCREMENT DAY
  if (HR>23)
{ DAY=DAY+1;
  HR=0;
}
// INCREMENT MONTH
  if (DAY>31)
{ MON=MON+1;
  DAY=1;
}
// INCREMENT YEAR
  if (MON>12)
{ YR=YR+1;
  MON=1;
}
}


void UpdateCount(void)
//****************************************************************
// UPDATE THE COUNTSTRING BASED ON CURRENT COUNT
//****************************************************************
{// GENERATE A COUNT STRING [COUNT #123=00123  ]
    COUNT=ReadTimer0(); //GET CUMULATIVE GM COUNT FROM TIMER0
    sprintf(COUNTSTRING,"COUNT #%03u=%5u    ",DATAPOINT,COUNT); // FORMAT STRING FOR LCD 	
}

void UpdateOnTime(void)
//*****************************************
//   Displays ONTIME on LCD 
//*****************************************
{
    unsigned long TEMP;
    unsigned int TIME;
    TEMP=ON_TIME; // convert to long
    TEMP=TEMP*PERIOD_TIME_UNIT;
    TEMP=TEMP/4000; // convert to microseconds
    TIME=TEMP; // convert to int
    sprintf(ONTIMESTRING,"ON=%6d microsec",TIME); // FORMAT STRING FOR LCD
}

void UpdateDateTime(void)
{// UPDATES DATETIMESTRING BASED ON CURRENT SETTINGS OF 
 // MON, DAY, YR, HR, MIN, SEC  => 12/30/06 14:15:00
sprintf(DATETIMESTRING,"%02u/%02u/%02u %02u:%02u:%02u",MON,DAY,YR,HR,MIN,SEC);
}

void UpdateVoltages(void)
//********************************************************
//  UPDATE VOLTAGE STRING FOR DISPLAY
//********************************************************
{  // unsigned int DES_INT;
    unsigned int OUT_INT;
    long IVOUT;
   // long IVDES;
// convert 0-1023 to 0 to MAXVOLTS 
//    IVDES=(VDES*MAXVOLTS)/1023; // get integer component of voltage
//    DES_INT=IVDES; // convert from long to int
//    sprintf(DESVSTRING,"TARGET BIAS=%3u V",DES_INT); // FORMAT STRING FOR LCD 
// convert 0-1023 to 0.0 to MAXVOLTS.0     
    IVOUT=(VOUT*MAXVOLTS)/1023; // get integer component of voltage
    OUT_INT=IVOUT; // convert from long to int
    sprintf(VOLTAGESTRING,"GM TUBE BIAS=%3u V  ",OUT_INT); // FORMAT STRING FOR LCD with 10ths
  // negative numbers indicate faulty conversion, try again
}

void main(void)
//****************************************************************************
// main(void) the main program
//****************************************************************************
{   
// DEFINE THE INPUTS AND OUTPUTS 
// CHANGE THESE AND SYMBOLIC NAMES IN XXLCD.H IF I/O PINS CHANGE
    // RA0 IS USED BY ANALOG INPUT DEFINED IN ADCINIT
	DDRAbits.RA1=1; // MAKE RA1 AN INPUT FOR USB SENSE
    // RA2, RA3 AND RA5 ARE USED BY LCD AND DEFINED IN XXLCD.H
	DDRAbits.RA4=1; // MAKE RA4 AN INPUT FOR GM PULSE COUNT/TIMER0
    DDRBbits.RB0=0; // MAKE RB0 OUTPUT FOR LCD CONTROL
    DDRBbits.RB1=1; // RB1 IS UNUSED
    DDRBbits.RB2=1; // MAKE RB2 AN INPUT FOR SWITCH
    DDRBbits.RB3=1; // MAKE RB3 AND INPUT FOR SWITCH
    // RB4-RB7 ARE USED FOR LCD AND DEFINED IN XXLCD.H
    // RC0 AND RC1 ARE USED BY TIMER1 DEFINED STARTCLOCK
    // RC3,RC4 AND RC5 ARE USED BY USB 
    DDRCbits.RC2=0; // MAKE RC2 AN OUTPUT FOR MOSFET/PWM
    DDRCbits.RC6=0; // MAKE RC7 AN OUTPUT FOR RED LED
    DDRCbits.RC7=0; // MAKE RC7 AN OUTPUT FOR GREEN LED
    RED_LED=1; // TURN ON HV LED (RED)
    GREEN_LED=1; // TURN ON AUX LED (GREEN) 
    Delay10KTCYx(0); // startup delay
    HV_MOSFET=0; // MAKE SURE INPUT TO MOSFET IS TURNED OFF, NO DROP ACROSS COIL
HR=12; // ZERO CLOCK 
MIN=0;
SEC=0;
MON=1;
DAY=1;
YR=6;
     InitializeDataStore();
     ADCInit(); // set an0 to analog, all other porta are digital
     InitializeLCD(); // TURN ON POWER TO LCD AND INITIALIZE
     StartClock(); //  START REAL TIME CLOCK
 	 RED_LED=0; // TURN OFF HV LED (RED)
     GREEN_LED=0; // TURN OFF AUX LED (GREEN)
   
 // and loop forever...    
     while(1) 
     {
	 Menu(); // DISPLAY  WELCOME SCREEN THEN MENU
	 Delay10KTCYx(150); // delay .125 SEC
	 ReadSwitches(); // READ MENU SWITCHES
	 Delay10KTCYx(60); // delay .05 SEC 
	 }     
} //end main

