
#include <xc.h>
#include <stdint.h>
#include <p33EP512GP502.h>
#include "ili9341.h"
#include "image.h"
#include "proj-image.h"


int16_t  win_xe = 0xFFFF, 
         win_ye = 0xFFFF;

uint16_t _width = 240, 
        _height = 320, // Display w/h as modified by current rotation
        textcolor = 0xFFFF, 
        textbgcolor = 0, 
        addr_row = 0xFFFF, 
        addr_col = 0xFFFF;

void spiWait12(void)
{
}


void spiWait14(void)
{
}

void spiWait15(void)
{
}

/***************************************************************************************
** Function name:           spiwrite
** Description:             Write 8 bits to SPI port
***************************************************************************************/
void spiwrite(uint8_t c)
{
    int16_t temp;

    temp = SPI2BUF;                 // dummy read of the SPI2BUF register to clear the SPIRBF flag
    SPI2BUF = (uint16_t)c;          // write the data out to the SPI peripheral
    while( !SPI2STATbits.SPIRBF )   // wait for the data to be sent out
        ;
}


/***************************************************************************************
** Function name:           setAddrWindow
** Description:             define an area to receive a stream of pixels
***************************************************************************************/
// Chip select stays low, use setWindow() from sketches
void setAddrWindow(int16_t x0, int16_t y0, int16_t x1, int16_t y1)
{
  // Column addr set
  TFT_DC_C;
  TFT_CS_L;
  spiwrite(ILI9341_CASET);
  spiWait15();

  TFT_DC_D;
  spiwrite(x0 >> 8);
  spiWait12();
  addr_col = 0xFFFF;
  spiwrite(x0 & 0xFF); 
  spiWait12();
  
  if(x1!=win_xe) {
    spiwrite(x1 >> 8); 
    spiWait12();
    win_xe=x1;
    spiwrite(x1 & 0xFF); 
    spiWait14();
  }

  // Row addr set
  TFT_DC_C;
  spiwrite(ILI9341_PASET); 
  spiWait15();

  TFT_DC_D;
  spiwrite(y0 >> 8); 
  spiWait12();
  addr_row = 0xFFFF;
  spiwrite(y0 & 0xFF); 
  spiWait12();
  
  if(y1!=win_ye) {
    spiwrite(y1 >> 8); 
    spiWait12();
    win_ye=y1;
    spiwrite(y1 & 0xFF); 
    spiWait14();
  }

  // write to RAM
  TFT_DC_C;
  spiwrite(ILI9341_RAMWR); 
  spiWait14();

  //CS, HIGH;
  //TFT_CS_H;
  TFT_DC_D;
}

/***************************************************************************************
** Function name:           setWindow
** Description:             define an area to receive a stream of pixels
***************************************************************************************/
void setWindow(int16_t x0, int16_t y0, int16_t x1, int16_t y1)
{
  setAddrWindow(x0, y0, x1, y1);
  TFT_CS_H;
}


/***************************************************************************************
** Function name:           spiWrite16
** Descriptions:            Delay based assembler loop for fast SPI write
***************************************************************************************/
inline void spiWrite16(uint16_t data, int16_t count)
{
    while (count--) {
        spiwrite(data >> 8);
        spiwrite(data & 0xFF);
    }
}


/***************************************************************************************
** Function name:           fillRect
** Description:             draw a filled rectangle
***************************************************************************************/
void fillRect(int16_t x, int16_t y, int16_t w, int16_t h, uint16_t color)
{
  setAddrWindow(x, y, x + w - 1, y + h - 1);

  while (h--) spiWrite16(color, w);
  TFT_CS_H;
}


/***************************************************************************************
** Function name:           drawFastVLine
** Description:             draw a vertical line
***************************************************************************************/
void drawFastVLine(int16_t x, int16_t y, int16_t h, uint16_t color)
{
  setAddrWindow(x, y, x, y + h - 1);
  spiWrite16(color, h);
  TFT_CS_H;
}


/***************************************************************************************
** Function name:           drawFastHLine
** Description:             draw a horizontal line
***************************************************************************************/
void drawFastHLine(int16_t x, int16_t y, int16_t w, uint16_t color)
{
  setAddrWindow(x, y, x + w - 1, y);
  spiWrite16(color, w);
  TFT_CS_H;
}


/***************************************************************************************
** Function name:           fillScreen
** Description:             Clear the screen to defined colour
***************************************************************************************/
void fillScreen(uint16_t color)
{
  fillRect(0, 0, _width, _height, color);
}

// Initialization commands for ILI9341 screens
static const uint8_t ILI9341_cmds[] =
{
    21,
    0xEF, 3,                        // 2
    0x03, 0x80, 0x02,
    0xCF, 3,                        // 3
    0x00, 0xC1, 0x30,
    0xED, 4,                        // 4
    0x64, 0x03, 0x12, 0x81,
    0xE8, 3,                        // 5
    0x85, 0x00, 0x78,
    0xCB, 5,                        // 6
    0x39, 0x2C, 0x00, 0x34, 0x02,
    0xF7, 1,                        // 7
    0x20,
    0xEA, 2,                        // 8
    0x00, 0x00,
    ILI9341_PWCTR1, 1,              // 9 power control 
    0x23,                           // VRH[5:0] 
    ILI9341_PWCTR2, 1,              // 10 power control 
    0x10,                           // SAP[2:0];BT[3:0]  
    ILI9341_VMCTR1, 2,              // 11 VCM control 
    0x3e, 0x28,
    ILI9341_VMCTR2, 1,              // 12 VCM control2 
    0x86,                           // --
    ILI9341_MADCTL, 1,              // 13
    (ILI9341_MADCTL_MX | ILI9341_MADCTL_BGR),
    ILI9341_PIXFMT, 1,              // 14
    0x55,
    ILI9341_FRMCTR1, 2,             // 15
    0x00, 0x18,
    ILI9341_DFUNCTR, 3,             // 16
    0x08, 0x82, 0x27,
    0xF2, 1,                        // 17 3Gamma Function Disable 
    0x00,
    ILI9341_GAMMASET, 1,            // 18 Gamma curve selected 
    0x01,
    ILI9341_GMCTRP1, 15,            // 19 Set Gamma 
    0x0F, 0x31, 0x2B, 0x0C, 0x0E, 0x08, 0x4E, 0xF1, 0x37, 0x07, 0x10, 0x03, 0x0E, 0x09, 0x00,
    ILI9341_GMCTRN1, 15,            // 20
    0x00, 0x0E, 0x14, 0x03, 0x11, 0x07, 0x31, 0xC1, 0x48, 0x08, 0x0F, 0x0C, 0x31, 0x36, 0x0F,
    ILI9341_SLPOUT, ILI9341_INIT_DELAY,          // 21
    120,
    ILI9341_DISPON, 0,              // 22
};

/* 
 Routine to delay by the number of milliseconds specified
    We use 16 bit timer2 to provide the delay.
    Based on the RC oscillator being 7.37 MHz, our
    clock frequency into the timer is 62.645MHz
    At this speed, 1ms is a count of 62645
 */
void delayMs(unsigned int ms)
{
    PR2 = 62645;
    T2CON = 0;
    while (ms--) {
        IFS0bits.T2IF = 0;
        TMR2 = 0;
        T2CONbits.TON = 1;
        while (!IFS0bits.T2IF);
        T2CON = 0;
    }
}


void spi_begin(void)
{
}
void spi_end(void)
{
}

/***************************************************************************************
** Function name:           writecommand
** Description:             Send an 8 bit command to the TFT
***************************************************************************************/
void writecommand(uint8_t c)
{
  TFT_DC_C;
  TFT_CS_L;
  spiwrite(c);
  TFT_CS_H;
}


/***************************************************************************************
** Function name:           writedata
** Description:             Send a 8 bit data value to the TFT
***************************************************************************************/
void writedata(uint8_t c)
{
  TFT_DC_D;
  TFT_CS_L;
  spiwrite(c);
  TFT_CS_H;
}


/***************************************************************************************
** Function name:           commandList
** Description:             Get initialisation commands from FLASH and send to TFT
***************************************************************************************/
void commandList (const uint8_t *addr)
{
	uint8_t  numCommands, numArgs;
	uint8_t  ms;
    
    spi_begin();
    
	numCommands = *addr++;            // Number of commands to follow
	while (numCommands--)                           // For each command...
	{
		writecommand(*addr++);    // Read, issue command
		numArgs = *addr++;        // Number of args to follow
		ms = numArgs & ILI9341_INIT_DELAY;      // If hibit set, delay follows args
		numArgs &= ~ILI9341_INIT_DELAY;         // Mask out delay bit
		while (numArgs--)                       // For each argument...
		{
			writedata(*addr++); // Read, issue argument
		}

		if (ms)
		{
			ms = *addr++;     // Read post-command delay time (ms)
			delayMs((ms == 255 ? 500 : ms));
		}
	}
    
    spi_end();
}

/*
 Configure the dsPIC I/O pins for SPI and GPIO outputs,
 configure the SPI bus for 8 bit mode
 and write the initialisation sequence to the display 
 */
void ili9341Init(void)
{           
    TRISBbits.TRISB14 = 0;
    TRISBbits.TRISB15 = 0;
    TRISBbits.TRISB12 = 0;
    // SPI Bus setting override the TRIS settings
    TRISBbits.TRISB13 = 1; 
    TRISBbits.TRISB11 = 0;
    TRISBbits.TRISB10 = 0;

    // Map SPI signals to pins
    RPOR4 = 0x0809;
    RPINR22 = 0x002D;
    
    // Configure SPI bus
    SPI2STAT = 0;
    SPI2CON1 = 0x0037;  
    SPI2CON1bits.CKE = 1;
    SPI2CON2 = 0;
    SPI2STATbits.SPIEN = 1;
    
    
    // Reset the display
    TFT_RST_H;
    delayMs(5);
    TFT_RST_L;
    delayMs(20);
    TFT_RST_H;
    delayMs(150);
    
    // Write the initialisation data to the display
    commandList(ILI9341_cmds);
  
}


/***************************************************************************************
** Function name:           pushColors
** Description:             push an array of pixels for 16 bit raw image drawing
***************************************************************************************/
// Assumed that setWindow() has previously been called

void pushColors(uint16_t *data, uint16_t len)
{
  TFT_CS_L;
  while (len--) {
    spiwrite(*(data) >> 8);
    spiWait12();
    spiwrite(*(data++) & 0xFF);
    spiWait12();
  }
  
  TFT_CS_H;
}


// Show a bargraph of 256 values in the range 0..200
void displayArrayInit(void)
{
    // Build screen background
    fillScreen(TFT_BLACK);
    
    drawFastVLine(0, 0+61, 258, TFT_GREEN);
    drawFastVLine(200, 0+61, 258, TFT_GREEN);
    drawFastHLine(0, 0+61, 201, TFT_GREEN);
    drawFastHLine(0, 258+61, 201, TFT_GREEN);
    
    // Add logo
    setWindow(50,5,165,52);
    pushColors((uint16_t *)&epeLogo, 5568);
    // Add title
    setWindow(201,61,239,319);
    pushColors((uint16_t *)&projLogo, 10101);
}


// Show a bargraph of 256 values in the range 0..200
void displayArray(uint16_t *arrayData)
{
    uint16_t lp;
    uint16_t val;
    
    for (lp = 0; lp < 256; lp++) {
        val = *arrayData++;
        val = (val > 199) ? 199 : val;
        drawFastHLine(1, lp+1+61, val, TFT_WHITE);
        if (val < 199)
        drawFastHLine(1+val, lp+1+61, 199-val, TFT_BLACK);
    }
}

/***************************************************

  ORIGINAL LIBRARY HEADER

  This is our library for the Adafruit  ILI9341 Breakout and Shield
  ----> http://www.adafruit.com/products/1651

  Check out the links above for our tutorials and wiring diagrams
  These displays use SPI to communicate, 4 or 5 pins are required to
  interface (RST is optional)
  Adafruit invests time and resources providing this open source code,
  please support Adafruit and open-source hardware by purchasing
  products from Adafruit!

  Written by Limor Fried/Ladyada for Adafruit Industries.
  MIT license, all text above must be included in any redistribution
  
  Updated with new functions by Bodmer 14/4/15
 ****************************************************/
