/****************************************************************************************
	Main.c

	Main program file for the Tracker program

	
	Copyright (C) 2013 Geoff Graham (projects@geoffg.net)
	All rights reserved.
	
	This file and the program created from it are FREE FOR COMMERCIAL AND 
	NON-COMMERCIAL USE as long as the following conditions are aheared to.
	
	Copyright remains Geoff Graham's, and as such any Copyright notices in the 
	code are not to be removed.  If this code is used in a product,  Geoff Graham 
	should be given attribution as the author of the parts used.  This can be in 
	the form of a textual message at program startup or in documentation (online 
	or textual) provided with the program or product.
	
	Redistribution and use in source and binary forms, with or without modification,
	are permitted provided that the following conditions  are met:
	1. Redistributions of source code must retain the copyright notice, this list 
	   of conditions and the following disclaimer.
	2. Redistributions in binary form must reproduce the above copyright notice, this 
	   list of conditions and the following disclaimer in the documentation and/or 
	   other materials provided with the distribution.
	3. All advertising materials mentioning features or use of this software must 
	   display the following acknowledgement:
	   This product includes software developed by Geoff Graham (projects@geoffg.net)
	
	THIS SOFTWARE IS PROVIDED BY GEOFF GRAHAM ``AS IS'' AND  ANY EXPRESS OR IMPLIED 
	WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF 
	MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT 
	SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, 
	SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
	PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR 
	BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
	CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING 
	IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
	SUCH DAMAGE.
	
	The licence and distribution terms for any publically available version or
	derivative of this code cannot be changed.  i.e. this code cannot simply be copied 
	and put under another distribution licence (including the GNU Public Licence).
****************************************************************************************/



#include <plib.h>                       // include PIC32 peripheral library
#include <ctype.h>
#include <time.h>
#include "Tracker.h"
#include "Configuration Bits.h"

//** USB INCLUDES ***********************************************************
#include "./USB/Microchip/Include/USB/usb.h"
#include "./USB/Microchip/Include/USB/usb_function_cdc.h"
#include "./USB/HardwareProfile.h"

#include "./USB/Microchip/Include/GenericTypeDefs.h"
#include "./USB/Microchip/Include/Compiler.h"
#include "./USB/usb_config.h"
#include "./USB/Microchip/Include/USB/usb_device.h"

//** SD CARD INCLUDES ***********************************************************
#include "SDCard/SDCard.h"
#include "SDCard/FSconfig.h"

#define ON	    1
#define OFF	    0


/*****************************************************************************************************************************
Configuration defines
******************************************************************************************************************************/
#define USB_RX_BUFFER_SIZE	64
#define USB_TX_BUFFER_SIZE	64
#define GPS_BUF_SIZE        (8 * 1024)
#define MAX_GPS_MSG         80

/*****************************************************************************************************************************
Global functions
******************************************************************************************************************************/
void CheckUSB(void);
void USBPutchar(char c);
void USBPrintString(char *p);
void dump(char *p, int nbr);
void GetCurrentData(void);
void GPSCommand(char *msg);
void WaitForCard(void);
int Write0Param(FSFILE *fp, char *s);
int Write3Float(FSFILE *fp, char *s, double f1, double f2, double f3);
int WriteKmlStart(FSFILE *fp, int i1, int i2, int d, int m, int y, double f1, double f2);
int WriteKmlMark(FSFILE *fp, int i1, int i2, double f0, double f1, double f2);
int WriteKmlPoi(FSFILE *fp, int i0, int i1, int i2, double f1, double f2);
int WriteKmlEnd(FSFILE *fp, int i1, int i2, double f0, int i3, int i4, double f1, double f2);
int WritePoiLog(FSFILE *fp, int i1, int i2, int i3, int i4, int i5, int i6, int i7, int i8, double f1, double f2);
int WriteDiary(FSFILE *fp, int i1, int i2, int i3, int i4, int i5, int i6, int i7, double f1, double f2, double f3, double f4, int i8, int i9, double f5, double f6);

extern void USBDeviceTasks(void);


/*****************************************************************************************************************************
Global memory locations
******************************************************************************************************************************/
char USB_RxBuf[USB_RX_BUFFER_SIZE];
char USB_TxBuf[2][USB_TX_BUFFER_SIZE];
volatile int USB_NbrCharsInTxBuf, UsbBufSw;
volatile int CheckUsbCnt;
volatile int LedSdTimer;
volatile int GeneralTimer;

volatile char GpsBuf[GPS_BUF_SIZE];                                 // the GPS ring buffer
volatile int GpsBufHead, GpsBufTail;

volatile int MsgTimeout, ValidDataTimeout;                          // used to count down the timeout for gps data
int ValidMsg, ValidData, UseMiles;
double Lat, Lon, Alt, Speed, Course;                                // used to hold the GPS data
double KmSoFar;

float TimeZone;
int year, mth, day, hour, min, sec, secTotal;                       // used to hold the date/time

int KmlFlag, KmlTimer, KmlInterval;
int KmlMarkFlag, KmlMarkTimer, KmlMarkInterval;

int GpxFlag, GpxTimer, GpxInterval;
char GpxTime[32];

int NmeaFlag, NmeaTimer, NmeaInterval;
char GgaRecord[MAX_GPS_MSG + 6], RmcRecord[MAX_GPS_MSG + 6];

int PoiKml, PoiLog, PoiDiary;                                       // what to do when the POI button is pushed
int PoiButton;

int SdCardInit;                                                         // true if we have initialised the SD card
volatile int MediaChanged;                                          // true if the media has been removed


/*****************************************************************************************************************************
 Log files header, footers and placemarks
******************************************************************************************************************************/
#define KML_HEAD1   "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<kml xmlns=\"http://earth.google.com/kml/2.1\">\n<Document>\n<name>Tracker log %04d-%02d-%02d %02d:%02d #%02d</name>\n"
#define KML_HEAD2   "<Style id=\"style1\"><LineStyle><color>990000ff</color><width>4</width></LineStyle></Style><Placemark> <name>Polyline 1</name> <description></description> <styleUrl>#style1</styleUrl> <LineString> <altitudeMode>relative</altitudeMode> <coordinates>\n"
#define KML_START   "<Placemark><name>Start %d:%02d (%d/%d/%d)</name><Point><coordinates>%f,%f</coordinates></Point></Placemark>\n"
#define KML_MARK_KM "<Placemark><name>%d:%02d (%03.1fKm)</name><Point><coordinates>%f,%f</coordinates></Point></Placemark>\n"
#define KML_MARK_MI "<Placemark><name>%d:%02d (%03.1fMi)</name><Point><coordinates>%f,%f</coordinates></Point></Placemark>\n"
#define KML_POI     "<Placemark><name>POI #%d %d:%02d</name><Point><coordinates>%f,%f</coordinates></Point></Placemark>\n"
#define KML_END_KM  "<Placemark><name>End %d:%02d (%03.1fKm  %dh %02dm)</name><Point><coordinates>%f,%f</coordinates></Point></Placemark>\n"
#define KML_END_MI  "<Placemark><name>End %d:%02d (%03.1fMi  %dh %02dm)</name><Point><coordinates>%f,%f</coordinates></Point></Placemark>\n"
#define KML_FOOT    "</coordinates></LineString></Placemark>\n"
#define KML_CLOSE   "</Document></kml>\n"

#define GPX_HEAD1   "<?xml version=\"1.0\"?>\n<gpx creator=\"GPS Tracker http://www.geoffg.net/tracker.html/\" version=\"1.0\" xmlns=\"http://www.topografix.com/GPX/1/0\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\""
#define GPX_HEAD2   " xsi:schemaLocation=\"http://www.topografix.com/GPX/1/0 http://www.topografix.com/GPX/1/0/gpx.xsd\">\n<trk>\n<name>Tracker log %04d-%02d-%02d %02d:%02d #%02d</name>\n<trkseg>\n"
#define GPX_DATA    "<trkpt lat=\"%f\" lon=\"%f\"><ele>%f</ele><time>%s</time><course>%f</course><speed>%f</speed></trkpt>\n"
#define GPX_FOOT    "</trkseg></trk></gpx>\n"

#define POILOG_HEAD "Date\tTime\tTrack\tPOI #\tLatitude\tLongitude\r\n"
#define POILOG_DATA "%d/%d/%d\t%d:%02d\t%02d--#%02d\t%d\t%f\t%f\r\n"

#define DIARY_HEAD "Date\tTime\tTrack\tStart Latitude\tStart Longtitude\tEnd Latitude\tEnd Longitude\tDuration\tPrivate\tBusiness\r\n"
#define DIARY_DATA_PRIVATE  "%d/%d/%d\t%d:%02d\t%02d--#%02d\t%f\t%f\t%f\t%f\t%dh %02dm\t%03.1f\t\r\n"
#define DIARY_DATA_BUSINESS "%d/%d/%d\t%d:%02d\t%02d--#%02d\t%f\t%f\t%f\t%f\t%dh %02dm\t\t%03.1f\r\n"

// This is the default configuration file that is written when no configuration file is found
#define TRACKERTXT "\
' GPS Tracker V1.1 (c) 2013 Geoff Graham.\r\n\
' For updates and notes go to http://geoffg.net/tracker.html\r\n\
\r\n\
' Anything after a single quote (') is ignored as a comment.\r\n\
\r\n\
' The TZ entry sets the current Time Zone in hours as an offset from UTC (a decimal point can be used).\r\n\
' For example,  TZ +9.5  is Australian Central Standard Time which is nine and a half hours ahead of UTC.\r\n\
 TZ +10       ' this is Australian Eastern Standard Time.\r\n\
 USE KM       ' this can be changed to USE MILES to record distance in miles.\r\n\
\r\n\
 KML 5        ' record a Google Earth KML track every five seconds (zero means do not record).\r\n\
 KMLMARK 3600 ' place an interval pin every hour (ie, 3600 seconds, zero means do not place a pin).\r\n\
\r\n\
 GPX 60       ' record a GPS eXchange Format track once every minute (zero means do not record).\r\n\
 NMEA 0       ' record a the NMEA data every this number of seconds (zero means do not record).\r\n\
\r\n\
' What to do when the POI button is pressed.  Multiple entries can be used if you want to use the POI button to\r\n\
' record data in more than one format.  Delete the leading comment character (') to enable a function.\r\n\
' POIPIN      ' when the POI button is pressed place a pin on the KML track marking the current location.\r\n\
' POILOG      ' when the POI button is pressed record the current location in the file LOG.XLS\r\n\
' POIDIARY    ' record each journey in DIARY.XLS and when the PIO button is pressed mark the trip as business.\r\n"


/*****************************************************************************************************************************
 Main program
******************************************************************************************************************************/
int main() {
    int i, BaudChange, KmlSeq, GpxSeq, NmeaSeq, PoiSeq, Business;
    double StartLat, StartLon;
    char *p, SBuf[256];
    char KmlSubDir[13], GpxSubDir[13], NmeaSubDir[13], KmlFile[13], GpxFile[13], NmeaFile[13];
    FSFILE *KmlFp, *GpxFp, *NmeaFp, *LogFp, *DiaryFp;
    SearchRec SRec;

    
    // initial setup of the chip (or why the hell are the defaults not reasonable!!)
    ANSELA = 0; ANSELB = 0;			                                // Default all pins to digital
    mJTAGPortEnable(0);                                             // turn off jtag
    SYSTEMConfigPerformance(CLOCKFREQ);                             // optimise for speed
    INTEnableSystemMultiVectoredInt();                              // allow vectored interrupts

    USBDeviceInit();                                                // Initialise USB module SFRs and firmware

    // start up the tracker
    // we return to this point if there has been an error serious enough to restart the program
    RetryStartup:
    
    // init variables
    USB_NbrCharsInTxBuf = 0;
    UsbBufSw = 0;
    CheckUsbCnt = 0;
    GpsBufHead = GpsBufTail = 0;
    ValidMsg = ValidData = false;
    BaudChange = 0;
    MediaChanged = true;
    TimeZone = 10;
    KmlInterval = 5;
    KmlMarkInterval = 3600;
    GpxInterval = 0;
    NmeaInterval = 0;
    SdCardInit = false;
    PoiButton = false;
    KmlSeq = 0;
    Business = false;
    StartLat = StartLon = 0;
    KmlFp = GpxFp = NmeaFp = LogFp = DiaryFp = 0;

    // setup the SPI interface for the SD Card
    PPSInput(3, SDI2, RPA4); PPSOutput(2, RPB5, SDO2); //PPSLock;

    // init the power and fault LEDs
    LED_PWR_ODC = ON; LED_PWR_TRIS = OUTPUT; LED_PWR_ON;            // setup the power LED
    LED_GPS_ODC = ON; LED_GPS_TRIS = OUTPUT; LED_GPS_OFF;           // setup the GPS LED
    LED_FAULT_ODC = ON; LED_FAULT_TRIS = OUTPUT; LED_FAULT_OFF;     // setup the fault LED

    SD_CD_TRIS = INPUT; CNCONBSET = (1 << 15); CNENBSET = (1 << 13); CNPUBSET = (1 << 13);        // enable the Card Detect (CD) input for the SD card and set a pullup
    SD_WE_TRIS = INPUT; CNCONBSET = (1 << 15); CNENBSET = (1 << 3); CNPUBSET = (1 << 3);          // enable the Write Enable (WE) input for the SD card and set a pullup

    // Initialise timer 4 - used for internal timekeeping.
    PR4 = 200 * ((BUSFREQ/2)/1000000) - 1;                          // 200 uSec
    T4CON = 0x8010;                                                 // T4 on, prescaler 1:2
    LedSdTimer = 0;
    GeneralTimer = 0;
    mT4SetIntPriority(1);                                           // lower priority
    mT4ClearIntFlag();                                              // clear interrupt flag
    mT4IntEnable(1);                                                // enable interrupt

    NEW_TRACK_TRIS = INPUT;
    POI_BUTTON_TRIS = INPUT; CNCONBSET = (1 << 15); CNENBSET = (1 << 14); CNPUBSET = (1 << 14);  // enable the POI button input and set a pullup
    
    // setup the UART interface for the GPS module
    PPSInput(2, U2RX, RPB1); PPSOutput(4, RPB0, U2TX);
    UARTConfigure(UART2, UART_ENABLE_PINS_TX_RX_ONLY);
    UARTSetFifoMode(UART2, UART_INTERRUPT_ON_TX_NOT_FULL | UART_INTERRUPT_ON_RX_NOT_EMPTY);
    UARTSetLineControl(UART2, UART_DATA_SIZE_8_BITS | UART_PARITY_NONE | UART_STOP_BITS_1);
    UARTSetDataRate(UART2, BUSFREQ, 9600);
    UARTEnable(UART2, UART_ENABLE_FLAGS(UART_PERIPHERAL | UART_RX | UART_TX));

    // Configure UART2 RX Interrupt
    INTEnable(INT_SOURCE_UART_RX(UART2), INT_ENABLED);
    INTSetVectorPriority(INT_VECTOR_UART(UART2), INT_PRIORITY_LEVEL_2);
    INTSetVectorSubPriority(INT_VECTOR_UART(UART2), INT_SUB_PRIORITY_LEVEL_0);

    INTEnableInterrupts();

    // wait for at least eight seconds in case the ignition switch moves on to start the engine - which could glitch operation
    // the actual wait loop is 20 lines down (after we have got communication with the GPS)
    GeneralTimer = 8000;

    //  make sure that we can get some data from the GPS, change baud rate if necessary
    MsgTimeout = 1500;
    ValidMsg = false;
    while(!ValidMsg) {
        GetCurrentData();                                           // test the incoming GPS data
        if(MsgTimeout == 0) {                                       // if nothing received
            BaudChange++;
            UARTSetDataRate(UART2, BUSFREQ, (BaudChange & 1) ? 4800 : 9600); // change baud rate
            GpsBufTail = GpsBufHead; ValidData = false;             // empty the buffer so that we start with fresh data
            MsgTimeout = 1500;                                      // and reset the timer
            if(BaudChange > 1) {                                    // if we have already tried both baud rates
                GPSCommand((BaudChange & 1) ? "$PSRF104,00,00,00,00,00,00,12,08*29":"PMTK104");  // reset the GPS (it might work)
                MsgTimeout = 8000;                                  // and use a longer timeout
                LED_FAULT_ON;
            }
        }
    }
    LED_FAULT_OFF;                                                  // clear any error indication caused by the GPS
    while(GeneralTimer != 0) GetCurrentData();                      // wait out the remainder of the startup time

    // we return to this point if there has been an error with the SD card
    RetrySdCard:

    if(SdCardInit) {                                                // if we were using the card
        LED_FAULT_ON;                                               // this is an error condition
        // close any files that have been left open
        if(KmlFp) FSfclose(KmlFp);
        if(GpxFp) FSfclose(GpxFp);
        if(NmeaFp) FSfclose(NmeaFp);
        if(LogFp) FSfclose(LogFp);
        if(DiaryFp) FSfclose(DiaryFp);
        KmlFp = GpxFp = NmeaFp = LogFp = DiaryFp = 0;
        while(MDD_MediaDetect()) GetCurrentData();                  // wait for the card to be removed
        GeneralTimer = 5000;
        while(GeneralTimer != 0) GetCurrentData();                  // then wait for 5 seconds
    }
    
    if(!MDD_MediaDetect()) {                                        // if the card is not present
        LED_FAULT_ON;                                               // flag an error
        while(!MDD_MediaDetect()) GetCurrentData();                 // wait for the card to be inserted
        GeneralTimer = 5000;
        while(GeneralTimer != 0) GetCurrentData();                  // then wait for 5 seconds
    }

    LED_FAULT_OFF;                                                  // clear any error indication caused by the SD card
    SdCardInit = true;                                              // from now on any SD card errors will require the card to be removed and re inserted
    if(!FSInit()) goto RetrySdCard;                                 // initialise the card
    
    GetCurrentData();
    MediaChanged = false;

    // try to open the configuration file as a test and if it fails write the default configuration data
    FSchdir("\\");
    KmlFp = FSfopen("TRACKER.TXT", "r");
    if(!KmlFp) {
        // if the file was not found create it on the SD card
        if(!(KmlFp = FSfopen("TRACKER.TXT", "w"))) goto RetrySdCard;
        if(!FSfwrite(TRACKERTXT, 1, strlen(TRACKERTXT), KmlFp)) goto RetrySdCard;
        FSfclose(KmlFp);
        FSfopen("TRACKER.TXT", "r");
    }
    if(!KmlFp) goto RetrySdCard;                                    // if there was an error retry opening the configuration file
    
    GetCurrentData();

    // load the configuration data from the root of the SD card.  If this fails then it is a fatal error
    if(KmlFp) {
        char c, pc = 0;
        int ic;
        for(ic = i = 0; i < sizeof(SBuf) - 1; ) {
            if(!FSfread(&c, 1, 1, KmlFp)) break;                    // read a character
            if(c == '\n' || c == '\r') ic = false;                  // if it is a new line it is the end of a comment
            if(c == '\'') ic = true;                                // if it is ' it is the start of a comment
            if(!ic && !(pc == ' ' && c == ' ')) SBuf[i++] = c;      // if we are not in a comment and it is not multiple spaces save the char in our buffer
            pc = c;                                                 // this is used to detect multiple space chars (which we do not want)
            GetCurrentData();
        }
        SBuf[i] = 0;                                                // terminate the string retrieved
        FSfclose(KmlFp);
    }
    
    // convert the string loaded from the configuration file to a format ready for scanning
    for(p = SBuf; *p; p++) {
        if(!isalnum(*p) && *p != '-' && *p != '.') *p = ' ';
        *p = toupper(*p);
    }

    // load the configuration data
    if(strstr(SBuf, "TZ ")) TimeZone = atof(strstr(SBuf, "TZ ") + 3);
    if(strstr(SBuf, "KML ")) KmlInterval = atoi(strstr(SBuf, "KML ") + 4);
    if(strstr(SBuf, "KMLMARK ")) KmlMarkInterval = atoi(strstr(SBuf, "KMLMARK ") + 7);
    if(strstr(SBuf, "GPX ")) GpxInterval = atoi(strstr(SBuf, "GPX ") + 4);
    if(strstr(SBuf, "NMEA ")) NmeaInterval = atoi(strstr(SBuf, "NMEA ") + 5);
    UseMiles = (strstr(SBuf, "USEMILES") != NULL);
    PoiKml = (strstr(SBuf, "POIPIN") != NULL);
    PoiLog = (strstr(SBuf, "POILOG") != NULL);
    PoiDiary = (strstr(SBuf, "POIDIARY") != NULL);
    if(TimeZone < -24 || TimeZone > 24 || KmlMarkInterval < 0 || KmlInterval < 0 || GpxInterval < 0 || NmeaInterval < 0)
        goto RetrySdCard;                                           // if there was an error retry getting the config data
    
    // at this point the GPS module is working (but it may not have a lock on our position yet)
    LED_FAULT_OFF;                                                  // clear any error indications
    
    // wait for the GPS to get a lock
    while(!ValidData) {
        if(!MDD_MediaDetect()) goto RetrySdCard;
        GetCurrentData();
   	 if(MsgTimeout == 0) {
            LED_FAULT_ON;
            goto RetryStartup;
        }
    }
    
    WaitOnNewTrack:
    while(!NEW_TRACK) {                                             // now wait for the NEW_TRACK input to go ready
        GetCurrentData();
    	if(MsgTimeout == 0) {
            LED_FAULT_ON;
            goto RetryStartup;
        }
        if(!MDD_MediaDetect()) goto RetrySdCard;
    }

    LED_FAULT_OFF;                                                  // clear any error indication caused by the SD card

    // make sure that we start with a current GPS record
    GpsBufTail = GpsBufHead; ValidData = false;                     // empty the buffer so that we start with fresh data
    ValidDataTimeout = MsgTimeout = 3000;

    do {                                                            // wait for the record (will be less than a second)
        GetCurrentData();
        if(MsgTimeout == 0) { LED_FAULT_ON; goto RetryStartup; }    // if the GPS has developed a fault
    } while(!ValidData);

    ValidDataTimeout = MsgTimeout = 10000;

    if(KmlInterval) {
        // we are ready to go with the KML format, so create the main directory
        SetClockVars(year + 2000, mth, day, hour, min, sec);
        FSchdir("\\");
        if(FindFirst("GEarth", ATTR_DIRECTORY, &SRec) != 0) FSmkdir("GEarth");
        if(FSchdir("GEarth") != 0) goto RetrySdCard;

        // and the sub directory
        sprintf(KmlSubDir, "%04d-%02d", year + 2000, mth);
        if(FindFirst(KmlSubDir, ATTR_DIRECTORY, &SRec) != 0) FSmkdir(KmlSubDir);
        if(FSchdir(KmlSubDir) != 0) goto RetrySdCard;

        // find the highest file sequence number used and increment it for the next number
        KmlSeq = 0;
        sprintf(SBuf, "%02d--#*.KML", day);
        i = FindFirst(SBuf, ATTR_ARCHIVE, &SRec);
        while(i == 0) {
            if(atoi(&SRec.filename[5]) > KmlSeq)
                KmlSeq = atoi(&SRec.filename[5]);
            i = FindNext(&SRec);
        }
        KmlSeq++;

        // open the new file
        sprintf(KmlFile, "%02d--#%02d.KML", day, KmlSeq);
        if(!(KmlFp = FSfopen(KmlFile, "w"))) goto RetrySdCard;
        if(!NEW_TRACK) goto Shutdown;
 
        // write the KML headers
        sprintf(SBuf, KML_HEAD1, year + 2000, mth, day, hour, min, KmlSeq);
        FSfwrite(SBuf, strlen(SBuf), 1, KmlFp);
        if(!NEW_TRACK) goto Shutdown;
        WriteKmlStart(KmlFp, hour, min, day, mth, year, Lon, Lat);   // write the start placemark
        if(!Write0Param(KmlFp, KML_HEAD2)) goto RetrySdCard;
        if(!NEW_TRACK) goto Shutdown;
        if(!Write3Float(KmlFp, "%f,%f,%f\n", Lon, Lat, Alt)) goto RetrySdCard;  // write the current location
        if(!NEW_TRACK) goto Shutdown;

        KmlFlag = true;
        KmlTimer = KmlInterval;
        KmlMarkFlag = false;
        KmlMarkTimer = KmlMarkInterval;
    }
     
    if(GpxInterval) {
        // we are ready to go with the GPX format, so create the main directory
        SetClockVars(year + 2000, mth, day, hour, min, sec);
        FSchdir("\\");
        if(FindFirst("GPX", ATTR_DIRECTORY, &SRec) != 0) FSmkdir("GPX");
        if(FSchdir("GPX") != 0) goto RetrySdCard;

        // and the sub directory
        sprintf(GpxSubDir, "%04d-%02d", year + 2000, mth);
        if(FindFirst(GpxSubDir, ATTR_DIRECTORY, &SRec) != 0) FSmkdir(GpxSubDir);
        if(FSchdir(GpxSubDir) != 0) goto RetrySdCard;

        // find the highest file sequence number used and increment it for the next number
        GpxSeq = 0;
        sprintf(SBuf, "%02d--#*.GPX", day);
        i = FindFirst(SBuf, ATTR_ARCHIVE, &SRec);
        while(i == 0) {
            if(atoi(&SRec.filename[5]) > GpxSeq)
                GpxSeq = atoi(&SRec.filename[5]);
            i = FindNext(&SRec);
        }
        GpxSeq++;
        
        // open the new file
        sprintf(GpxFile, "%02d--#%02d.GPX", day, GpxSeq);
        if(!(GpxFp = FSfopen(GpxFile, "w"))) goto RetrySdCard;
        if(!NEW_TRACK) goto Shutdown;

        // write the GPX headers
        Write0Param(GpxFp, GPX_HEAD1);
        sprintf(SBuf, GPX_HEAD2, year + 2000, mth, day, hour, min, GpxSeq);
        FSfwrite(SBuf, strlen(SBuf), 1, GpxFp);
        if(!NEW_TRACK) goto Shutdown;

        GpxFlag = true;
        GpxTimer = GpxInterval;
    }
    
    if(NmeaInterval) {
        // we are ready to go with the NMEA format, so create the main directory
        SetClockVars(year + 2000, mth, day, hour, min, sec);
        FSchdir("\\");
        if(FindFirst("NMEA", ATTR_DIRECTORY, &SRec) != 0) FSmkdir("NMEA");
        if(FSchdir("NMEA") != 0) goto RetrySdCard;

        // and the sub directory
        sprintf(NmeaSubDir, "%04d-%02d", year + 2000, mth);
        if(FindFirst(NmeaSubDir, ATTR_DIRECTORY, &SRec) != 0) FSmkdir(NmeaSubDir);
        if(FSchdir(NmeaSubDir) != 0) goto RetrySdCard;

        // find the highest file sequence number used and increment it for the next number
        NmeaSeq = 0;
        sprintf(SBuf, "%02d--#*.TXT", day);
        i = FindFirst(SBuf, ATTR_ARCHIVE, &SRec);
        while(i == 0) {
            if(atoi(&SRec.filename[5]) > NmeaSeq)
                NmeaSeq = atoi(&SRec.filename[5]);
            i = FindNext(&SRec);
        }
        NmeaSeq++;
        
        // open the new file
        sprintf(NmeaFile, "%02d--#%02d.TXT", day, NmeaSeq);
        if(!(NmeaFp = FSfopen(NmeaFile, "w"))) goto RetrySdCard;
        if(!NEW_TRACK) goto Shutdown;

        NmeaFlag = true;
        NmeaTimer = NmeaInterval;
    }
    
    if(PoiLog) {
        SetClockVars(year + 2000, mth, day, hour, min, sec);
        FSchdir("\\");
        if(!(LogFp = FSfopen("LOG.XLS", "a"))) goto RetrySdCard;
        if(LogFp->size == 0) {
            // the log file is empty so we must write the appropriate header row
            if(!Write0Param(LogFp, POILOG_HEAD)) goto RetrySdCard;
        }
        if(!NEW_TRACK) goto Shutdown;
    }

    if(PoiDiary) {
        SetClockVars(year + 2000, mth, day, hour, min, sec);
        FSchdir("\\");
        if(!(DiaryFp = FSfopen("DIARY.XLS", "a"))) goto RetrySdCard;
        if(DiaryFp->size == 0)
            if(!Write0Param(DiaryFp, DIARY_HEAD)) goto RetrySdCard; // the diary file is empty so we must write the appropriate header row
        if(!NEW_TRACK) goto Shutdown;
        Business = false;                                           // will be set true if this is a business trip
        StartLat = Lat; StartLon = Lon;                             // the start of this trip
    }

    secTotal = 0;                                                   // counts how long for this trip
    KmSoFar = 0;                                                    // counts the distance into the trip
    Business = false;                                               // flag if this is a business trip
    PoiSeq = 0;                                                     // counts the number of POIs



    // this is the main loop, get the data and write it out
    while(1) {
        int DataWritten = false;
        if(!NEW_TRACK) goto Shutdown;
        GetCurrentData();
        if(MsgTimeout == 0) {                                       // if the GPS has developed a fault
            LED_FAULT_ON;
            goto RetryStartup;                                      // go back to the start
        }
        
        if(ValidData) {                                             // if we have some valid data
            SetClockVars(year + 2000, mth, day, hour, min, sec);    // set the time to be recorded as the file modification time
            if(KmlInterval) {                                       // if we are recording KML data
                if(KmlFlag) {                                       // and it is time to write the data out
                    KmlFlag = false;
                    if(!Write3Float(KmlFp, "%f,%f,%f\n", Lon, Lat, Alt))  // write the current location
                        goto RetrySdCard;                           // on error retry opening the log file(s)
                    if(!NEW_TRACK) goto Shutdown;
                    if(KmlMarkInterval && KmlMarkFlag) {            // it is time to write a KML marker pin
                        KmlMarkFlag = false;
                        Write3Float(KmlFp, "%f,%f,%f\n", Lon, Lat, Alt);        // write the current location
                        if(!NEW_TRACK) goto Shutdown;
                        Write0Param(KmlFp, KML_FOOT);
                        WriteKmlMark(KmlFp, hour, min, KmSoFar, Lon, Lat);      // write the placemark (pin)
                        Write0Param(KmlFp, KML_HEAD2);
                        if(!Write3Float(KmlFp, "%f,%f,%f\n", Lon, Lat, Alt))  // and write the current location again so that there is not a break in the track
                            goto RetrySdCard;                       // on error retry opening the log file(s)
                    }
                    DataWritten = true;
                }
            }
                        
            if(!NEW_TRACK) goto Shutdown;
            if(GpxInterval) {                                       // if we are recording GPX data
                if(GpxFlag) {                                       // and it is time to write the data out
                    GpxFlag = false;
                    sprintf(SBuf, GPX_DATA, Lat, Lon, Alt, GpxTime, Course, Speed * 0.51444444444);
                    if(!Write0Param(GpxFp, SBuf))                   // write the current location
                        goto RetrySdCard;                           // on error retry opening the log file(s)
                    DataWritten = true;
                }
            }
            
            if(!NEW_TRACK) goto Shutdown;
            if(NmeaInterval) {                                      // if we are recording NMEA data
                if(NmeaFlag) {                                      // and it is time to write the data out
                    NmeaFlag = false;
                    Write0Param(NmeaFp, GgaRecord);                 // write the GGA record
                    if(!Write0Param(NmeaFp, RmcRecord))             // write the RMC record
                        goto RetrySdCard;                           // on error retry opening the log file(s)
                    DataWritten = true;
                }
            }
            
            if(PoiButton) {
                Business = true;
                PoiSeq++;
                
                if(PoiKml) {
                    Write3Float(KmlFp, "%f,%f,%f\n", Lon, Lat, Alt);// write the current location
                    if(!NEW_TRACK) goto Shutdown;
                    Write0Param(KmlFp, KML_FOOT);
                    WriteKmlPoi(KmlFp, PoiSeq, hour, min, Lon, Lat);// write the placemark (pin)
                    Write0Param(KmlFp, KML_HEAD2);
                    if(!Write3Float(KmlFp, "%f,%f,%f\n", Lon, Lat, Alt))  // and write the current location again so that there is not a break in the track
                        goto RetrySdCard;
                    if(!NEW_TRACK) goto Shutdown;
                    DataWritten = true;
                }
                
                if(PoiLog) {
                    if(!WritePoiLog(LogFp, day, mth, year, hour, min, day, KmlSeq, PoiSeq, Lat, Lon)) goto RetrySdCard;
                    DataWritten = true;
                }
                
                PoiButton = false;                                  // signal that we have dealt with the button press
            }
                    
            ValidData = false;                                      // signal that we have used the data
            if(DataWritten) LedSdTimer = 100;                       // blink the power led
            DataWritten = false;
        }
    }

    // shutdown routine
    // control jumps to here on power loss
    Shutdown:
    INTDisableInterrupts();                                         // maximum speed required so no interrupts
    if(KmlInterval) {                                               // if we are recording KML data
        if(KmSoFar < 0.1) {                                         // if this is a trivial trip
            FSfclose(KmlFp);
            FSchdir("\\"); FSchdir("GEARTH"); FSchdir(KmlSubDir);   // get to the correct directory
            FSremove(KmlFile);                                      // and delete the file
        } else {                                                    // otherwise, close off the track
            Write3Float(KmlFp, "%f,%f,%f\n", Lon, Lat, Alt);        // write the current location, don't care about errors
            Write0Param(KmlFp, KML_FOOT);                           // write the footer
            WriteKmlEnd(KmlFp, hour, min, KmSoFar, secTotal/3600, (secTotal/60)%60, Lon, Lat); // write the end placemark
            Write0Param(KmlFp, KML_CLOSE);                           // close off the KML file
            FSfclose(KmlFp);
        }
    }

    if(GpxInterval) {                                               // if we are recording GPX data
        if(KmSoFar < 0.1) {                                         // if this is a trivial trip
            FSfclose(GpxFp);
            FSchdir("\\"); FSchdir("GPX"); FSchdir(GpxSubDir);      // get to the correct directory
            FSremove(GpxFile);                                      // and delete the file
        } else {                                                    // otherwise, close off the track
            Write0Param(GpxFp, GPX_FOOT);                           // close the XML record, don't care about errors
            FSfclose(GpxFp);
        }
    }
    
    if(NmeaInterval) {
        FSfclose(NmeaFp);                                           // if we are recording NMEA data just close the file
        if(KmSoFar < 0.1) {                                         // if it is a trivial journey
            FSchdir("\\"); FSchdir("NMEA"); FSchdir(NmeaSubDir);    // get to the correct directory
            FSremove(NmeaFile);                                     // and delete the file
        }
    }
        
    if(PoiLog) FSfclose(LogFp);                                     // just close the file if we are recording a POI log

    if(PoiDiary) {                                                  // for the diary record the diary entry including private/business kilometres
        if(!(KmSoFar < 0.1)) WriteDiary(DiaryFp, day, mth, year, hour, min, day, KmlSeq, StartLat, StartLon, Lat, Lon, secTotal/3600, (secTotal/60)%60, Business?0:KmSoFar, Business?KmSoFar:0);
        FSfclose(DiaryFp);
    }
    
    INTEnableInterrupts();                                          // finished, so re-enable interrupts
    GeneralTimer = 10000;
    while(GeneralTimer) GetCurrentData();                           // wait 10 seconds in case the user used the new track input
    goto WaitOnNewTrack;                                            // we are still running so the new track was used - in that case go back and restart

}  // end of main()



int Write0Param(FSFILE *fp, char *s) {
    if(!FSfwrite(s, strlen(s), 1, fp)) {
        FSfclose(fp); 
        return false;
    }
    return true;
}



int Write3Float(FSFILE *fp, char *s, double f1, double f2, double f3) {
    char buf[256];
    sprintf(buf, s, f1, f2, f3);                                    // format the entry
    if(!FSfwrite(buf, strlen(buf), 1, fp)) {
        FSfclose(fp); 
        return false;
    }
    return true;
}


int WriteKmlStart(FSFILE *fp, int i1, int i2, int d, int m, int y, double f1, double f2) {
    char buf[256];
    sprintf(buf, KML_START, i1, i2, d, m, y, f1, f2);                            // format the entry
    if(!FSfwrite(buf, strlen(buf), 1, fp)) {
        FSfclose(fp);
        return false;
    }
    return true;
}


int WriteKmlMark(FSFILE *fp, int i1, int i2, double f0, double f1, double f2) {
    char buf[256];
    if(UseMiles)
        sprintf(buf, KML_MARK_MI, i1, i2, f0*0.62137, f1, f2);      // format the entry
    else
        sprintf(buf, KML_MARK_KM, i1, i2, f0, f1, f2);              // format the entry
    if(!FSfwrite(buf, strlen(buf), 1, fp)) {
        FSfclose(fp);
        return false;
    }
    return true;
}


int WriteKmlPoi(FSFILE *fp, int i0, int i1, int i2, double f1, double f2) {
    char buf[256];
    sprintf(buf, KML_POI, i0, i1, i2, f1, f2);                            // format the entry
    if(!FSfwrite(buf, strlen(buf), 1, fp)) {
        FSfclose(fp);
        return false;
    }
    return true;
}


int WriteKmlEnd(FSFILE *fp, int i1, int i2, double f0, int i3, int i4, double f1, double f2) {
    char buf[256];
    if(UseMiles)
        sprintf(buf, KML_END_MI, i1, i2, f0*0.62137, i3, i4, f1, f2);// format the entry
    else
        sprintf(buf, KML_END_KM, i1, i2, f0, i3, i4, f1, f2);        // format the entry
    if(!FSfwrite(buf, strlen(buf), 1, fp)) {
        FSfclose(fp);
        return false;
    }
    return true;
}


int WritePoiLog(FSFILE *fp, int i1, int i2, int i3, int i4, int i5, int i6, int i7, int i8, double f1, double f2) {
    char buf[256];
    sprintf(buf, POILOG_DATA, i1, i2, i3, i4, i5, i6, i7, i8, f1, f2);     // format the entry
    if(!FSfwrite(buf, strlen(buf), 1, fp)) {
        FSfclose(fp);
        return false;
    }
    return true;
}


int WriteDiary(FSFILE *fp, int i1, int i2, int i3, int i4, int i5, int i6, int i7, double f1, double f2, double f3, double f4, int i8, int i9, double f5, double f6) {
    char buf[256];
    if(f6 == 0)
        if(UseMiles)
            sprintf(buf, DIARY_DATA_PRIVATE, i1, i2, i3, i4, i5, i6, i7, f1, f2, f3, f4, i8, i9, f5*0.62137); // format a private journey entry with miles
        else
            sprintf(buf, DIARY_DATA_PRIVATE, i1, i2, i3, i4, i5, i6, i7, f1, f2, f3, f4, i8, i9, f5);         // format a private journey entry
    else
        if(UseMiles)
        	sprintf(buf, DIARY_DATA_BUSINESS, i1, i2, i3, i4, i5, i6, i7, f1, f2, f3, f4, i8, i9, f6*0.62137); // format a business journey entry with miles
        else
        	sprintf(buf, DIARY_DATA_BUSINESS, i1, i2, i3, i4, i5, i6, i7, f1, f2, f3, f4, i8, i9, f6);         // format a business journey entry
    if(!FSfwrite(buf, strlen(buf), 1, fp)) {
        FSfclose(fp);
        return false;
    }
    return true;
}


/****************************************************************************************************************************
USB related functions
*****************************************************************************************************************************/


/******************************************************************************************
Check the USB for work to be done.
Used to send and get data to or from the USB interface.
This is called from the Timer 4 interrupt every 200uS.
Each call takes typically 6uS but sometimes it can be up to 600uS.
*******************************************************************************************/
void CheckUSB(void) {
	int i, numBytesRead;

	if(U1OTGSTAT & 0b1000) {                                        // is there 5V on the USB?
	    USBDeviceTasks();                                           // do any USB work

        if(USBGetDeviceState() == CONFIGURED_STATE) {               // if the USB is enumerated and connected
			numBytesRead = getsUSBUSART(USB_RxBuf,USB_RX_BUFFER_SIZE);	// check for data to be read
			for(i = 0; i < numBytesRead; i++) {
                USB_NbrCharsInTxBuf = numBytesRead;
			}

			if(USB_NbrCharsInTxBuf && mUSBUSARTIsTxTrfReady()) {	// next, check for data to be sent
				putUSBUSART(USB_TxBuf[UsbBufSw],USB_NbrCharsInTxBuf); // and send it
				USB_NbrCharsInTxBuf = 0;
                UsbBufSw = !UsbBufSw;
			}

		    CDCTxService();                                         // send anything that needed sending
		}
	}
}



/******************************************************************************************
BOOL USER_USB_CALLBACK_EVENT_HANDLER
This function is called from the USB stack to notify a user application that a USB event
occured.  This callback is in interrupt context when the USB_INTERRUPT option is selected.

Args:  event - the type of event
       *pdata - pointer to the event data
       size - size of the event data

This function was derived from the demo CDC program provided by Microchip
*******************************************************************************************/
BOOL USER_USB_CALLBACK_EVENT_HANDLER(USB_EVENT event, void *pdata, WORD size)
{
    switch(event)
    {
        case EVENT_CONFIGURED:
            CDCInitEP();
             break;
//        case EVENT_SET_DESCRIPTOR:
//            break;
        case EVENT_SOF:
            break;
        case EVENT_SUSPEND:
            break;
        case EVENT_RESUME:
            break;
        case EVENT_BUS_ERROR:
            break;
        case EVENT_TRANSFER:
            Nop();
            break;
        default:
            break;
    }
    return TRUE;
}



// put a character out to the USB interface
void USBPutchar(char c) {
    if((U1OTGSTAT & 0b1000) == 0 || USBDeviceState < CONFIGURED_STATE ||USBSuspendControl == 1) {// check USB status
        USB_NbrCharsInTxBuf = 0;
    	return;                                                     // and skip if the USB is not connected
    }

    if(USB_NbrCharsInTxBuf < USB_TX_BUFFER_SIZE) {                  // skip if the buffer is still full (not being drained)
        mT4IntEnable(0);                                            // don't let CheckUSB be called by timer 4
        USB_TxBuf[UsbBufSw][USB_NbrCharsInTxBuf++] = c;             // Place char into the buffer
        mT4IntEnable(1);
    }
}



void USBPrintString(char *p) {
    while(*p) {
        USBPutchar(*p++);
    }
}



/****************************************************************************************************************
Timer 4 interrupt processor
This fires every 0.2 mSec and is responsible for tracking the various timing variables and servicing the USB
*****************************************************************************************************************/
void __ISR( _TIMER_4_VECTOR, ipl1) T4Interrupt(void) {
    static int SecCnt = 5000;
    static int PoiBtnTimer = 0;

    if(!MDD_MediaDetect()) {
        LED_FAULT_ON;
        MediaChanged = true;
    }

    if(PoiBtnTimer) {
        PoiBtnTimer--;
    }
    
    if(!POI_BUTTON) {
        if(PoiBtnTimer == 0) PoiButton = true;
        PoiBtnTimer = 600;
    }

    if(SecCnt % 5 == 0) {
        // this code executes every 1 mSec

        // this counts down the GPS message timer, when it reaches zero it means that we are not receiving any messages from the GPS
        if(MsgTimeout) {
            if(--MsgTimeout == 0) ValidMsg = ValidData = false;
        }

        // manage the GPS LED.
        // full on if we have a valid fix, blinking if we have valid messages but no fix and off if there are no valid messages at all
        if(ValidDataTimeout) {                                      // if we have data
            LED_GPS_ON;                                             // keep the LED on
            if(--ValidDataTimeout == 0) ValidData = false;
        } else {
            if(ValidMsg) {                                          // otherwise check if we are receiving messages
                if(SecCnt < 2500)                                   // and if so, blink the LED
                    LED_GPS_ON;
                else
                    LED_GPS_OFF;
            } else {
                LED_GPS_OFF;                                        // no data and no messages... turn the LED off
            }
        }

        // blink the power LED when data is saved to the SD card
        if(LedSdTimer) {
            LedSdTimer--;
            if(LedSdTimer == 0)
                LED_PWR_ON;
            else
                LED_PWR_OFF;
        }

        // used for general purpose timing and delays
        if(GeneralTimer) {
            GeneralTimer--;
        }

     }

    if(--SecCnt <= 0) {
        // This code is executed once for every second
        if(--KmlTimer == 0) {
            KmlFlag = true;
            KmlTimer = KmlInterval;
        }
        if(--KmlMarkTimer == 0) {
            KmlMarkFlag = true;
            KmlMarkTimer = KmlMarkInterval;
        }
        if(--GpxTimer == 0) {
            GpxFlag = true;
            GpxTimer = GpxInterval;
        }
        if(--NmeaTimer == 0) {
            NmeaFlag = true;
            NmeaTimer = NmeaInterval;
        }

        secTotal++;

        SecCnt = 5000;
    }

    CheckUSB();

    // Clear the interrupt flag
    mT4ClearIntFlag();
}


// send a byte to the GPS
void GpsPutchar(char c) {
    while(!UARTTransmitterIsReady(UART2));
    UARTSendDataByte(UART2, c);
}


// send a command to the GPS.  Eg, GPSCommand("PMTK251,38400");
// the leading $ and trailing * and checksum are automatically added
void GPSCommand(char *msg) {
    unsigned char chksum;
    GpsPutchar('$');
    for(chksum = 0; *msg; msg++) {
        GpsPutchar(*msg);
        chksum = chksum ^ *msg;
    }
    GpsPutchar('*');
    GpsPutchar(((chksum >> 4) < 10) ? (chksum >> 4) + 48 : (chksum >> 4) + 55);
    GpsPutchar(((chksum & 0x0f) < 10) ? (chksum & 0x0f) + 48 : (chksum & 0x0f) + 55);
    GpsPutchar('\r');
    GpsPutchar('\n');
    
}



/************************************************************************************************
UART 2 interrupt handler - data from the GPS module
************************************************************************************************/
void __ISR(_UART2_VECTOR, ipl2) IntUart2Handler(void) {

    while(UARTReceivedDataIsAvailable(UART2)) {
        GpsBuf[GpsBufHead]  = (char)ReadUART2() & 0x7f;             // store the byte in the ring buffer
        USBPutchar(GpsBuf[GpsBufHead]);
        GpsBufHead = (GpsBufHead + 1) % GPS_BUF_SIZE;               // advance the head of the queue
        if(GpsBufHead == GpsBufTail) {                              // if the buffer has overflowed
            GpsBufTail = (GpsBufTail + 1) % GPS_BUF_SIZE;           // throw away the oldest char
        }
    }
    if(U2STA & 0b1110) U2STACLR = 0b1110;                           // clear any errors on the UART
    INTClearFlag(INT_SOURCE_UART_RX(UART2));                        // Clear the RX interrupt Flag
}


#define BufChar(idx)    GpsBuf[(GpsBufTail + (idx)) % GPS_BUF_SIZE]
#define BufSize(idx)    ((GpsBufHead - (idx)) >= 0) ? (GpsBufHead - (idx)) : (GpsBufHead - (idx) + GPS_BUF_SIZE)

// function to return the index of a field in the GPS message
// message number 1 is after the first comma (ie, the time)
// args:  start = the start of the message (ie, the $ char)
//        nbr = the field number to find
// returns the index number of the first char in the field
int GetFieldPos(int nbr) {
    int start = 0;
    while(nbr)
        if(BufChar(start++) == ',') nbr--;
    return start;
}


// function to return the floating point value of a field in the GPS message
double GetFieldFloat(int nbr) {
    char b[32];
    int i = 0;
    int j = 0;

    while(nbr) if(BufChar(i++) == ',') nbr--;
    while(BufChar(i) != ',') b[j++] = BufChar(i++);
    b[j] = 0;
    return atof(b);
}


/************************************************************************************************
 Get data from the GPS module
 The text string from the gps module is held in a buffer.  This function scans the buffer for
 message start and message end and when found checks the checksum.  If all is OK it then sets
 the ValidMsg flag.  It then goes on to check if the message is an RMC and if so, checks the
 "satellite lock flag" and if OK sets the ValidData flag.  Finally the data is extracted from
 the RMC message.

 Format of the NMEA messages that we are interested in:
    01234567890123456789012345678901234567890123456789012345678901234567890123456789
    $GPGGA,043400.000,2958.7598,S,10332.8693,E,1,05,3.4,25.0,M,-29.3,M,,0000*58
    ====== ======     =========== ============   ==     ====                 ==
    fixed   time       latitude   longitude      sat  altitude            checksum
    header hhmmss     ddmm.mmmm   dddmm.mmmm     cnt   meters

    $GPRMC,043356.000,A,2958.7599,S,10332.8689,E,0.24,54.42,101008,,*20
    ====== ======     = =========== ============ ==== ===== ======   ==
    fixed   time   valid  latitude   longitude   speed course date    checksum
    header hhmmss   data  ddmm.mmmm dddmm.mmmm   knots deg   ddmmyy
    
************************************************************************************************/
void GetCurrentData(void) {
    char tbuf[16], *p;
    int bsize, i, j;
    unsigned char chksum, c1, c2;
    struct tm ltm, *rtm;
    time_t ttm;

    if(U2STA & 0b1110) U2STACLR = 0b1110;                           // clear any errors on the UART

    // get the amount of data in the buffer
    bsize = ((GpsBufHead - GpsBufTail) >= 0) ? (GpsBufHead - GpsBufTail) : (GpsBufHead - GpsBufTail + GPS_BUF_SIZE);
    if(bsize == 0) return;                                          // nothing in the GPS buffer so exit
    
    // search for a $ symbol marking the start of a message
    for(i = 0; BufChar(i) != '$'; i++)
        if(i >= bsize) {
            GpsBufTail = GpsBufHead;                                // show that we have checked the buffer
            return;                                                 // and exit
        }
    GpsBufTail = (GpsBufTail + i) % GPS_BUF_SIZE;
    bsize -= i;                                                     // adjust the buffer size because we changed GpsBufTail
    if(bsize < 8) return;

    // search for a * symbol marking the end of a message while at the same time accumulating the checksum
    for(chksum = 0, i = 1; BufChar(i) != '*'; i++) {
        if(i >= bsize) {
            if(i > MAX_GPS_MSG) {                                   // if the data is larger than the NMEA max something is wrong
                GpsBufTail = GpsBufHead;                            // empty the buffer
            }
        return;                                                     // exit if not found
        }
        chksum = chksum ^ BufChar(i);
    }

    // we have the * symbol, now compare the checksums
    if(bsize - i < 5) return;                                       // exit if we do not have the two checksum digits and the CR/LF
    c1 = BufChar(i + 1) - '0';
    if(c1 > 9) c1 -= 'A' - 58;
    c2 = BufChar(i + 2) - '0';
    if(c2 > 9) c2 -= 'A' - 58;
    if(chksum != ((c1 << 4) | c2)) {                                // compare the checksum
        GpsBufTail = (GpsBufTail + i + 4) % GPS_BUF_SIZE;           // if error; delete the message
        return;                                                     // and abort processing
    }

    // at this point we have a valid message
    ValidMsg = true;
    if(MsgTimeout < 1500) MsgTimeout = 1500;

    // process an GGA message
   if(BufChar(3) == 'G' && BufChar(4) == 'G' && BufChar(5) == 'A') {
        for(j = 0; j < i + 5; j++) GgaRecord[j] = BufChar(j);       // save the record in case we are recording NMEA data
        GgaRecord[j] =0;
        Alt = GetFieldFloat(9);
    }
    
    // process an RMC message
   if(BufChar(3) == 'R' && BufChar(4) == 'M' && BufChar(5) == 'C') {
        if(BufChar(18) == 'A') {                                    // check if the GPS has a lock on the satellites
            ValidData = true;
            if(ValidDataTimeout < 1500) ValidDataTimeout = 1500;
            
            for(j = 0; j < i + 5; j++) RmcRecord[j] = BufChar(j);   // save the record in case we are recording NMEA data
            RmcRecord[j] =0;
            
            // get the time
            ltm.tm_hour = (BufChar(7) - '0') * 10 + (BufChar(8) - '0');
            ltm.tm_min = (BufChar(9) - '0') * 10 + (BufChar(10) - '0');
            ltm.tm_sec = (BufChar(11) - '0') * 10 + (BufChar(12) - '0');

            // get the date
            i = GetFieldPos(9);                                     // find the start of the date
            ltm.tm_mday = (BufChar(i) - '0') * 10 + (BufChar(i + 1) - '0');
            ltm.tm_mon = (BufChar(i + 2) - '0') * 10 + (BufChar(i + 3) - '0') - 1;
            ltm.tm_year = (BufChar(i + 4) - '0') * 10 + (BufChar(i + 5) - '0') + 100;

            // while we have the UTC time save it in the GPX format
            sprintf(GpxTime, "%04d-%02d-%02dT%02d:%02d:%02dZ", ltm.tm_year + 1900, ltm.tm_mon + 1, ltm.tm_mday, ltm.tm_hour, ltm.tm_min, ltm.tm_sec);
            
            // adjust for the time zone
            ttm = mktime(&ltm);                                     // convert the GMT time to a time value (seconds since 1970)
            ttm += TimeZone * 3600;                                 // add (or subtract) the time zone offset in seconds
            rtm = gmtime(&ttm);                                     // convert back to a time structure
            hour = rtm->tm_hour;                                    // and extract the local time and date from the structure
            min = rtm->tm_min; sec = rtm->tm_sec;
            day = rtm->tm_mday; mth = rtm->tm_mon + 1; year = rtm->tm_year - 100;

            // get the latitude
            i = GetFieldPos(3);                                     // point to the start of the latitude
            Lat = (BufChar(i) - '0') * 10 + (BufChar(i + 1) - '0'); // get the degrees
            for(p = tbuf, i += 2; BufChar(i) != ','; )
                *p++ = BufChar(i++);                                // copy the minutes into a string
            *p = 0;                                                 // and terminate the string
            Lat += atof(tbuf)/60;                                   // and add to the latitude
            if(BufChar(GetFieldPos(4)) == 'S') Lat = -Lat;          // southern latitudes are negative

            // get the longitude
            i = GetFieldPos(5);                                     // point to the start of the longitude
            Lon = (BufChar(i) - '0') * 100 + (BufChar(i + 1) - '0') * 10 + (BufChar(i + 2) - '0'); // get the degrees
            for(p = tbuf, i += 3; BufChar(i) != ','; )
                *p++ = BufChar(i++);                                // copy the minutes into a string
            *p = 0;                                                 // and terminate the string
            Lon += atof(tbuf)/60;                                   // and add to the longitude
            if(BufChar(GetFieldPos(6)) == 'W') Lon = -Lon;          // western longitudes are negative

            // get the speed and course for the GPX format
            Speed = GetFieldFloat(7);
            Course = GetFieldFloat(8);

            // accumulate the distance travelled for the KML format
            // this assumes one RMC message per second and does not try to compensate for missing messages
            KmSoFar += (Speed * 1.852)/3600;
        }
    }

    GpsBufTail = (GpsBufTail + i + 2) % GPS_BUF_SIZE;

}



void dump(char *p, int nbr) {
	char buf1[60], buf2[30], *b1, *b2;
        char inpbuf[256];

	b1 = buf1; b2 = buf2;
	b1 += sprintf(b1, "%8x: ", (unsigned int)p);
	while(nbr > 0) {
		b1 += sprintf(b1, "%02x ", *p);
		b2 += sprintf(b2, "%c", (*p >= ' ' && *p < 0x7f) ? *p : ' ');
		p++;
		nbr--;
		if((unsigned int)p % 16 == 0) {
			sprintf(inpbuf, "%s   %s", buf1, buf2);
			USBPrintString(inpbuf);
			b1 = buf1; b2 = buf2;
			b1 += sprintf(b1, "\r\n%8x: ", (unsigned int)p);
		}
	}
	if(b2 != buf2) {
		sprintf(inpbuf, "%s   %s", buf1, buf2);
		USBPrintString(inpbuf);
	}
	USBPrintString("\r\n");
}




