/*
 * File:   kitchen.c
 * Author: mhibbett. mike.hibbett@gmail.com, mjhdesigns.com
 *
 * Created on 4th May 2014
 */

#include <stdio.h>
#include <pic18.h>
#include "timers.h"
#include "lcd.h"
#include "interrupts.h"

static const unsigned char timeStr[] =     "Time %2.2ldh %2.2ldm %2.2lds";
static const unsigned char txtStartUp[] =  "Kitchen Timer   ";
static const unsigned char txtSetTime[] =  "Set the time    ";
static const unsigned char txtRunning[] =  "Timer Running...";
static const unsigned char txtAlarming[] = "Timer EXPIRED   ";

unsigned char timerState;
unsigned long timerVal = 0;
unsigned char slot = 0;


// This hold copies of the data to be shown on the LCD
unsigned char line1[17];


void kitchenInit(void)
{
    setPos(LCD_LINE1,0);
    sprintf(line1,txtStartUp);
    LCDWriteStr(line1);

    // A short delay to display the startup banner
    usleep(USLEEP_80MS);
    usleep(USLEEP_80MS);
    usleep(USLEEP_80MS);
    usleep(USLEEP_80MS);
    usleep(USLEEP_80MS);

    // Start off with the timer running in low power mode
    timerState = 0;
    LCDOff();
}


/*************************************************************************
 * Function: KeyTimeWaitRelease
 *           Waits until the Do key is released ( debouncing the pin to be
 *           certain )
 *************************************************************************
 */
void KeyTimeWaitRelease(void)
{
    do {
        // Wait for key to be released
        while (!PORTBbits.RB4)
            ; // Do nothing

        usleep(USLEEP_DEBOUNCE);

        // If it isn't still high, loop back and try again.
    } while (!PORTBbits.RB4);   
}


/*************************************************************************
 * Function: KeyTimeGetState
 *           Returns the state of the Do key ( debouncing first if it
 *           is believed to be pressed )
 *           Therefore this call may take 1us ( no key ) or 20ms ( key )
 *           This is active low : 0 means pressed
 *************************************************************************
 */
unsigned char KeyTimeGetState(void)
{
    while (1) {
        if (PORTBbits.RB4) {
            usleep(USLEEP_DEBOUNCE);

            if (PORTBbits.RB4)
                return 1;
        } else {
            usleep(USLEEP_DEBOUNCE);

            if (!PORTBbits.RB4)
                return 0;
        }
    }
    return 0;
}


/*************************************************************************
 * Function: KeyDoWaitRelease
 *           Waits until the Time key is released ( debouncing the pin to be
 *           certain )
 *************************************************************************
 */
void KeyDoWaitRelease(void)
{
    do {
        // Wait for key to be released
        while (!PORTBbits.RB3)
            ; // Do nothing

        usleep(USLEEP_DEBOUNCE);

        // If it isn't still high, loop back and try again.
    } while (!PORTBbits.RB3);
}


/*************************************************************************
 * Function: KeyDoGetState
 *           Returns the state of the Do key ( debouncing first if it
 *           is believed to be pressed )
 *           Therefore this call may take 1us ( no key ) or 20ms ( key )
 *           This is active low : 0 means pressed
 *************************************************************************
 */
unsigned char KeyDoGetState(void)
{
    while (1) {
        if (PORTBbits.RB3) {
            usleep(USLEEP_DEBOUNCE);

            if (PORTBbits.RB3)
                return 1;
        } else {
            usleep(USLEEP_DEBOUNCE);

            if (!PORTBbits.RB3)
                return 0;
        }
    }
    return 0;
}


unsigned char timerAsleep(void)
{
    return timerState;
}


unsigned char decTimer(void)
{
    return (--timerVal);
}

void kitchenOn(void)
{
    timerVal = 0;
    
    LCDOn();
    
    setPos(LCD_LINE1,0);
    sprintf(line1,timeStr, timerVal/3600, (timerVal/60)%60, timerVal%60);
    LCDWriteStr(line1);
    setPos(LCD_LINE2,0);
    LCDWriteStr((unsigned char *)txtSetTime);
}


void showCursor(void)
{
    switch (slot) {
        case 0:
            setPos(LCD_LINE1,15);
        break;
        case 1:
            setPos(LCD_LINE1,11);
        break;
        case 2:
            setPos(LCD_LINE1,7);
        break;
    }

    LCDCursorOn();
}

/*************************************************************************
 * Function: kitchenHandler
 *           Called every 2s ( or continuously when the timer is active )
 *           manages the keypesses and the general "state" of the timer
 *************************************************************************
 */
void kitchenHandler(void)
{
    switch (timerState) {
        case 0:
            // Timer asleep.
            // If either key is pressed, wake up.
            if (!KeyTimeGetState() || !KeyDoGetState()) {
                timerState = 1;
                slot = 0;
                kitchenOn();
                showCursor();
                KeyTimeWaitRelease();
                KeyDoWaitRelease();
            }
        break;

        case 1:
            // Timer set
            if (!KeyTimeGetState()){
                KeyTimeWaitRelease();
                switch (slot) {
                    case 0:
                        timerVal += 1;
                    break;
                    case 1:
                        timerVal += 60;
                    break;
                    case 2:
                        timerVal += 3600;
                    break;
                }

                setPos(LCD_LINE1,0);
                sprintf(line1,timeStr, timerVal/3600, (timerVal/60)%60, timerVal%60);
                LCDWriteStr(line1);
                showCursor();
                KeyTimeWaitRelease();
            } else {
                if (!KeyDoGetState()){
                    KeyDoWaitRelease();

                    // if slot is currently 2, then start timer
                    if (slot == 2)  {
                        timerState = 2;
                        setPos(LCD_LINE2,0);
                        LCDWriteStr((unsigned char *)txtRunning);
                        LCDCursorOff();
                        KeyDoWaitRelease();
                    } else {
                        slot++;
                        showCursor();
                    }
                }
            }
        break;

        case 2:
            // Timer running
            // If either key is pressed, cancel alarm.
            if (!KeyTimeGetState() || !KeyDoGetState()) {
                KeyDoWaitRelease();
                KeyTimeWaitRelease();
                LCDOff();
                timerState = 0;
            } else {
                // Although no button has been pressed, the timer is running
                // and the display may need updating.
                switch (IntOccurred) {
                    case 1:
                        // Update the time display
                        setPos(LCD_LINE1,0);
                        sprintf(line1,timeStr, timerVal/3600, (timerVal/60)%60, timerVal%60);
                        LCDWriteStr(line1);
                        IntOccurred = 0;
                    break;

                    case 2:
                        // The timer has expired - alarm!
                        timerState = 3;
                        setPos(LCD_LINE1,0);
                        sprintf(line1,timeStr, 0, 0, 0);
                        setPos(LCD_LINE2,0);
                        LCDWriteStr((unsigned char *)txtAlarming);
                        TRISCbits.TRISC3 = 0; // Turn on the PWM
                        IntOccurred = 0;
                    break;
                }
            }
        break;

        case 3:
            // Timer alarming
            // If a key is pressed, turn everything off.
            if (!KeyTimeGetState() || !KeyDoGetState()) {
                TRISCbits.TRISC3 = 1; // Turn the PWM Off
                KeyDoWaitRelease();
                KeyTimeWaitRelease();
                LCDOff();
                timerState = 0;
            }
        break;
    }
}
