

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>
#include <sys/time.h>
                            

#define GPIO_ADD    0x20200000L // physical address of I/O peripherals on the ARM processor
#define GPIO_SEL0   0           // Offset of SEL register for GP4 into GPIO bank   
#define GPIO_SEL1   1           // Offset of SEL register for GP17 & GP18 into GPIO bank   
#define GPIO_SEL2   2           // Offset of SEL register for GP21..25 into GPIO bank  
#define GPIO_SET    7           // Offset of PIN HIGH register into GPIO bank  
#define GPIO_CLR    10          // Offset of PIN LOW register into GPIO bank  
#define GPIO_INP    13          // Offset of PIN INPUT value register into GPIO bank  

#define PAGE_SIZE   4096        
#define BLOCK_SIZE  PAGE_SIZE


int  mem_fd     = 0;
char *gpio_mmap = NULL;
char *gpio_ram  = NULL;
volatile unsigned int *gpio = NULL;


/* These 'defines' map the peripheral pin functions to our circuits LCD pins */
/* See LCD datasheet available from Farnell under part number 1847937 */
/* and Broadcom BCM2835-ARM-Peripherals.pdf 6/2/2012 */
#define PINS_OUTPUT0   *(gpio+GPIO_SEL0) &= 0xFFFF8FFFL; *(gpio+GPIO_SEL0) |= 0x00001000L 
#define PINS_OUTPUT1   *(gpio+GPIO_SEL1) &= 0xF81FFFFFL; *(gpio+GPIO_SEL1) |= 0x01200000L 
#define PINS_OUTPUT2   *(gpio+GPIO_SEL2) &= 0xFFFC0E07L; *(gpio+GPIO_SEL2) |= 0x00009048L 


#define E_HIGH    *(gpio+GPIO_SET) = 0x02000000L
#define RS_HIGH   *(gpio+GPIO_SET) = 0x00000010L
#define RW_HIGH   *(gpio+GPIO_SET) = 0x01000000L
#define D4_HIGH   *(gpio+GPIO_SET) = 0x00020000L
#define D5_HIGH   *(gpio+GPIO_SET) = 0x00040000L
#define D6_HIGH   *(gpio+GPIO_SET) = 0x00200000L
#define D7_HIGH   *(gpio+GPIO_SET) = 0x00400000L

#define E_LOW     *(gpio+GPIO_CLR) = 0x02000000L
#define RS_LOW    *(gpio+GPIO_CLR) = 0x00000010L
#define RW_LOW    *(gpio+GPIO_CLR) = 0x01000000L
#define D4_LOW    *(gpio+GPIO_CLR) = 0x00020000L
#define D5_LOW    *(gpio+GPIO_CLR) = 0x00040000L
#define D6_LOW    *(gpio+GPIO_CLR) = 0x00200000L
#define D7_LOW    *(gpio+GPIO_CLR) = 0x00400000L


/* This routine write 4 bits to the display. RS and RW are included */
/* in bits 5 and 4 respectively */
void write_nibble( unsigned char val )
{

   if ( val & 0x20 ) RS_HIGH;
   if ( val & 0x10 ) RW_HIGH;
   if ( val & 0x08 ) D7_HIGH;
   if ( val & 0x04 ) D6_HIGH;
   if ( val & 0x02 ) D5_HIGH;
   if ( val & 0x01 ) D4_HIGH;

   usleep(2);  // Let pins settle
   E_HIGH;
   usleep(2);  // E signal hold time
   E_LOW;

   /* Set all the pins to default (low) */
   E_LOW;
   RS_LOW;
   RW_LOW;
   D4_LOW;
   D5_LOW;
   D6_LOW;
   D7_LOW;
}


void setup_io()
{

   /* open /dev/mem to get acess to physical ram */
   if ((mem_fd = open("/dev/mem", O_RDWR|O_SYNC) ) < 0) {
      printf("can't open /dev/mem. Did you run the program with administrator rights?\n");
      exit (-1);
   }

   /* Allocate a block of virtual RAM in our application's address space */
   if ((gpio_ram = malloc(BLOCK_SIZE + (PAGE_SIZE-1))) == NULL) {
      printf("allocation error \n");
      exit (-1);
   }

   /* Make sure the pointer is on 4K boundary */
   if ((unsigned long)gpio_ram % PAGE_SIZE)
     gpio_ram += PAGE_SIZE - ((unsigned long)gpio_ram % PAGE_SIZE);

   /* Now map the physical addresses of the peripheral control registers 
      into our address space */
   gpio_mmap = (unsigned char *)mmap(
      (caddr_t)gpio_ram,
      BLOCK_SIZE,
      PROT_READ|PROT_WRITE,
      MAP_SHARED|MAP_FIXED,
      mem_fd,
      GPIO_ADD
   );

   if ((long)gpio_mmap < 0) {
      printf("unable to map the memory. Did you run the program with administrator rights?\n");
      exit (-1);
   }

   /* Always use a volatile pointer to hardware registers */
   gpio = (volatile unsigned *)gpio_mmap;

   /* Now we have access to the hardware reigsters we can start twiddling I/O pins */

   /* Switch GPIO 4,17,18,21,22,24,25 to output mode */
   PINS_OUTPUT0;
   PINS_OUTPUT1;
   PINS_OUTPUT2;

   /* Set all the pins to default (low) */
   E_LOW;
   RS_LOW;
   RW_LOW;
   D4_LOW;
   D5_LOW;
   D6_LOW;
   D7_LOW;

   /* Short delay to allow the LCD to startup. Twice recommended for certainty! */
   usleep(80000);
}


void init_lcd( void )
{
   /* Initialise the display */
   write_nibble(0x03);
   usleep(8000);
   write_nibble(0x03);
   usleep(200);
   write_nibble(0x03);
   usleep(200);
   write_nibble(0x02);
   usleep(200);
   write_nibble(0x02);
   write_nibble(0x0C); // Number of lines and font.
   usleep(3000);
   write_nibble(0x00);
   write_nibble(0x0C); // display on
   usleep(3000);
   write_nibble(0x00); 
   write_nibble(0x01); // Clear the display
   usleep(3000);
   write_nibble(0x00); 
   write_nibble(0x06); // Set entry mode- no cursor
   usleep(3000);
}


/* Set's the write position. line is 0 or 1. col is 0..15 */
void  setpos( unsigned char line, unsigned char col )
{
   unsigned char add = col + (line > 0 ? 0x40 : 0x00);
   unsigned char val;

   val = (add & 0xF0) >> 4;
   write_nibble(0x08 + val); 

   val = add & 0x0F;
   write_nibble(val); // Set Data ram address to first line

   usleep(100);   
}



/* writes up to 16 characters to the display from the current position */
void writelcd( char *str )
{
   unsigned char lp;
   char ch;
   unsigned char nibble;

   for (lp=0; (lp<16) && *str; lp++) {
      ch = *str++;
      nibble = 0x20 + ((ch & 0xF0) >> 4);
      write_nibble(nibble); 
      nibble = 0x20 + (ch & 0x0F);
      write_nibble(nibble); 
      usleep(100);
   }
}

int main(int argc, char **argv)
{ 
   int valid_args = 0;  /* Assume valid use of the program, until we detect invalid option */
   char *textp = NULL;  /* pointer to the text to display */
   int res = 0;         /* return value from getopt */
   int c = 0;           /* column number - 0 to 15 */
   int l = 0;           /* line number - 0 to 1 */

   /* Set up gpi pointer for direct register access */
   setup_io();


   while ((res = getopt (argc, argv, "il:c:t:")) != -1) {
      switch (res) {
         case 'i':
            /* Initialise the display */
            valid_args = 1;
            init_lcd();
         break;
         case 't':
            valid_args = 1;
            /* get a pointer to the text option's string */
            textp = optarg;
         break;
         case 'l':
            l = atoi(optarg);
            if ( (l < 0 ) || (l > 1) ) {
               valid_args = 0;
               printf("Invalid line number. Should be between 0 and 1\r\n");
            }
         break;
         case 'c':
            c = atoi(optarg);
            if ( (c < 0 ) || (c > 15) ) {
               valid_args = 0;
               printf("Invalid column number. Should be between 0 and 15\r\n");
            }
         break;
         case '?':
            if (optopt == 'c')
               printf( "Option -%c requires an argument.\n", optopt);
            else printf( "Unknown option `-%c'.\n", optopt);
            
            exit (-1);
         break;
      }
   }  
   
   /* Check that the program was called correctly */
   if ( !valid_args ) {
      printf("Invalid argument specified.\nRun as:\nlcd-pi -t \"text\"\nor\nlcd-pi -l y -c x -t \"text\"\nor\nlcd-pi -i\n");
      exit (-1);
   } 

   setpos(l,c);
   writelcd(textp);

   return 0;
}
