// LCDText is a 32 byte shadow of the LCD text.  Write to it and
// then call LCDUpdate() to copy the string into the LCD module.
BYTE LCDText[2][16];

void lcd_init( void );
void Delay10us( WORD );
void DelayMs( WORD );
void DelaySecs( WORD secs );
void LCDWrite(BYTE, BYTE);
void LCDErase(void);
void LCDWriteLine(WORD, char *);
void LCDUpdate(void);


int main (void)
{
    // cpu_init();

    /* Initialise the LCD output pins */
    lcd_init();

    LCDWriteLine(1, "Test" );
    
    do {} while(1);
}



void cpu_init( void )
{
    // Configure Oscillator to operate the device at (very nearly) 40MIPS
    // Fosc= 78749990Hz with 3.5 something MHz xtal
    PLLFBD=86;               // M=88
    CLKDIVbits.PLLPOST=0;    // N1=2
    CLKDIVbits.PLLPRE=0;     // N2=2
    OSCTUN=0;                // Tune FRC oscillator, if FRC is used

    // Disable Watch Dog Timer
    RCONbits.SWDTEN=0;

    __builtin_write_OSCCONH(0x03);
    __builtin_write_OSCCONL(0x01);
    // Wait for Clock switch to occur
    while (OSCCONbits.COSC != 0b011);
    // Wait for PLL to lock
    while(OSCCONbits.LOCK!=1) {};
}


#define LCD_RS_TRIS TRISBbits.TRISB5
#define LCD_E_TRIS  TRISBbits.TRISB7
#define LCD_RW_TRIS TRISBbits.TRISB6
#define LCD_RS_LAT  LATBbits.LATB5
#define LCD_E_LAT   LATBbits.LATB7
#define LCD_RW_LAT  LATBbits.LATB6
#define LCD_0_TRIS  TRISBbits.TRISB8
#define LCD_1_TRIS  TRISBbits.TRISB9
#define LCD_2_TRIS  TRISBbits.TRISB10
#define LCD_3_TRIS  TRISBbits.TRISB11
#define LCD_0_LAT   LATBbits.LATB8
#define LCD_1_LAT   LATBbits.LATB9
#define LCD_2_LAT   LATBbits.LATB10
#define LCD_3_LAT   LATBbits.LATB11


void lcd_init( void )
{
    BYTE i;
    
    /* set the driving pins to output low - this will disable the display */
    LCD_RS_TRIS = 0;
    LCD_RS_LAT = 0;
    LCD_E_TRIS = 0;
    LCD_E_LAT = 0;
    LCD_RW_TRIS = 0;
    LCD_RW_LAT = 0;
    LCD_0_TRIS = 0;
    LCD_0_LAT = 0;
    LCD_1_TRIS = 0;
    LCD_1_LAT = 0;
    LCD_2_TRIS = 0;
    LCD_2_LAT = 0;
    LCD_3_TRIS = 0;
    LCD_3_LAT = 0;

    // Wait the required time for the LCD to reset
    DelayMs(40);

    // Set the default function
    // Go to 8-bit mode first to reset the instruction state machine
    // This is done in a loop 3 times to absolutely ensure that we get
    // to 8-bit mode in case if the device was previously booted into
    // 4-bit mode and our PIC got reset in the middle of the LCD
    // receiving half (4-bits) of an 8-bit instruction
    LCD_RS_TRIS = 0;
    LCD_0_TRIS = 1;
    LCD_1_TRIS = 1;
    LCD_2_TRIS = 0;
    LCD_3_TRIS = 0;
    
    Delay10us(50);

    for(i = 0; i < 3; i++)
    {
            LCD_E_TRIS = 1;
            Delay10us(1);			// Wait E Pulse width time (min 230ns)
            LCD_E_TRIS = 0;
            DelayMs(2);
    }

    // Enter 4-bit mode with two lines (requires 8-bits on most LCD controllers)
    LCDWrite(0, 0x38);
    Delay10us(50);

    // Set the entry mode
    LCDWrite(0, 0x28);	// Increment after each write, do not shift
    Delay10us(50);

    // Set the display control
    LCDWrite(0, 0x28);	// Turn display on, no cusor, no cursor blink
    Delay10us(50);

    // Clear the display
    LCDWrite(0, 0x0f);
    DelayMs(2);
    // Clear the display
    LCDWrite(0, 0x01);
    DelayMs(2);
    // Clear the display
    LCDWrite(0, 0x06);
    DelayMs(2);
    /* Set the LCD cursor to home  */
    LCDWrite(0, 0x02);
    DelayMs(2);
    /* Hide the cursor */
    LCDWrite(0, 0x0C);
    DelayMs(2);

}


void Delay10us( WORD us10 )
{
     WORD lp;

     do {
          for ( lp=0; lp < 30; lp++ )  // 37 == number of instruction cycles to burn in 10us
               Nop();
     } while ( us10-- );
}


void DelayMs( WORD ms )
{
     WORD lp;

     do {
          for ( lp=0; lp < 22000; lp++ )  // 3685 == number of instruction cycles to burn in 1ms
               Nop();
     } while ( ms-- );
}


void DelaySecs( WORD secs )
{
     do {
          DelayMs(300);
     } while ( secs-- );
}



/******************************************************************************
 * Function:        static void LCDWrite(BYTE RS, BYTE Data)
 *
 * PreCondition:    None
 *
 * Input:           RS - Register Select - 1:RAM, 0:Config registers
 *					Data - 8 bits of data to write
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Controls the Port I/O pins to cause an LCD write
 *
 * Note:            None
 *****************************************************************************/
void LCDWrite(BYTE RS, BYTE Data)
{
    unsigned char lp=0;     /* Simple loop counter */

    LCD_RW_TRIS = 0;
    LCD_RS_TRIS = RS;

    LCD_0_TRIS = ((Data & 0x10) == 0x10);
    LCD_1_TRIS = ((Data & 0x20) == 0x20);
    LCD_2_TRIS = ((Data & 0x40) == 0x40);
    LCD_3_TRIS = ((Data & 0x80) == 0x80);

    // Wait Data setup time (min 40ns)
    for (lp=0; lp<5; lp++)
        Nop();

    LCD_E_TRIS = 1;

    // Wait E Pulse width time (min 230ns)
    for (lp=0; lp<20; lp++)
        Nop();

    LCD_E_TRIS = 0;

    LCD_0_TRIS = ((Data & 0x01) == 0x01);
    LCD_1_TRIS = ((Data & 0x02) == 0x02);
    LCD_2_TRIS = ((Data & 0x04) == 0x04);
    LCD_3_TRIS = ((Data & 0x08) == 0x08);

    // Wait Data setup time (min 40ns)
    for (lp=0; lp<10; lp++)
        Nop();
    
    LCD_E_TRIS = 1;

    // Wait E Pulse width time (min 230ns)
    for (lp=0; lp<30; lp++)
        Nop();

    LCD_E_TRIS = 0;
}


/******************************************************************************
 * Function:        void LCDUpdate(void)
 *
 * PreCondition:    LCDInit() must have been called once
 *
 * Input:           LCDText[]
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Copies the contents of the local LCDText[] array into the
 *					LCD's internal display buffer.  Null terminators in
 *					LCDText[] terminate the current line, so strings may be
 *					printed directly to LCDText[].
 *
 * Note:            None
 *****************************************************************************/
void LCDUpdate(void)
{
    BYTE i;

    // Go home
    LCDWrite(0, 0x02);
    DelayMs(2);

    for ( i=0; i<16; i++) {
        LCDWrite(1, LCDText[0][i]);
        Delay10us(50);
    }

    LCDWrite(0, 0xC0);
    Delay10us(50);

    for ( i=0; i<16; i++) {
        LCDWrite(1, LCDText[1][i]);
        Delay10us(50);
    }
}


/******************************************************************************
 * Function:        void LCDErase(void)
 *
 * PreCondition:    LCDInit() must have been called once
 *
 * Input:           None
 *
 * Output:          None
 *
 * Side Effects:    None
 *
 * Overview:        Clears LCDText[] and the LCD's internal display buffer
 *
 * Note:            None
 *****************************************************************************/
void LCDErase(void)
{
	// Clear display
	LCDWrite(0, 0x01);
	DelayMs(2);

	// Clear local copy
	memset(LCDText, ' ', 32);
}


void LCDWriteLine(WORD number, char *line)
{

    BYTE    i;

    memset(LCDText[number-1],' ',16);

    for (i=0; (i<16) && *line; i++) {
        LCDText[number-1][i] = *(line++);
    }
    
    LCDUpdate();
}

