	NOLIST 
;System macros - Table management. 

;****** ******* ******* ******* ******* ******* ******* *** * 
;COPYRIGHT (C) 2005 Keith Anderson 
;------ ------- ------- ------- ------- ------- ------- --- - 

;****** ******* ******* ******* ******* ******* ******* *** * 
;SYSTEM TABLES 
;------ ------- ------- ------- ------- ------- ------- --- - 

;This strategy to manage lookup tables is similar to 
;earlier strategies, but addresses two problems. 
;1/ 
;The linker seems unable to process global symbols 
;other than address labels, and so the instructions 
;to constrain w to the size of the table 
;must be in the same source file as the table. 
;2/ 
;The Microchip documentation of 
;	pagesel 
;is ambiguous. 
;The .hex files reveal that it generates 
;only two instructions similar to: 
;	bcf PCLATH,3 
;	bcf PCLATH,4 
;never modifies w, and always leaves PCLATH<2:0> unchanged. 
;This means that it is useful for call and goto, 
;but not for addwf PCL,f. 
;There seems to be no assembler directive 
;that will manage PCLATH for addwf PCL,f. 

;This strategy is then in three parts: 
;The table definition, including the definition 
;of the size of the table. 
;The procedure to constrain w to the size of the table. 
;The procedure(s) that use the table. 

;The discipline: 
;1/ 
;PCLATH is only ever updated to page boundaries, 
;not to block boundaries. 
;A page is the 2^11 = 2048 byte segment seen by 
;	goto and call, 
;A block is the 2^8 = 256 byte segment seen by 
;	addwf PCL,f. 
;2/ 
;Code using addwf PCL is always in the zero'th block 
;of the page containing the code using it. 
;3/ 
;Other code on each page starts at the first block 
;of the page. 
;The relevant addresses are: 
;	org 0x0010 ;Page 0, Block 0 
;	org 0x0100 ;Page 0, Block 1 
;	org 0x0800 ;Page 1, Block 0 
;	org 0x0900 ;Page 1, Block 1 
;	org 0x1000 ;Page 2, Block 0 
;	org 0x1100 ;Page 2, Block 1 
;	org 0x1800 ;Page 3, Block 0 
;	org 0x1900 ;Page 3, Block 1 
;The setting for Page 0, Block 0 assumes that most of 
;the reset and interrupt code is located away from 
;the relatively precious space on Block 0. 
;However, the necessary interrupt preamble needs 16 bytes, 
;so code using addwf, PCL can't start before 0x0010. 
;4/ 
;The only overhead allowed on the relatively precious 
;block 0 is the single and essential 
;	addwf PCL,f 
;instruction that precedes the table. 
;5/ 
;Range checking is always by the macro: TableRangeCheck. 
;This generates eight instructions. 
;Out of range input is mapped consistently to 
;the last byte in the table, 
;and the tables can be of any size up to 255 bytes. 
;The benefit of TableRangeCheck is so high, 
;and the cost is so low that 
;inferior techniques such as "hope for the best" or 
;range limiting using, 
;	andlw Mask 
;can not be justified. 
;6/ 
;Each table is in its own CODE section, 
;and this must be defined manually. 
;Each table is defined by two names, of the form: 
;<table>Start 
;	<table goes here> 
;<table>End 
;Although the constrain code must be in a different 
;CODE section from the table, the benefit of placing 
;the constrain code in a different file from 
;its table is tiny and unnecessary. 
;The constrain code should always be in the same file 
;as its table and the symbols: <table>Start and <table>End 
;should never be defined globally. 
;7/ 
;Within the same source file, 
;but in a different CODE section, 
;TableRangeCheck is used in a small procedure 
;to constrain w to the size of the table. 
;The structure is: 
;<table>ConstrainCode CODE 
;<table> 
;	TableRangeCheck <table>Start,<table>End 
;Usually, the table, the constrain code, and the procedure(s) 
;using the table should share a common file, 
;and the entry to the constrain procedure, <table> 
;should not be defined globally. 
;The rather complicated exceptions are described below. 
;See clause (11). 
;8/ 
;In the procedure(s) using the table, for table lookup, use: 
;	movf src,w 
;	call <table> 
;9/ 
;In the procedure(s) using the table, for computed goto, use: 
;	movf src,w 
;	goto <table> 
;10/ 
;If the components share a common file, 
;the linker script should put the code containing the table 
;into a zero'th block and the other code into a non-zero'th 
;block within the same page. 
;This convention must be maintained manually. 

;Exceptions to (7) above. 

;11/ 
;Rarely, it can be useful to place the procedure(s) 
;using the table in files separate from the table 
;and its constrain code. 
;In this case, the entry to the constrain code 
;can be defined globally, 
;but there are several precautions to consider: 
;11a/ 
;If the components share a common file, 
;the linker script should place them on a common page, 
;so that 
;	pagesel <table> 
;statements are not needed. 
;11b/ 
;For lookup tables, 
;in all files where <table> is defined as EXTERN, 
;	pagesel <table> 
;should be used universally for all calls to <table>. 
;In this case, the linker script can place the procedure(s) 
;using the table on any page. 
;11c/ 
;For computed goto, 
;even in files where <table> is defined as EXTERN, 
;	pagesel <table> 
;should NEVER be used. 
;It manages PCLATH for the transfer to the table, 
;but not for the goto from the table. 
;In this case, the linker script must place the table, 
;the constrain code, and the procedure(s) 
;using the table on the same page. 

;****** ******* ******* ******* ******* ******* ******* *** * 
;EXAMPLE 
;------ ------- ------- ------- ------- ------- ------- --- - 

;This example assumes that the table, 
;the constrain code, and the procedure(s) 
;using the table are all in the same file 
;and all on the same page. 

;ExampleTableCode CODE 
;ExampleTableStart 
;	addwf PCL,f 
;	retlw 0xAA 
;	retlw 0xBB 
;	... 
;ExampleTableEnd 
;... other tables 

;ExampleTableConstrainCode CODE 
;ExampleTable 
;	TableRangeCheck ExampleTableStart,ExampleTableEnd 
;... constrain code for other tables 

;It is usual, but not essential for each table 
;and each constrain procedure to have its own CODE section. 

;... 

;ExampleCode CODE 

;... 
;	movf src,w 
;	call ExampleTable 
;	movwf dst 
;... 

;Note that the transfer is to the constrain procedure, 
;not to the start of the table. 

;****** ******* ******* ******* ******* ******* ******* *** * 
;CONTENTS 
;------ ------- ------- ------- ------- ------- ------- --- - 

;TableRangeCheck Range checker for table lookup. 

;****** ******* ******* ******* ******* ******* ******* *** * 
;TableRangeCheck 
;------ ------- ------- ------- ------- ------- ------- --- - 

;Used to ensure that the value of W used in 
;	addwf PCL,f 
;lookup table addressing is within the range of the table. 

;W is the zero origin address 
;of the required byte within the table. 
;If W is within range, 
;its value on exit from the macro 
;is the same as its value on entry. 
;If W is not within range, 
;it is set to point to the last item in the table. 

;PCLATH is constrained to point to block zero 
;of the current page. 

TableRangeCheck MACRO TableStart,TableEnd 

	bcf PCLATH,2 
	bcf PCLATH,1 
	bcf PCLATH,0 

	addlw TableStart-TableEnd+1 
	btfsc STATUS,C 
	movlw -1 
	addlw TableEnd-TableStart-1 

	goto TableStart 

	ENDM 

;****** ******* ******* ******* ******* ******* ******* *** * 
;COPYRIGHT (C) 2005 Keith Anderson 
;------ ------- ------- ------- ------- ------- ------- --- - 

	LIST 



