;*****************************************************************************        
;
;   Module:     eeprom.inc
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    1.0 5/10/03                                                  
;
;               Functions to communicate with an 24LC64 eeprom over I2C bus
;
;*****************************************************************************        


;*****************************************************************************        
;
;   EEPROM Memory Map
;
;   The data stored in the eeprom is defined below
;
;
;    Address          Description                Size
;    (Hex)                                      (Bytes)   
;    0000    Magic Number ( 0x55, 0xAA )         2
;    0002    Configuration byte 1                1
;              MSB 7 : RFU
;                  6 : RFU         
;                  5 : RFU
;                  4 : RFU
;                  3 : RFU
;                  2 : RFU
;                  1 : Gallons/Litres 
;                  0 : Miles/KM
;    0003    Clock calibration byte              1
;    0004    Current Date & Time                 5 
;    0009    Total Mileage Recorded              3
;    000C    Total Fuel Load Recorded            3
;    000F    Number of Fuel Load Records         2 
;    0011    Next Record address to write        2  
;    0013    Total Data Records present          700 * 8 
;             Each record is 8 bytes:
;             Date (LSB unused)                     4
;             Mileage                               3 
;             Litres loaded                         1 
;    15F3    unused                                 2 
;    15F5    unused                                 2 
;    15F7    Service point records               60 * 24
;             Type ( inactive/miles/time/both )     1 
;             Date Time to trigger (No minutes)     4 
;             Mileage to trigger                    3 
;             Text to Display                       16 
;            
;    1B97    Last recorded mileage               3
;    1B9A    Spare:                              1126
;            
;   Note: Data & Time stored as minutes since 1st Jan 2000
;         Magic number is use to check is the eeprom has
;         been configured.   
;         All multibyte values are stored MSB first
;         Time stored as ddmmyyhhmm  
;
;*****************************************************************************        



;*****************************************************************************        
;
;   Function :  eepHWInit
;               configures the I2C peripherial hardware
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
eepHWInit
    ; I2C Master Mode Configuration
    movlw   B'00101000'
    movwf   SSPCON
    
    ; I2C Bit Rate Setup
    bsf     STATUS,RP0                ;SSPADD is in Bank1
    movlw   .9
    movwf   SSPADD
    bcf     STATUS,RP0                ;Restore Bank0
    
    ; I2C Slew Rate Control
    bsf     STATUS,RP0                ;SSPSTAT is in Bank1
    bsf     SSPSTAT,SMP
    bcf     STATUS,RP0                ;Restore Bank0
    
    ;SCL/SDA PIN DIRECTION SETUP
    bsf     STATUS,RP0                ;TRISC is in Bank1
    movlw   B'00011000'                ; setup value
    iorwf   TRISC,F                    ; SCL and SDA
    bcf     STATUS,RP0                ;Restore Bank0
    return



;*****************************************************************************        
;
;   Function :  eepCheck
;               If the eeprom is uninitialised, calls eepClear
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
eepCheck
    clrf    eepNum    
    clrf    eepAddH
    clrf    eepAddL
    call    eepRead

    movfw   eepData
    sublw   0x55
    btfss   STATUS, Z
    goto    eepc_001
    
    incf    eepAddL, F
    call    eepRead

    movfw   eepData
    sublw   0xAA
    btfss   STATUS, Z
    goto    eepc_001
    
    movlw   0x03
    movwf   eepAddL
    call    eepRead
    movfw   eepData    
    movwf   minOffset
    
    ; Load time & date from eeprom
    
    incf    eepAddL, F
    call    eepRead
    movfw   eepData    
    movwf   daysBCDTmp

    incf    eepAddL, F
    call    eepRead
    movfw   eepData
    movwf   monthsBCDTmp
    
    incf    eepAddL, F
    call    eepRead
    movfw   eepData
    movwf   yearsBCDTmp
    
    incf    eepAddL, F
    call    eepRead
    movfw   eepData
    movwf   hoursBCDTmp
    
    bcf     INTCON, GIE         ; Disable global interrupts
    movfw   daysBCDTmp
    movwf   daysBCD
    movfw   monthsBCDTmp
    movwf   monthsBCD
    movfw   yearsBCDTmp
    movwf   yearsBCD
    movfw   hoursBCDTmp
    movwf   hoursBCD
    clrf    minutesBCD
    bsf     INTCON, GIE         ; Enable global interrupts
    return
    
eepc_001
    call    eepClear
    return



;*****************************************************************************        
;
;   Function :  eepClear
;               Clears the eeprom array ( well, the important parts anyway )
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
eepClear
    movlw   0x55
    movwf   eepData
    clrf    eepNum    
    clrf    eepAddH
    clrf    eepAddL
    call    eepWrite            ; Magic number 1
    movlw   0xAA
    movwf   eepData
    call    eepWrite            ; Magic number 2
    clrf    eepData
    call    eepWrite            ; Config flags
    call    eepWrite            ; Clock Calibration byte
    call    eepWrite            ; Date & Time
    call    eepWrite            ; 
    call    eepWrite            ; 
    call    eepWrite            ; 
    call    eepWrite            ; 

    call    eepWrite            ; Total Miles
    call    eepWrite            
    call    eepWrite            
    call    eepWrite            ; Total Fuel
    call    eepWrite            ; 
    call    eepWrite            ; 
    call    eepWrite            ; Number of fuel records
    call    eepWrite            ; Number of fuel records
    call    eepWrite            ; Next fuel record #
    movlw   0x13
    movwf   eepData
    call    eepWrite            ; Next fuel record #

    clrf    eepData
    movlw   0x15
    movwf   eepAddH
    movlw   0xf7
    movwf   eepAddL

eec001    
    call    eepWrite            ; Reminder 1 status flag - inactive

    movlw   0x17                    ;Increment the EEPROM Adress by 1 record
    addwf   eepAddL,F
    btfsc   STATUS,C
    incf    eepAddH,F

    ; Are we at the end? Address 15f7 + (60 * 24)d = 1b97
    
    movfw   eepAddH
    sublw   0x1b
    btfss   STATUS, Z
    goto    eec001              ; No
    movfw   eepAddL
    sublw   0x97
    btfss   STATUS, Z
    goto    eec001              ; No

    clrf    eepData
    movlw   0x1B
    movwf   eepAddH
    movlw   0x97
    movwf   eepAddL
    call    eepWrite            ; Last recorded mileage
    call    eepWrite            ; Last recorded mileage
    call    eepWrite            ; Last recorded mileage

    ; Now clear all RAM
    
    ; Clear all bank0 ram
    bcf     INTCON, GIE         ; Disable global interrupts
    movlw   0x20
    movwf   FSR

clrMem2
    clrf    INDF
    incf    FSR, F
    btfss   FSR, 7              ; Have we reached 0x80?
    goto    clrMem2             ; - No, so continue with clear
    bsf     INTCON, GIE         ; Enable global interrupts
    return



;*****************************************************************************        
;
;   Function :  i2c_error
;               Error handler. We do none, allowing the watchdog to cause  
;               a reset
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
i2c_error
    sleep
    nop
    goto    i2c_error



;*****************************************************************************        
;
;   Function :  start
;               Generate a start condition on the I2C bus
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
start    
    ; Generate start Bit Condition
    bsf     STATUS,RP0            ;SSPCON2: is in Bank1
    bsf     SSPCON2,SEN

strt001
    btfsc   SSPCON2,SEN            ;Poll until end of start condition
    goto    strt001

    bcf     STATUS,RP0            ;Restore Bank0
    return



;*****************************************************************************        
;
;   Function :  restart
;               Generate a restart condition on the bus
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
restart    
    ; Generate start Bit Condition
    bsf     STATUS,RP0            ;SSPCON2: is in Bank1
    bsf     SSPCON2,RSEN

rst001
    btfsc   SSPCON2,RSEN        ;Poll until end of restart condition
    goto    rst001

    bcf     STATUS,RP0            ;Restore Bank0
    return



;*****************************************************************************        
;
;   Function :  eepHWInit
;               generate an I2C stop condition on the bus
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
stop
    bsf     STATUS,RP0            ;SSPCON2: is in Bank1
    bsf     SSPCON2,PEN
    
stp001
    btfsc   SSPCON2,PEN            ;Poll until end of stop condition
    goto    stp001
    
    bcf     STATUS,RP0            ;Restore Bank0
    return



;*****************************************************************************        
;
;   Function :  i2cxtx
;               Sends a byte over the I2C bus, and detects an ACK
;
;   Input:      8 bit byte in W
;
;   Output:     W=1 == No ACK, W=0 == ACK received
;
;*****************************************************************************        
i2cxtx
    bcf     PIR1 , SSPIF    
    movwf   SSPBUF    

    ; Wait for I2C Transmit to complete
    bsf     STATUS,RP0            ;SSPSTAT: is in Bank1
    btfss   SSPSTAT,BF            ;CheckBufferIsActuallyFull
    goto    itx002
    bcf     STATUS,RP0            

itx001
    btfss   PIR1,SSPIF
    goto    itx001
    goto    itx003

itx002
    bcf     STATUS,RP0            ;Restore Bank0
    goto    itx002

itx003
    ; Test I2C ACK Result read from slave
    bsf     STATUS,RP0            
    movlw   .1                    ;1 = Acknowledge was not received from slave
    btfss   SSPCON2,ACKSTAT
    movlw   .0                    ;0 = Acknowledge was received from slave
    bcf     STATUS,RP0            
    return



;*****************************************************************************        
;
;   Function :  tx
;               Sends an 8 bit byte over the I2C bus and monitors the ACK bit,
;               causing a system reset if not received
;
;   Input:      8 bit byte in W
;
;   Output:     None
;
;*****************************************************************************        
tx    
    call    i2cxtx
    iorlw    .0                    ;Check that the expected acknowledge is received
    btfss    STATUS,Z            ;Expect 0 = Ack received
    goto    i2c_error
    return



;*****************************************************************************        
;
;   Function :  rx
;               receives an 8 bit byte over the I2C bus
;               This function blocks until the byte is received
;
;   Input:      None
;
;   Output:     Received byte in W
;
;*****************************************************************************        
rx    
    bcf     PIR1,SSPIF            ;Clear the interrupt flag before rx
    bsf     STATUS,RP0            
    bsf     SSPCON2,RCEN
    bcf     STATUS,RP0        

rx001
    btfss   PIR1,SSPIF
    goto    rx001
    movf    SSPBUF,W
    return



;*****************************************************************************        
;
;   Function :  nack
;               Asserts a NACK condition on the BUS
;
;   Input:      None
;
;   Output:     Received byte in W
;
;*****************************************************************************        
nack
    bcf     PIR1,SSPIF            ;Clear the interrupt flag before ACK
    bsf     STATUS,RP0    
    bsf     SSPCON2,ACKDT        ;Set ACKDT to transmit an NACK Bit
    bsf     SSPCON2,ACKEN        ;start Acknowledge Sequence
    bcf     STATUS,RP0
    
nak001
    btfss   PIR1,SSPIF            ;Interrupt will be set at end of Acknowledge
    goto    nak001
    return



;*****************************************************************************        
;
;   Function :  eepWrite
;               This is a global function for writing data to the eeprom
;
;   Input:      EEPROM address in eepNum + eepAddH
;               + eepAddL. 
;               
;
;   Output:     Byte to send in eepData. Input address is incremented by 1
;
;*****************************************************************************        
eepWrite
    call     start

ew001
    call    restart                ;Generate a start Condition
    movlw   B'10100000'            ;control byte
    iorwf   eepNum, W
    call    i2cxtx
    iorlw   .0                    ;If the EERPOM is still writing then it will not ACKNOWLEDGE. 
    btfss   STATUS,Z    
    goto    ew001                ;Hence we keep trying until the EEPROM is available.
    movf    eepAddH,W            ;Move the address to transmit and send it
    call    tx
    movf    eepAddL,W
    call    tx
    movf    eepData,W    
    call    tx
    call    stop            
    movlw   .1                    ;Increment the EEPROM Adress by 1
    addwf   eepAddL,F
    btfsc   STATUS,C
    incf    eepAddH,F

    movlw   0x02                ; Allow eeprom time to write the byte
    call    uiWait10ms          
    return



;*****************************************************************************        
;
;   Function :  eepRead
;               This is a global function for reading data from the eeprom
;
;   Input:      EEPROM address in eepNum + eepAddH
;               + eepAddL. 
;               
;
;   Output:     Received byte in eepData
;
;*****************************************************************************        
eepRead
    call    start        
    movlw    B'10100000'            ;control byte
    iorwf   eepNum, W
    call    tx        
    movf    eepAddH,W
    call    tx
    movf    eepAddL,W
    call    tx
    call    restart                ;Generate a start Condition
    movlw    B'10100001'            ;control byte
    iorwf   eepNum, W
    call    tx
    call    rx
    movwf    eepData
    call     nack
    call    stop        
    return
