;*****************************************************************************        
;
;   Module:     master.asm
;               
;   Author:     Mike Hibbett, mikehibbett@oceanfree.net
;                                                                  
;   Version:    1.0 18/01/06                                                  
;
;               Source code of the RS485 Master Device
;               Everyday Practical Electronics
;
;*****************************************************************************
;
; The circuit I used is based on a PIC 16F877 running at 3.579545MHz
; Why? Because I had those parts available, no other reason.
;
; PortB.0 is an output to drive a status LED, LED0
; PortB.1 is an output to drive a status LED, LED1
; PortC.6 is configured as the async serial output
; PortC.7 is configured as the async serial input
; PortC.5 is configured as an output for enabling the RS485 transmitter
; on the bus
;
; ALL other I/O ports are configured as outputs and set to 0 volts
; so I do not have to pull them high or low with resistors.
; In a real production design you should make unused I/O ports inputs
; and tie them to ground. This will protect them from being damaged by
; static electricity.


    list p=16f877, st=OFF, x=OFF, n=0
    errorlevel -302
    errorlevel -306

    #include <p16f877.inc>

    __config  0x3f32

    ; Pull in variable definitions and general constants
    #include "equates.inc"
    

;*****************************************************************************        
;
;   Function :  Reset vector
;               Hardware entry point to the code
;               When the processor powers up, this is always the first 
;               location executed.
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
    
    ORG    0

RESET        GOTO    Main 


;*****************************************************************************        
;
;   Function :  Interrupt vector
;               Hardware entry point for interrupts
;               This address, location 4, is the address that is called when 
;               an interrupt occurs.
;               You do of course have to enable interrupts if you want this
;               routine to be executed.
;
;   Input:      None.
;
;   Output:     None.
;
;*****************************************************************************        
    
    ORG    4
    
INTERRUPT
    movwf   W_TEMP
    swapf   STATUS, W
    clrf    STATUS
    movwf   STATUS_TEMP    
    movfw   PCLATH
    movwf   PCLATH_TEMP
    clrf    PCLATH

    ; A timer interrupt occurred. Update the timer value  
    movlw   LOW TIMER1_1MS_VAL  
    movwf   TMR1L
    movlw   HIGH TIMER1_1MS_VAL  
    movwf   TMR1H                   ; Restart the timer
    
    ; If the application timer has not expired ( reached 0 ) then decrement it
    movfw   timer1msCount           
    btfss   STATUS, Z
    decf    timer1msCount, F

    bcf     PIR1, TMR1IF            ; Must clear interrupt flag

    movfw   PCLATH_TEMP
    movwf   PCLATH
    swapf   STATUS_TEMP, W
    movwf   STATUS
    swapf   W_TEMP,F
    swapf   W_TEMP, W
    retfie
    


;*****************************************************************************        
;
;   Function :  initHardware
;               This function configures the initial hardware features of the 
;               processor. To be more specific it changes only the features that
;               have not already been configured by the power on reset.
;
;               This function is called once following power-up. In a real 
;               production system you should really configure *everything*, and
;               also re-do the configuration periodically. This is because in 
;               real life noise or ESD can sometimes cause things like I/O port 
;               direction definitions to 'flip'. Resetting the configuration
;               periodically would correct such 'soft' errors.
;
;               As this is a simple demonstration of the 'normal' code, those
;               techniques to produce 'rock solid' reliable code are left out
;               for clarity.
;
;               I may make an article on 'writing reliable code' one day!
;               
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
initHardware
    ; The status of RAM following a reset is undefined.
    ; So set them to a known value.
    movlw   0x20
    movwf   FSR
clrMem
    clrf    INDF
    incf    FSR, F
    btfss   FSR, 7              ; Have we reached 0x80?
    goto    clrMem              ; - No, so continue with clear


    clrf    sPORTB              ; Shadow value of PORTB output settings
    clrf    sPORTC              ; Shadow value of PORTC output settings
    
    ; Port A, D and E are unused so set to outputs, low level
    bsf     STATUS, RP0 
    movlw   0x06
    movwf   ADCON1
    clrf    TRISA
    clrf    TRISD
    clrf    TRISE
    bcf     STATUS, RP0 
    
    ; Port B: RB0/RB1 are outputs for status LED's
    ; Initial state of status LED's is off.
    ; The rest are unused, so set to outputs
    bsf     STATUS, RP0 
    clrf    TRISB
    bcf     STATUS, RP0 
    
    ; Port C: RC6/RC6 are set as the async uart;
    ; RC5 is set to output to control the transmitter ( and set low, off)
    ; The rest are unused, so set to outputs
    bsf     STATUS, RP0 
    movlw   0x80
    movwf   TRISC
    bcf     STATUS, RP0 
    
    ; Configure the Async serial port: 
    ; 9 bit data ( 9th bit is used to indicate an address byte )
    bsf     STATUS, RP0 
    movlw   BAUD_CONSTANT    ;set baud rate 
    movwf   SPBRG
    bsf     TXSTA, BRGH    ;baud rate high speed option
    bsf     TXSTA, TX9      ; 9 bit data transfer
    bsf     TXSTA, TXEN    ;enable transmission
    bcf     STATUS,RP0    
    bsf     RCSTA, CREN    ;enable reception
    bsf     RCSTA, RX9
    bsf     RCSTA, SPEN    ;enable serial port
    
    ; Configure timer1 to give us a 1ms tick interval
    movlw   0x35    ; Timer on, int clock, 1:8 prescale
    movwf   T1CON
    bsf     STATUS, RP0    
    bsf     PIE1, TMR1IE
    bcf     STATUS, RP0    

    ; Enable global interrupts, so that the timer 
    ; interrupt routine will get called
    clrf    INTCON
    bsf     INTCON, PEIE
    bsf     INTCON, GIE         
    
    return
    


;*****************************************************************************        
;
;   Function : uiWait10ms
;              delays for a multiple of 10ms
;
;   Input:     multiple in W
;
;*****************************************************************************        
uiWait10ms
    movwf   delay1

uiw000
    ; Load the interrupt routines timer for 10ms
    movlw   D'10'
    movwf   timer1msCount

uiw001
    ; Keep reading the timer value, and quit
    ; when it has reached zero.
    movfw   timer1msCount
    btfss   STATUS, Z
    goto    uiw001    
    decfsz  delay1, F
    goto    uiw000
    
    return



;*****************************************************************************        
;
;   Function :  getSlaveData
;               Sends a message to the specified slave on the RS485 bus,
;               and waits for a response. If a message is not recieved within
;               the time period RS485_COMMS_TIMEOUT_MS, then the call returns
;               and error in deviceReadOK
;
;   Input:      RS485 Slave address (8bit) in W
;
;   Output:     data byte returned in deviceData ( undefined on timeout )
;               status in deviceReadOK: 0x01 for ok, 0x00 for no response
;
;*****************************************************************************        
getSlaveData
    movwf   deviceData
    call    RS485TXEnabled
    
    ; Send the address byte out (ie, set the 9th bit)
    btfss   PIR1,TXIF    ;check that buffer is empty
    goto    $-1
    bsf     STATUS, RP0 
    bsf     TXSTA, TX9D
    bcf     STATUS, RP0 
    movfw   deviceData
    movwf   TXREG
           
    ; Send out the request data byte (ie, clear the 9th bit)
    btfss   PIR1,TXIF    ;check that buffer is empty
    goto    $-1
    bsf     STATUS, RP0 
    bcf     TXSTA, TX9D
    bcf     STATUS, RP0 
    movlw   READ_DATA_COMMAND
    movwf   TXREG
    
    ; Wait for 2ms to allow data to be sent
    movlw   D'2'
    movwf   timer1msCount
    movfw   timer1msCount
    btfss   STATUS, Z  
    goto    $-2
    
    call    RS485TXDisabled
    
    ; setup the response timer
    movlw   SLAVE_READ_TIMEOUT
    movwf   timer1msCount    
    
gsd001    
    ; Check to see if we have timed out
    movfw   timer1msCount
    
    btfsc   STATUS, Z
    goto    gsdRxErr        ; Timeout, so handle it
    
    ; received byte == address of master?
    btfss   PIR1,RCIF       ; check if data received
    goto    gsd001          ; wait until new data
    btfsc   RCSTA, RX9D
    goto    gsd001a
    
    ; Not an address, so read the byte and keep looking
    movf    RCREG,W
    goto    gsd001 

gsd001a
    movf    RCREG,W         ; get received data into W
    sublw   MASTER_ADDRESS  ; Is the master being addressed?
    btfss   STATUS, Z
    goto    gsd001          ; Not to the master, keep looking
    
    ; receive data byte
gsd002
    ; Check to see if we have timed out
    movfw   timer1msCount
    
    btfsc   STATUS, Z
    goto    gsdRxErr        ; Timeout, so handle it

    btfss   PIR1,RCIF       ; check if data received
    goto    gsd002          ; wait until new data
    btfsc   RCSTA, RX9D
    goto    gsdRxErr        ; Not data, so its a protocol error
    movf    RCREG,W         ; get received data into W
    
    movwf   deviceData
    movlw   0x01
    movwf   deviceReadOK
    return
    
gsdRxErr
    movf    RCREG,W
    clrf    deviceData  
    clrf    deviceReadOK    ; Indicate an error   
    return



;*****************************************************************************        
;
;   Function :  Various LED control functions.
;               These functions are used to turn the status leds on or off
;               Although they have a very simple job, they have been given 
;               their own functions to make the code easier to read.
;
;   Input:      None
;
;   Output:     LED changed; PORTB shadow variable sPORTB updated
;
;*****************************************************************************        
slave1LEDOff
    bcf     sPORTB, SLAVE_1_LED_BIT
    movfw   sPORTB
    movwf   PORTB   
    return
slave1LEDOn
    bsf     sPORTB, SLAVE_1_LED_BIT
    movfw   sPORTB
    movwf   PORTB   
    return
slave2LEDOff
    bcf     sPORTB, SLAVE_2_LED_BIT
    movfw   sPORTB
    movwf   PORTB   
    return
slave2LEDOn
    bsf     sPORTB, SLAVE_2_LED_BIT
    movfw   sPORTB
    movwf   PORTB   
    return



;*****************************************************************************        
;
;   Function :  RS485TXEnabled, RS485TXDisabled
;               These functions are used to turn the RS485 transmitter in the
;               interface IC on or off.
;               Although they have a very simple job, they have been given 
;               their own functions to make the code easier to read.
;
;   Input:      None
;
;   Output:     Transmitter driver enabled or disabled
;               PORTC shadow variable sPORTC updated
;
;*****************************************************************************        
RS485TXEnabled
    bsf     sPORTC, TX_ENABLE_BIT
    movfw   sPORTC
    movwf   PORTC    
    return
RS485TXDisabled    
    bcf     sPORTC, TX_ENABLE_BIT
    movfw   sPORTC
    movwf   PORTC    
    return
           


;*****************************************************************************        
;
;   Function :  Main
;               Main application loop
;
;   Input:      None.
;
;   Output:     N/A
;
;*****************************************************************************        
Main
    clrf    STATUS                ; Select bank 0
    clrf    INTCON                ; No interrupts enabled, at start-up
    clrf    PCLATH                ; Our code starts in first bank

    call    initHardware  
    
    ; light the leds during startup,
    ; to show the thing works
    call    slave1LEDOn
    call    slave2LEDOn

    ; time 2 seconds for the LED's to stay on.
    movlw   D'200'
    call    uiWait10ms
    
    ; Leds off
    call    slave1LEDOff
    call    slave2LEDOff
    

    ; Run the test continuously
    
mainloop
    
    ; Send a message to the slave: It return the value on its port RB.7 pin
    movlw   SLAVE_1_ADDRESS
    call    getSlaveData            
    
    ; If an error occurred during the read (like, it isn't connected)
    ; then handle the error
    btfss   deviceReadOK, 0
    goto    slave1NoResponse
    
    ; Update the status LED to match the signal level
    movfw   deviceData
    sublw   0x01
    btfsc   STATUS, Z
    call    slave1LEDOn

    movfw   deviceData
    sublw   0x01
    btfss   STATUS, Z
    call    slave1LEDOff
    
slave2
    
    ; Send a message to the slave: It return the value on its port RB.7 pin
    movlw   SLAVE_2_ADDRESS
    call    getSlaveData            
    
    ; If an error occurred during the read (like, it isn't connected)
    btfss   deviceReadOK, 0
    goto    slave2NoResponse
    
    ; Update the status LED to match the signal level
    movfw   deviceData
    sublw   0x01
    btfsc   STATUS, Z
    call    slave2LEDOn
    movfw   deviceData
    sublw   0x01
    btfss   STATUS, Z
    call    slave2LEDOff
    
    goto    mainloop


    ; If there was an error reading the port, just turn the LED off
slave1NoResponse
    call    slave1LEDOff
    goto    slave2  

    ; If there was an error reading the port, just turn the LED off
slave2NoResponse
    call    slave2LEDOff
    goto    mainloop  


    
    END                ; End of program
