
;*****************************************************************************        
;
;   Module:     app.inc
;               
;   Author:     Mike Hibbett, mikehibbett@oceanfree.net 
;                                                                  
;   Version:    1.0 26/07/06                                              
;
;               The main user application code , entered
;               following execution of the bootloader. 
;
;               All your application code should go in here, or be #included.
;               The exception is interrupt routines, which should go in
;               interrupts.inc and the setting of the config registers, which 
;               should go in config.inc
;
;               CS is on RC2; SPI stuff on normal pins; RS232 on normal pins
;
;*****************************************************************************        


;*****************************************************************************        
;
; MACROS
;
;*****************************************************************************        

DISPLAY_STR     macro   str_addr
                clrf    TBLPTRU     ; Memory is limited to 64KB
                movlw   HIGH str_addr 
                movwf   TBLPTRH
                movlw   LOW str_addr 
                movwf   TBLPTRL
                call    DspRomStr                
                endm


    ; Pull in the source code for the SPI low level routines
    #include "spi.inc"    

    ; Pull in the source code for the MMC Card low level routines
    #include "mmc.inc"    



;*****************************************************************************        
;
; RAM Variables
;
;*****************************************************************************        
tmpHexByte  EQU 0x14        ; used by DspHexByte for formatting text
MMCCmd      EQU 0x15
MMCOp1      EQU 0x16
MMCOp2      EQU 0x17
MMCOp3      EQU 0x18
MMCOp4      EQU 0x19
MMCOp5      EQU 0x1A
tmpCount    EQU 0x1B        ; General purpose counter
blockSize   EQU 0x1C        ; Block size indicator; 0 = one, 1 = 512
row         EQU 0x1D        ; Used to display 512 byte block buffer
column      EQU 0x1E        ; Used to display 512 byte block buffer
indH        EQU 0x1F        ; high byte of index into buffer
indL        EQU 0x20        ; low byte of index into buffer
tmpByte     EQU 0x21


BLOCK_BUFFER_ADDRESS  EQU 0x100  ; We use 512 bytes from this address



;*****************************************************************************        
;
;   Function :  Main
;               This is the entry point of your actual application program
;
;               This code just demonstrates the SPI functions in SPI.INC
;
;   Input:      None.
;
;   Output:     None - it never returns
;
;*****************************************************************************        
Main
    ; Do the basic hardware initialisation
    call    AppHWInit
    
    ; Configure the PICs internal SPI module for communication
    call    SPIHWInit
    
    ; Show the welcome screen, and help display.
    DISPLAY_STR STR_CRLF
    DISPLAY_STR STR_MAIN
    DISPLAY_STR STR_CRLFPROMPT
    call    DspHelpScreen
    
    ; Reset MMC card following power up
    call    MMCReset

    ; Software reset of card
    call    MMCInit
        
    ; Now, enter a loop awaiting commands over the RS232
mloop    
    DISPLAY_STR STR_CRLFPROMPT

    call    RS232RxByte    
    
    movf    hostByte, W
    sublw   'Z'
    btfsc   STATUS, Z
    call    ActionInitCard

    movf    hostByte, W
    sublw   'z'
    btfsc   STATUS, Z
    call    ActionInitCard

    movf    hostByte, W
    sublw   'h'
    btfsc   STATUS, Z
    call    DspHelpScreen
    
    movf    hostByte, W
    sublw   'H'
    btfsc   STATUS, Z
    call    DspHelpScreen

    movf    hostByte, W
    sublw   '1'
    btfsc   STATUS, Z
    call    ActionBlockSize1

    movf    hostByte, W
    sublw   '5'
    btfsc   STATUS, Z
    call    ActionBlockSize512    

    movf    hostByte, W
    sublw   'd'
    btfsc   STATUS, Z
    call    ActionDspBuffer    

    movf    hostByte, W
    sublw   'D'
    btfsc   STATUS, Z
    call    ActionDspBuffer    

    movf    hostByte, W
    sublw   'E'
    btfsc   STATUS, Z
    call    ActionEditBuffer    

    movf    hostByte, W
    sublw   'e'
    btfsc   STATUS, Z
    call    ActionEditBuffer    

    movf    hostByte, W
    sublw   'W'
    btfsc   STATUS, Z
    call    ActionWriteBuffer    

    movf    hostByte, W
    sublw   'w'
    btfsc   STATUS, Z
    call    ActionWriteBuffer    

    movf    hostByte, W
    sublw   'R'
    btfsc   STATUS, Z
    call    ActionReadBuffer    

    movf    hostByte, W
    sublw   'r'
    btfsc   STATUS, Z
    call    ActionReadBuffer    

    goto    mloop
    


;*****************************************************************************        
;
;   Function :  ActionReadBuffer
;               Reads 8 hex digits from the rs232 port, then reads the MMC card
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
ActionReadBuffer
    ; Get the space character
    call    RS232RxByte  
    
    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    movwf   MMCOp1 
    swapf   MMCOp1, F
    
    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    addwf   MMCOp1, F

    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    movwf   MMCOp2 
    swapf   MMCOp2, F
    
    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    addwf   MMCOp2, F

    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    movwf   MMCOp3 
    swapf   MMCOp3, F
    
    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    addwf   MMCOp3, F

    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    movwf   MMCOp4 
    swapf   MMCOp4, F
    
    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    addwf   MMCOp4, F
    
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_READING
    
    call    MMCReadBlock

    DISPLAY_STR STR_CRLFPROMPT
    
    return


    
;*****************************************************************************        
;
;   Function :  ActionWriteBuffer
;               Reads 8 hex digits from the rs232 port, then writes the buffer
;               to the MMC card
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
ActionWriteBuffer
    ; Get the space character
    call    RS232RxByte  

    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    movwf   MMCOp1 
    swapf   MMCOp1, F
    
    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    addwf   MMCOp1, F

    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    movwf   MMCOp2 
    swapf   MMCOp2, F
    
    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    addwf   MMCOp2, F

    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    movwf   MMCOp3 
    swapf   MMCOp3, F
    
    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    addwf   MMCOp3, F

    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    movwf   MMCOp4 
    swapf   MMCOp4, F
    
    call    RS232RxByte  
    movf    hostByte, W   
    call    char2Bin 
    addwf   MMCOp4, F
    
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_WRITING
    
    call    MMCWriteBlock

    DISPLAY_STR STR_CRLFPROMPT
    
    return
    
    

;*****************************************************************************        
;
;   Function :  ActionEditBuffer
;               Starts editing the ram buffer ( after entry of a 3 digit hex 
;               index into the buffer.
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
ActionEditBuffer
    ; Get the space character
    call    RS232RxByte  
    
    ; get index digit 1
    call    RS232RxByte  
    movf    hostByte, W
    andlw   0x01            ; Only '1' or '0' valid
    movwf   indH
    call    RS232RxByte  
    movf    hostByte, W
    call    char2Bin        ; convert into a binary number
    movwf   indL
    swapf   indL, F
    call    RS232RxByte  
    movf    hostByte, W
    call    char2Bin        ; convert into a binary number
    addwf   indL, F
    
    ; Set the buffer pointer to that address, and start editing
    movf    indH, W
    addlw   HIGH BLOCK_BUFFER_ADDRESS
    movwf   FSR0H
    movf    indL, W
    movwf   FSR0L
    
aeb001    
    DISPLAY_STR STR_CRLFPROMPT
    
    movlw   '('
    call    RS232TxByte
    movf    POSTINC0, W
    call    DspHexByte
    movlw   ')'
    call    RS232TxByte
    movlw   ' '
    call    RS232TxByte
    
    ; get key - digit, ENTER to leave unchanged, ESc to end
    call    RS232RxByte
    movf    hostByte, W
    sublw   0x0d
    btfsc   STATUS, Z
    goto    aeb001

    movf    hostByte, W
    sublw   0x0a
    btfsc   STATUS, Z
    goto    aeb001
    
    movf    hostByte, W
    sublw   0x1b
    btfsc   STATUS, Z
    goto    aebexit
    
    ; get the character, in 2 digits
    movf    hostByte, W
    call    char2Bin        ; convert into a binary number
    movwf   tmpByte
    swapf   tmpByte, F
    call    RS232RxByte  
    movf    hostByte, W
    call    char2Bin        ; convert into a binary number
    addwf   tmpByte, F
    
    ; Write the byte back
    movf    POSTDEC0, W
    movf    tmpByte, W
    movwf   POSTINC0
    
    goto    aeb001
    
aebexit
    DISPLAY_STR STR_CRLFPROMPT
    
    return

    
    
;*****************************************************************************        
;
;   Function :  ActionDspBuffer
;               Displays the size of the block setting, then dumps the buffer
;               - all 512 bytes, even if only 1 byte is set
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
ActionDspBuffer
    DISPLAY_STR STR_CRLFPROMPT
    
    ; Display the block setting
    
    movf    blockSize, W
    btfss   STATUS, Z
    goto    adb001
    
    DISPLAY_STR STR_BSIZE1
    goto    adb002
    
adb001
    DISPLAY_STR STR_BSIZE512
    
adb002
    ; Now dump the 512 bytes, 32 bytes per line

    ; Setup indirect register to point to the buffer
    lfsr    FSR0, BLOCK_BUFFER_ADDRESS
    
    movlw   D'16'
    movwf   row
    
adb003    
    DISPLAY_STR STR_CRLFPROMPT

    movlw   D'32'
    movwf   column

adb004
    movf    POSTINC0, W
    call    DspHexByte    
    
    decfsz  column, F
    goto    adb004
    
    decfsz  row, F
    goto    adb003

    return    
    
    
    
;*****************************************************************************        
;
;   Function :  ActionInitCard
;               Inits the MMC card, gets the CID and displays it
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
ActionInitCard
    DISPLAY_STR STR_CRLFPROMPT
    
    ; Software reset of card
    call    MMCInit

    call    MMCDspType

    return    
    


;*****************************************************************************        
;
;   Function :  ActionBlockSize1
;               Sets the block buffer size to a single byte, and informs
;               the user
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
ActionBlockSize1
    DISPLAY_STR STR_CRLFPROMPT
    
    call    MMC1ByteBuff
    
    movlw   0x00
    movwf   blockSize
    DISPLAY_STR STR_BSIZE1
    
    return



;*****************************************************************************        
;
;   Function :  ActionBlockSize512
;               Sets the block buffer size to 512
;               the user
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
ActionBlockSize512
    DISPLAY_STR STR_CRLFPROMPT
    
    call    MMC512ByteBuff
    
    movlw   0x01
    movwf   blockSize
    DISPLAY_STR STR_BSIZE512
    
    return
        


;*****************************************************************************        
;
;   Function :  AppHWInit
;               Perform the hardware initialisation required by the application
;
;   Input:      None.
;
;   Output:     None
;
;*****************************************************************************        
AppHWInit
    ; Ensure all unused I/O pins are set to outputs
    
    ; PortA as I/O pins
    movlw   0x07    ; Disconnect comparitor from I/O pins
    movwf   CMCON 
    
    movlw   0x00    ; reference voltage not used
    movwf   CVRCON 
    
    movlw   0x0F    ; Disconnect A/D from I/O pins
    movwf   ADCON1 
    
    clrf    TRISA
    clrf    PORTA
    clrf    TRISB
    clrf    PORTB
    
    ; The main application will call an SPI init routine to set up the 
    ; SPI hardware requirements, and we will use the default bootloader RS232 
    ; interface for user I/O

    ; Set our default buffer block size, which is 512 bytes    
    movlw   0x01    
    movwf   blockSize       
        
    return
    


;*****************************************************************************        
;
;   Function :  DspHelpScreen
;               Displays the help text
;
;   Input:      None.
;
;   Output:     Bytes written over rs232
;
;*****************************************************************************        
DspHelpScreen
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_HELP1
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_HELP2
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_HELP3
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_HELP4
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_HELP5
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_HELP6
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_HELP7
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_HELP8
    DISPLAY_STR STR_CRLFPROMPT
    DISPLAY_STR STR_HELP9
    DISPLAY_STR STR_CRLFPROMPT
    
    return
    


;*****************************************************************************        
;
;   Function :  DspRomStr
;               Writes a null terminated string, stored in code space, to the
;               rs232 port.
;
;               The strings should be defines using the DA derective, storing
;               2 bytes per word location.
;
;   Input:      TBLPTR registers point to begining of string
;
;   Output:     bytes written
;
;*****************************************************************************        
DspRomStr
    tblrd*+
    movf    TABLAT, W
    btfsc   STATUS, Z
    return
    call    RS232TxByte
    goto    DspRomStr



;*****************************************************************************        
;
;   Function :  DspHexByte
;               Write a 2 digit hex byte in uppercase to the user interface
;
;   Input:      byte to display in W
;
;   Output:     2 characters written to the UI
;
;*****************************************************************************        
DspHexByte    
    movwf   tmpHexByte
    swapf   tmpHexByte,W
    sublw   0x09
    swapf   tmpHexByte,W
    andlw   0x0F
    btfss   STATUS,DC
    addlw   'A' - .10 - '0'
    addlw   '0'
    call    RS232TxByte
    movf    tmpHexByte, W
    sublw   0x09
    movf    tmpHexByte, W
    andlw   0x0F
    btfss   STATUS,DC
    addlw   'A' - .10 - '0'
    addlw   '0'
    call    RS232TxByte
    
    return



;*****************************************************************************        
;
;   Function :  char2Bin
;               converts a single character 0..F into a binary value
;
;   Input:      char to convert in W. Upper or lower case
;
;   Output:     binary value 0 .. 15 returned in W
;
;*****************************************************************************        
char2Bin
    addlw   0xC0    ; This will overflow for A..F
    btfsc   STATUS, C
    addlw   D'9'
    andlw   0x0F    
    
    return
    


;*****************************************************************************        
;
; STRING CONSTANTS
;
;*****************************************************************************        

STR_MAIN        DA  "EPE MMC Interface v1.0", 0x00
STR_CRLF        DA   0x0a0d, 0x00
STR_CRLFPROMPT  DA   0x0a0d, "> ", 0x00
STR_HELP1       DA  "The following commands are supported:", 0x00
STR_HELP2       DA  "1       Set the block size to 1 byte", 0x00
STR_HELP3       DA  "5       Set the block size to 512 bytes", 0x00
STR_HELP4       DA  "R addr  Read a block from address addr ( 8 digit hex )", 0x00
STR_HELP5       DA  "W addr  Write a block to address addr ( 8 digit hex )", 0x00
STR_HELP6       DA  "H       Display this screen", 0x00
STR_HELP7       DA  "E addr  Edit the block buffer ( 3 digit hex )", 0x00
STR_HELP8       DA  "D       Display the block buffer", 0x00
STR_HELP9       DA  "Z       Initilise MMC and display the CID", 0x00
STR_CID         DA  "Card CID is ",0x00
STR_BSIZE1      DA  "Block Size is 1 byte", 0x00
STR_BSIZE512    DA  "Block Size is 512 bytes", 0x00
STR_READING     DA  "Reading Block from card...", 0x00
STR_WRITING     DA  "Writing Block to card...", 0x00

