;BIKE440.PIC 31JAN97
;set for 3.2768MHz XTAL

#DEFINE PAGE0   BCF $03,5
#DEFINE PAGE1   BSF $03,5
        .AVSYM
INDF:   .EQU $00        ;page 0 & 1
TMR0:   .EQU $01        ;0
OPTION: .EQU $01        ;1
PCL:    .EQU $02        ;0 & 1
STATUS: .EQU $03        ;0 & 1
FSR:    .EQU $04        ;0 & 1
PORTA:  .EQU $05        ;0
TRISA:  .EQU $05        ;1
PORTB:  .EQU $06        ;0
TRISB:  .EQU $06        ;1
                        ;$07 not available
EEDATA: .EQU $08        ;0
EECON1: .EQU $08        ;1
EEADR:  .EQU $09        ;0
EECON2: .EQU $09        ;1
PCLATH: .EQU $0A        ;0 & 1
INTCON: .EQU $0B        ;0 & 1

LOOPA:  .EQU $0C        ;loop counter
TURN10: .EQU $0D        ;turns in 10 secs counter
TRNSEC: .EQU $0E        ;turns in 10 secs store
STORE1: .EQU $0F        ;general store
TSTFLG: .EQU $10        ;initialisation flag, with other uses

TURNS:  .EQU $11        ;wheel turns counted
TRNVAL: .EQU $12        ;additive value for turns distance LSB
TRNVLM: .EQU $13        ;ditto MSB
KMSBN1: .EQU $14        ;kilometres byte 1
KMSBN2: .EQU $15        ;kilometres byte 2
KMSBN3: .EQU $16        ;kilometres byte 3

CLKCNT: .EQU $17        ;pre-counter for CLOCK
CLKSEC: .EQU $18        ;CLOCK main counter - secs
CLKMIN: .EQU $19        ;CLOCK - mins
CLKHRS: .EQU $1A        ;CLOCK - hours

RSTFLG: .EQU $1B        ;reset flag counter
WORK1:  .EQU $1C        ;working store 1
WORK2:  .EQU $1D        ;working store 2
WORK3:  .EQU $1E        ;working store 3
WORK4:  .EQU $1F        ;working store 4

DEC1:   .EQU $20        ;decimalisation byte 1 & other uses
DEC2:   .EQU $21        ; byte 2
DEC3:   .EQU $22        ; byte 3
DEC4:   .EQU $23        ; byte 4
DEC5:   .EQU $24        ; byte 5
ANSA1:  .EQU $25        ;decimalisation answer store 1 & other uses
ANSA2:  .EQU $26        ; answer 2
ANSA3:  .EQU $27        ; answer 3
ANSA4:  .EQU $28        ; answer 4
ANSA5:  .EQU $29        ; answer 5

STORE3: .EQU $2A        ;general store 3
LOOPB:  .EQU $2B        ;general loop 2
STORE2: .EQU $2C        ;general store 2
MLTPL1: .EQU $2D        ;multiply lsb & other uses
MLTPL2: .EQU $2E        ;multiply nmsb & other uses
MLTPL3: .EQU $2F        ;multiply msb & other uses

W:      .EQU 0  ;Result to go into working register (accumulator)
F:      .EQU 1  ;Result to go into a file register.

C:      .EQU 0  ;Carry flag  (located in STATUS register)
DC:     .EQU 1  ;Digit carry            "
Z:      .EQU 2  ;Zero flag              "
PD:     .EQU 3  ;Power Down bit         "
TO:     .EQU 4  ;Time-out bit           "
WR:     .EQU 1  ;eeprom write initiate address
WREN:   .EQU 2  ;eeprom write enable address
RD:     .EQU 0  ;eeprom read enable address
GIE:    .EQU 7  ;GIE bit in INTCON reg

                ;TSTFLG USE:
                ;bit 0 store for 4-bit switch
                ;bit 1 -ditto-
                ;bit 2 -ditto-
                ;bit 3 screen line flag
                ;bit 4 kms/miles flag. kms=0 miles=1
                ;bit 5 RS flag for LCD
                ;bit 6 Add/Subtract flag
                ;bit 7 set-up routine complete flag

        .ORG $0004      ;Interrupt vector address
        GOTO GIEOFF     ;Jump to interrupt routine on interrupt
        .ORG $0005      ;Start of program memory

GIEOFF: BCF INTCON,GIE  ;turn off global interrupts
        BTFSC INTCON,GIE
        goto GIEOFF

        PAGE1           ;Defined above.
        MOVLW %00000001
        MOVWF TRISB     ;Configure port B1-B7 as output, B0 as input
        movlw %00010111
        MOVWF TRISA     ;Port A0-A2, A4 as input, A3 as output
        MOVLW %00000110 ;set timer ratio 1:128
        MOVWF OPTION
        PAGE0

        CLRF PORTA
        CLRF PORTB
        BSF PORTA,3     ;turn off buzzer

RESETA: call RESETB     ;reset all working bytes

        btfss PORTA,4   ;is full reset switch detected? (bit 4=0)
        call RSTPRM     ;yes bit 4 = 0

SETUP:  call PRMGET     ;get eeprom data
        clrf TSTFLG
        movlw 10        ;initial value reset flag
        movwf RSTFLG
        movlw 8         ;set LCD set-up loop for 8
        movwf LOOPA
        movlw 5         ;1/5th sec wait set
        movwf CLKCNT
        clrf INTCON     ;Clear interupt flag

PAUSE:                  ;initial 1/5th sec wait before setting up LCD
        call NEGTIV     ;toggle LCD -VE gen

        btfss INTCON,2  ;has a timer time-out been detected?
        goto PAUSE      ;no
        BCF INTCON,2    ;yes
        decfsz CLKCNT,F
        goto PAUSE
        movlw 25        ;initial basic CLKCNT val for secs timing
        movwf CLKCNT

INTRPT:                 ;working interrupt routine
        call NEGTIV     ;toggle LCD -VE gen

        btfss INTCON,2  ;has a timer time-out been detected?
        goto TSTBT1     ;no
        BCF INTCON,2    ;yes
        call TSTBT2
TSTBT1: btfss INTCON,1  ;has a wheel rotation been detected?
        goto INTRPT     ;no
        incf TURNS,F    ;inc wheel turns count
        incf TURN10,F   ;inc turns per 10 secs counter
        btfss STATUS,Z  ;is TURN10 = 0 (i.e. >255)?
        goto TSTBT4     ;no
        decf TURN10,F   ;yes, dec to 255
TSTBT4: BCF INTCON,1
        goto INTRPT

TSTBT2: btfsc TSTFLG,7  ;is set-up routine complete?
        goto DISPLY     ;yes
        BCF PORTB,5     ;clear RS line
        BCF TSTFLG,5    ;clear RS flag
        movf TSTFLG,W   ;get instruction address
        call TABINT     ;get set-up instruction
        call LCDOUT     ;perform it
        incf TSTFLG,F   ;inc flag
        btfss TSTFLG,3  ;has last set-up instruction now been done?
        goto TSTBT3     ;no
        clrf TSTFLG     ;yes
        BSF TSTFLG,7    ;set flag
TSTBT3: RETURN

DISPLY: BSF PORTB,5     ;set RS line high
        BSF TSTFLG,5    ;set RS flag high

                        ;next 3 commands are for external diameter use
                        ;& may be deleted if diameter setting internally
        PAGE1
        BSF TRISA,3     ;Port A3 as input
        PAGE0

        btfss PORTA,4   ;is power-down detected? (bit 4=0)
        goto STORIT     ;yes, bit 4 = 0, store current data & end it

        call CLKIN
        BSF PORTB,7     ;set switch on
        movlw %11101000 ;reset TSTFLG bits 0-2 & 4
        andwf TSTFLG,F
        movf PORTA,W    ;get switch settings
        andlw 7
        movwf STORE1    ;store switch into STORE1
        iorwf TSTFLG,F  ;store switch into TSTFLG

                        ;next 5 commands are for external diameter use
                        ;& may be deleted if setting diameter internally
        btfss PORTA,3   ;is diameter change detected? (bit 3=0)
        goto WHEEL      ;yes
        PAGE1
        BCF TRISA,3     ;Port A3 as output
        PAGE0

        BCF PORTB,7     ;turn off switch
        call MILFLG     ;get kms/miles flag
        iorwf TSTFLG,F
        btfsc TSTFLG,3  ;which line?
        goto LINE2      ;line 2

LINE1:  movf STORE1,W   ;modes for Screen line 1. Get switch settings
        ADDWF PCL,F     ;add program counter
        goto TIMSHW     ;0. show trip elapsed time - for kms flag
        goto SPEED      ;1. show trip kms speed
        goto PEAKSP     ;2. show trip kms peak speed
        goto TIMSHW     ;3. show trip elapsed time - for miles flag
        goto SPEED      ;4. show trip miles speed
        goto PEAKSP     ;5. show trip miles peak speed
        goto TRNSHW     ;6. show turns count
        goto RSETIT     ;7. reset routine

LINE2:  movf STORE1,W   ;modes for Screen line 2. Get switch settings
        ADDWF PCL,F     ;add program counter
        goto KMSSHW     ;0. show trip kms distance
        goto AVRAGE     ;1. show trip kms average speed
        goto TOTAL      ;2. show total absolute distance kms
        goto MILSHW     ;3. show trip miles distance
        goto AVRAGE     ;4. show trip miles average speed
        goto TOTAL      ;5. show total absolute distance miles
        goto TRNSH2     ;6. show TURN10 turns/10secs count
        goto RSETIT     ;7. reset routine

LCDOUT: movwf STORE1
        movf TSTFLG,W   ;get RS and store it in STORE2
        andlw %00100000 ;(RS = bit 5). NB: RS has also been set
        movwf STORE2    ;into PORT B prior to this subroutine
        movlw 10        ;set minimum time between sending full bytes
        movwf LOOPB     ;to LCD - value of 10 seems OK for a PIC
DELAY:  decfsz LOOPB,F  ;XTAL clk of upto 5MHz, possibly 5.5MHz
        goto DELAY
        call SENDIT     ;send MSB
        call SENDIT     ;send LSB
        RETURN

SENDIT:                 ;get and send data nibble
        swapf STORE1,F  ;get nibble
        rlf STORE1,W    ;rotate left as data goes to PORT B1-B4
        andlw %00011110
        iorwf STORE2,W  ;OR the RS bit
        movwf PORTB     ;output the byte
        BSF PORTB,6     ;set E high
        BCF PORTB,6     ;set E low
        return

TABINT:                 ;LCD initialisation table
        ADDWF PCL,F     ;add program counter
        retlw %00110011 ;initialise lcd - first byte
        retlw %00110011 ;2nd byte (repeat of first)
        retlw %00110010 ;set for 4-bit operation
        retlw %00101100 ;set for 2 lines
        retlw %00000110 ;set entry mode to increment each address
        retlw %00001100 ;set display on, cursor off, blink off
        retlw %00001100 ;dummy (repeat) to make 8 bytes being sent
        retlw %00000010 ;return home, cursor & RAM to zero
                        ;end inititalisation table

TABLE2:                 ;insert character table
        ADDWF PCL,F     ;add to program counter
        retlw 128       ;0. line 1     time
        retlw 86        ;1. line 1  V  speed - velocity
        retlw 80        ;2. line 1  P  peak
        retlw 128       ;3. line 1     time
        retlw 86        ;4. line 1  V  speed - velocity
        retlw 80        ;5. line 1  P  peak
        retlw 76        ;6. line 1  L  turns - Live
        retlw 32        ;7. line 1     reset
        retlw 68        ;8. line 2  D  distance
        retlw 65        ;9. line 2  A  average
        retlw 128       ;10 line 2     total
        retlw 68        ;11 lime 2  D  distance
        retlw 65        ;12 line 2  A  average
        retlw 128       ;13 line 2     total
        retlw 65        ;14 line 2  A  turns - Accumulative / 10 secs
        retlw 32        ;15 line 2     reset
                        ;128 is dummy to force K/M show

PWROFF: ADDWF PCL,F     ;table for reset message
        retlw 82        ;R
        retlw 69        ;E
        retlw 83        ;S
        retlw 69        ;E
        retlw 84        ;T
        retlw 33        ;!
        retlw 32        ;blank

                        ;next 2 commands should be reinstated if
                        ;diameter is to be set internally
                        ;delete the colon on each of them and ensure
                        ;that commands start fully at left hand side
                        ;(no blanks in front of them)
                        ;final commands at very end of program must
                        ;be deleted as stated there.
;SIZE1:  retlw 45        ;lsb wheel size 27.5 inches
;SIZE2:  retlw 56        ;msb wheel size 27.5 inches
                        ;above values to be changed to suit wheel size
                        ;see EPE text

TBDEC1:                 ;table for decimalisation lsb
        ADDWF PCL,F     ;add program counter
        retlw $10       ;lsb of 10000
        retlw $E8       ;lsb of 1000
        retlw $64       ;lsb of 100
        retlw $0A       ;lsb of 10

TBDEC2:                 ;table for decimalisation msb
        ADDWF PCL,F     ;add program counter
        retlw $27       ;msb of 10000
        retlw $03       ;msb of 1000
        retlw 0         ;msb of 100
        retlw 0         ;msb of 10

MILFLG:                 ;table for kms/miles flag
        ADDWF PCL,F     ;add program counter
        retlw 0
        retlw 0
        retlw 0
        retlw 16
        retlw 16
        retlw 16
        retlw 0
        retlw 0

CLKIN:                  ;increment clock routine
        decfsz CLKCNT,F ;is it = 0?
        RETURN          ;no
CLKADD: movlw 25        ;reset start value of CLKCNT
        movwf CLKCNT
        btfss RSTFLG,4  ;has reset been done?
        decf RSTFLG,F   ;no decrement reset flag

CLKAD2: movf TRNSEC,W   ;divide TRNSEC by 10
        btfsc STATUS,Z  ;is it =0
        goto CLKAD3     ;yes
        call CLRANS
        call CLRDEC
        movwf ANSA1
        movlw 10
        movwf DEC1
        call DIVIDE
        movf WORK1,W
        subwf TRNSEC,F

CLKAD3: movf TURNS,W
        addwf TRNSEC,F
        btfss STATUS,C  ;is there a carry ?
        goto SECCLK     ;no
        movlw 255       ;yes
        movwf TRNSEC    ;so set it to 255

SECCLK: movlw CLKSEC
        movwf FSR
        movlw 3
        movwf LOOPA

ADDCLK: incf INDF,F     ;inc units
        movlw 6
        addwf INDF,W    ;if 6 is added is there a digit carry?
        btfss STATUS,DC
        goto CHKH24     ;no
        movwf INDF      ;yes
        movlw 160
        addwf INDF,W    ;if 160 is added is there a carry
        btfss STATUS,C  ;(adding 160 checks if tens of units = 6)
        goto CHKH24     ;no
        clrf INDF       ;yes
        incf FSR,F
        decfsz LOOPA,F
        goto ADDCLK

CHKH24: movlw 220       ;is clock = 24 hrs? (adding 220 checks it)
        addwf CLKHRS,W
        btfsc STATUS,C
        clrf CLKHRS
CLKTEN: movf CLKSEC,W   ;check if tens of secs has occured
        andlw 15
        btfss STATUS,Z  ;is units = 0 ?
        goto DSTNCE     ;no
        movf TURN10,W   ;yes, copy TURN10 into TRNSEC store
        movwf TRNSEC
        call PRMPKS     ;get peak turns from eeprom
        movf TURN10,W   ;peak value in ANSA1
        subwf ANSA1,W   ;is TURN10 > PEAK
        btfss STATUS,C  ;is there a borrow
        call PKSPRM     ;yes, so update eeprom
SEC2:   clrf TURN10

DSTNCE: movf TURNS,W    ;calculate distance travelled
        btfsc STATUS,Z  ;are TURNS = 0 i.e is Z = 1?
        return          ;yes
        movwf STORE1    ;store TURNS
        clrf TURNS      ;clear TURNS

DIST2:  call SIZE1
        addwf TRNVAL,F
        btfss STATUS,C
        goto DIST3
        incf TRNVLM,F
        btfss STATUS,Z
        goto DIST3
        movlw 3
        movwf LOOPA
        movlw KMSBN1
        call BININC

DIST3:  call SIZE2
        addwf TRNVLM,F
        btfss STATUS,C
        goto DIST4
        movlw 3
        movwf LOOPA
        movlw KMSBN1
        call BININC

DIST4:  decfsz STORE1,F
        goto DIST2
        return

BININC: movwf FSR       ;binary increment routine
BIN2:   incf INDF,F
        btfss STATUS,Z
        goto ENDBIN
        incf FSR,F
        decfsz LOOPA,F
        goto BIN2
ENDBIN: return

DECIML:                  ;decimalise binary number
                         ;copy source into working area
                         ;source address is brought in on W
                         ;DEC1-5 becomes source
                         ;answer goes into ANSA
        call COPYFD      ;copy data at FSR into DEC
        clrf STORE1
        clrf LOOPA
        call CLRANS

        movlw ANSA4     ;set answer store address
        movwf FSR

DCML0:  movf LOOPA,W
        call TBDEC1     ;TBDEC1     ;subtract lsb from source value
        subwf DEC1,F
        btfsc STATUS,C  ;is there a borrow?
        goto DCML1      ;no
        movlw 1
        subwf DEC2,F    ;yes so decrement next byte
        btfsc STATUS,C  ;is there a borrow?
        goto DCML1      ;no
        movlw 1         ;yes
        subwf DEC3,F
        btfsc STATUS,C  ;is there a borrow?
        goto DCML1      ;no

DCML3:  movf LOOPA,W    ;yes, so re-add last table values lsb
        call TBDEC1
        addwf DEC1,F
        btfss STATUS,C  ;is there a carry?
        goto DCML5      ;no so exit loop
        incf DEC2,F     ;yes so inc next byte
        btfss STATUS,Z  ;is there a carry? (is it zero)
        goto DCML5      ;no so exit loop
        incf DEC3,F     ;yes
        goto DCML5      ;exit loop

DCML1:  movf LOOPA,W
        call TBDEC2     ;subtract msb from source value
        subwf DEC2,F
        btfsc STATUS,C  ;is there a borrow?
        goto DCML2      ;no
        movlw 1
        subwf DEC3,F    ;yes so decrement next byte
        btfss STATUS,C  ;is there a borrow?
        goto DCML4      ;yes

DCML2:  incfsz STORE1,F ;inc counter & continue looping till zero
        goto DCML0

DCML4:  movf LOOPA,W    ;re-add last table values lsb
        call TBDEC1
        addwf DEC1,F
        btfss STATUS,C  ;is there a carry?
        goto DCML4A     ;no so add msb
        incf DEC2,F     ;yes so inc next byte
        btfsc STATUS,Z  ;is there a carry? (is it zero)
        incf DEC3,F     ;yes, so inc next byte

DCML4A: movf LOOPA,W    ;re-add last table values msb
        call TBDEC2
        addwf DEC2,F
        btfsc STATUS,C  ;is there a carry?
        incf DEC3,F     ;yes, so inc next byte

DCML5:  movf STORE1,W   ;store counter and loop for next values
        movwf INDF
        clrf STORE1
        decf FSR,F
        incf LOOPA,F
        btfss LOOPA,2   ;is it = 4?
        goto DCML0      ;no

DCML6:  movlw 15        ;compact answers into 3 bytes
        andwf ANSA1,F
        andwf ANSA2,F
        andwf ANSA3,F
        swapf ANSA1,F
        movf DEC1,W
        iorwf ANSA1,F
        swapf ANSA3,W
        iorwf ANSA2,F
        movf ANSA4,W
        movwf ANSA3
        movlw 6
        addwf ANSA3,W
        btfsc STATUS,DC
        movwf ANSA3
        return

MILES:                  ;convert kilometres to miles
                        ;kms goes into DEC1-3
                        ;answer goes into ANSA1-5
        movlw KMSBN1
        call COPYFD     ;copy source into DEC
        call MILCNV     ;convert to miles. Source in DEC. A in ANSA
        return

MULTI:                  ;multiply routine. Source is in DEC1-5
                        ;multiplier is in MLTPL1 & MLTPL2
                        ;answer goes into ANSA1-5
        BCF STATUS,C    ;rotate divisor right by 1
        rrf MLTPL2,F
        rrf MLTPL1,F
        btfss STATUS,C  ;is carry = 1?
        goto MULT2      ;no
        movlw 5         ;yes
        movwf LOOPA     ;number of bytes to be added from
        movlw DEC1      ;set first source address
        movwf STORE1
        movlw ANSA1     ;set first answer address
        movwf STORE2

MULTB:  movf LOOPA,W
        movwf LOOPB     ;number of bytes for answer
        movf STORE1,W   ;get source address
        movwf FSR
        movf INDF,W     ;get source byte to be added
        movwf STORE3    ;temp store it
        movf STORE2,W   ;get first answer address
        movwf FSR
        movf STORE3,W   ;get stored source byte to be added
        addwf INDF,F    ;add it
        decfsz LOOPB,F
        call MLTINC     ;no, so increment next set of bytes as needed

MULTA:  incf STORE2,F   ;increment answer address
        incf STORE1,F   ;increment source address
        decfsz LOOPA,F  ;do same with next source byte
        goto MULTB

MULT2:  BCF STATUS,C    ;rotate left all source bytes
        rlf DEC1,F
        rlf DEC2,F
        rlf DEC3,F
        rlf DEC4,F
        rlf DEC5,F
        movf MLTPL1,W   ;is multiplier = 0?
        iorwf MLTPL2,W
        btfss STATUS,Z
        goto MULTI      ;no, loop back for next set
        return          ;yes

MLTINC: incf FSR,F
        movlw 1
        andwf STATUS,W
        addwf INDF,F
        decfsz LOOPB,F  ;yes
        goto MLTINC     ;if LOOPB > 0 then loop again
        return

DIVIDE:                 ;divide source val in ANSA1-4 by val in DEC1-3
                        ;result goes back into ANSA1-4
                        ;bit flag 1 is held in MLTPL1
                        ;byte flag 2 is held in MLTPL2
                        ;flag result total is in WORK1-5 (i.e. answer)
                        ;main loop held in MLTPL3

SETBYT: movf ANSA4,F    ;check if ANSA4 >0
        btfsc STATUS,Z
        goto BYTOK      ;it equals 0
        BCF STATUS,C    ;divide ANSA x 2
        rrf ANSA4,F
        rrf ANSA3,F
        rrf ANSA2,F
        rrf ANSA1,F
        BCF STATUS,C    ;divide DEC x 2
        rrf DEC4,F
        rrf DEC3,F
        rrf DEC2,F
        rrf DEC1,F
        goto SETBYT     ;repeat till ANSA4 = 0

BYTOK:  call CLRWRK
        clrf MLTPL1
        clrf MLTPL2
        clrf MLTPL3
        BSF MLTPL1,0    ;set flag
        movlw 3         ;set FLAGX (DEC5)
        movwf DEC5

DIV0:   movlw 3
        movwf LOOPA
        movlw DEC3      ;find 1st non-zero divide byte
        movwf FSR

DIV0A:  movf INDF,F     ;is byte = 0
        btfss STATUS,Z
        goto DIV0B      ;no
        decf FSR,F      ;yes, so look at next byte
        decf DEC5,F     ;dec FLAGX
        decfsz LOOPA,F  ;is loop = 0?
        goto DIV0A      ;no
        return          ;yes, divisor = 0 therefore return

DIV0B:  movlw 3
        movwf LOOPA
        clrf STORE1

DIV1:   BCF STATUS,C    ;rotate L divisor & flag until carryout from DEC3
        incf STORE1,F
        rlf MLTPL1,F
        btfss STATUS,C  ;is there a carry out?
        goto DIV2       ;no
        BSF MLTPL1,0    ;yes, so set bit 1
        decfsz LOOPA,F  ;is loop = 0
        goto DIV1A      ;no
        RETURN          ;yes, therefore divisor = 0, return

DIV1A:  movlw 3         ;set MLTPL2
        movwf MLTPL2

DIV2:   incf MLTPL3,F
        BCF STATUS,C
        rlf DEC1,F      ;rotate divisor
        rlf DEC2,F
        rlf DEC3,F
        btfss DEC3,7    ;is DEC3 >= 128?
        goto DIV1       ;no
        incf MLTPL3,F   ;yes
        movlw 3
        movwf MLTPL2
        goto DIV3A

DIV2A:  call DIV3       ;rotate right divisor by 1 place

DIV3A:  BCF TSTFLG,6    ;set flag for Subtract
        call DIV4

        movf ANSA4,F    ;has there been a borrow? (is ANSA4>0 ?)
        btfss STATUS,Z
        goto DIV3C      ;yes

        movlw WORK1     ;no, so add flag to total
        addwf MLTPL2,W  ;set answer address
        movwf FSR
        movf MLTPL1,W   ;OR flag with running answer
        iorwf INDF,F
        goto DIV3B

DIV3C:  BSF TSTFLG,6    ;set flag for Add
        call DIV4

DIV3B:  decf MLTPL3,F   ;is main counter = 0?
        btfsc STATUS,Z
        goto DIV7       ;yes, so correct for decimal point
        BCF STATUS,C
        goto DIV2A      ;no, so repeat

DIV7:   movf DEC5,F     ;is FLAGX = 0?
        btfsc STATUS,Z
        goto DIV7C      ;yes, so end division and return

DIV7A:  movlw 3         ;shift right WORK bytes by one place
        movwf LOOPA
        movlw WORK2
        movwf FSR

DIV7B:  movf INDF,W
        decf FSR,F
        movwf INDF
        incf FSR,F
        incf FSR,F
        decfsz LOOPA,F
        goto DIV7B
        clrf WORK4
        decfsz DEC5,F   ;if FLAGX <> 0 then repeat
        goto DIV7A
DIV7C:  return

DIV4:                   ;sub-routine for DIVIDE, plus ADD TOTAL
        movlw 3         ;set outer byte loop
        movwf LOOPA
        movlw ANSA1     ;set address for source
        movwf STORE1
        movlw DEC1      ;set address for divisor/adder
        movwf STORE2

DIV5:   movf LOOPA,W    ;set inner byte loop - 1 more than outer
        movwf LOOPB
        incf LOOPB,F
        movf STORE2,W   ;set address for divisor/adder byte
        movwf FSR
        movf INDF,W     ;get divisor/adder byte
        movwf STORE3    ;store it
        movf STORE1,W   ;set address for source
        movwf FSR

DIV5C:  movf STORE3,W   ;recall divisor byte
        btfsc TSTFLG,6  ;is Add needed?
        goto DIV5A      ;yes
        call SBTRCT     ;no
        goto DIV6
DIV5A:  call ADDIT

DIV6:   incf STORE1,F   ;inc source address
        incf STORE2,F   ;inc divisor address
        decfsz LOOPA,F  ;is outer loop = 0?
        goto DIV5       ;no, so continue loop
        return

SBTRCT: btfsc STATUS,Z  ;is byte zero?
        goto SBT2       ;yes
        subwf INDF,F
        btfsc STATUS,C  ;is there a borrow?
        goto SBT2       ;no
        incf FSR,F      ;yes
        movlw 1         ;so set up decrement for next byte
        decfsz LOOPB,F  ;is loop = 0?
        goto SBTRCT     ;no, so decrement next byte
SBT2:   RETURN          ;yes

ADDIT:  addwf INDF,F
        movf STATUS,W
        andlw 1
        incf FSR,F
        decfsz LOOPB,F  ;is loop > 0
        goto ADDIT      ;yes, so repeat for next byte
        RETURN

DIV3:   rrf DEC3,F      ;rotate right divisor by 1 place
        rrf DEC2,F      ;carry flag is preset by calling routine
        rrf DEC1,F

        BCF STATUS,C
        rrf MLTPL1,F    ;rotate flag right by 1 place
        btfss STATUS,C  ;is there a carry out?
        goto DIV3D      ;no
        decf MLTPL2,F   ;yes, dec flag 2
        BSF MLTPL1,7    ;and set bit 7 of flag 1
DIV3D:  return

DIVBY8: movlw 3         ;divide ANSA1-4 by 8
        movwf LOOPA

DIVx8A: BCF STATUS,C
        rrf ANSA4,F
        rrf ANSA3,F
        rrf ANSA2,F
        rrf ANSA1,F
        decfsz LOOPA,F
        goto DIVx8A
        return

PEAKSP: call PRMPKS     ;get peak speed from eeprom, A in ANSA1
        movf ANSA1,W
        goto HOWFAR
SPEED:  incf TRNSEC,W   ;is speed at max (TRNSEC=255)?
        btfsc STATUS,Z
        goto MAXSPD     ;yes
        movf TRNSEC,W   ;no
        goto HOWFAR

MAXSPD: call PREP3
        movlw 32
        movwf DEC2
        movwf DEC1
        movwf WORK4
        movlw 77        ;M
        movwf WORK3
        movlw 65        ;A
        movwf WORK2
        movlw 88        ;X
        movwf WORK1
        goto SCREEN

HOWFAR:                 ;calculate speed
        btfsc STATUS,Z  ;are PEAK TURNS = 0 i.e is Z = 1?
        goto ZERO       ;yes, therefore set all to zero for show
        call CLRDEC
        movwf MLTPL1    ;store PKTURN as multiplier LSB
        clrf MLTPL2     ;clear MSB
        call SIZE1      ;diameter lsb
        movwf DEC1      ;store as source byte 1
        call SIZE2      ;diameter msb
        movwf DEC2      ;byte 2

        call CLRANS
        call MULTI      ;multiply them
        call DIV256     ;divide by 256. Answer in DEC

        call CLRANS
        movlw $68       ;multiply by 360 to give rate per hour
        movwf MLTPL1    ;lsb (at 10-sec rate)
        movlw $01
        movwf MLTPL2    ;msb
        call MULTI      ;multiply them
        call DIV256     ;divide by 256. Answer in DEC

        movlw DEC1
        btfss TSTFLG,4  ;is flag for kms or miles?
        goto HOW2       ;kms
        call MILCNV     ;convert to miles. Source in DEC. A in ANSA
        movlw ANSA1
HOW2:   goto PREPRP     ;decimalise and prepare for screen

ZERO:   call CLRANS
        movlw ANSA1
        goto PREPRP

DIV256: movf ANSA2,W    ;divide answer by 256
        movwf DEC1
        movf ANSA3,W
        movwf DEC2
        movf ANSA4,W
        movwf DEC3
        movf ANSA5,W
        movwf DEC4
        clrf DEC5
        return

MILCNV: call CLRANS     ;source in DEC, answer in ANSA
        movlw 5         ;convert to miles
        movwf MLTPL1
        clrf MLTPL2
        call MULTI
        call DIVBY8
        movlw ANSA1
        return

TIMSHW:                 ;show TIME
        movlw CLKSEC    ;address of seconds
        call PRPSHW     ;prepare byte data
        movf DEC3,W
        movwf DEC4
        movf DEC2,W
        movwf DEC3
        movlw 58        ;colon
        movwf DEC2
        goto SCREEN

KMSSHW:                 ;show DISTANCE in KMS
        movlw KMSBN1    ;address 1st byte of KMS distance
        goto PREPRP     ;decimalise and prepare for screen

MILSHW: call MILES      ;show DISTANCE in MILES
        movlw ANSA1
        goto PREPRP     ;decimalise and prepare for screen

AVRAGE:                 ;get average speed
        call CLRWRK     ;clear WORK1-4
        call CLRANS     ;clear ANSA1-5

AV1:    movf KMSBN1,W   ;check if KMS =0
        iorwf KMSBN2,W
        iorwf KMSBN3,W
        btfsc STATUS,Z
        goto ZERO

        movf CLKHRS,W   ;check if time=0
        iorwf CLKMIN,W
        iorwf CLKSEC,W
        btfsc STATUS,Z
        goto ZERO

        movlw CLKHRS    ;convert hours to binary
        movwf FSR
        call CVBIN      ;answer in STORE1
        call CLRDEC
        movf STORE1,W
        movwf DEC1      ;source for divider
        call MLTx60     ;multiply hrs*60, answer in ANSA

        movlw CLKMIN    ;convert mins to binary
        movwf FSR
        call CVBIN      ;answer in STORE1
        call ADDTIM     ;add mins to hrs*60 (in ANSA), answer in DEC
        call MLTx60     ;multiply mins*60 (in DEC), answer in ANSA

CVSEC:  movlw CLKSEC    ;convert secs to binary
        movwf FSR
        call CVBIN      ;answer in STORE1
        call ADDTIM     ;add secs to min*60 (in ANSA), answer in DEC
                        ;answer = total secs of elapsed time

        movf DEC1,W     ;store secs answer in WORK
        movwf WORK1
        movf DEC2,W
        movwf WORK2
        movf DEC3,W
        movwf WORK3

KMSMLT:                 ;multiply kms * 3600
        movlw KMSBN1    ;copy kms into DEC as source to be multiplied
        call COPYFD
        movlw $10
        movwf MLTPL1
        movlw $0E
        movwf MLTPL2
        call CLRANS
        call MULTI      ;answer in ANSA

        movlw WORK1     ;copy stored secs answer back to DEC
        call COPYFD
        call DIVIDE     ;S in ANSA, Div in DEC, answer in WORK
        movlw WORK1

        btfss TSTFLG,4  ;is flag for kms or miles?
        goto PREPRP     ;kms
        call COPYFD     ;copy WORK into DEC
        call MILCNV     ;convert to miles. Source in DEC. A in ANSA
        movlw ANSA1
        goto PREPRP

CVBIN:  clrf STORE1     ;convert decimal byte to binary
        movf INDF,W
        btfsc STATUS,Z  ;is it = 0?
        goto CVB3       ;yes = 0
        andlw 15        ;no
        movwf STORE1    ;store units
        swapf INDF,W
        andlw 15
        btfsc STATUS,Z  ;is 10s = 0
        goto CVB3       ;yes
        movwf LOOPA
CVB2:   movlw 10
        addwf STORE1,F
        decfsz LOOPA,F
        goto CVB2
CVB3:   return

MLTx60: movlw 60        ;multiply * 60
        movwf MLTPL1
        clrf MLTPL2
        call CLRANS     ;clear ANSA
        call MULTI      ;answer into ANSA
        return

ADDTIM: movf STORE1,W   ;add STORE to ANSA
        addwf ANSA1,W   ;answer into DEC1-3
        movwf DEC1
        movf STATUS,W
        andlw 1
        addwf ANSA2,W
        movwf DEC2
        movf STATUS,W
        andlw 1
        addwf ANSA3,W
        movwf DEC3
        clrf DEC4
        clrf DEC5
        return

CLRWRK: clrf WORK1
        clrf WORK2
        clrf WORK3
        clrf WORK4
        return

CLRANS: clrf ANSA1
        clrf ANSA2
        clrf ANSA3
        clrf ANSA4
        clrf ANSA5
        return

CLRDEC: clrf DEC1
        clrf DEC2
        clrf DEC3
        clrf DEC4
        clrf DEC5
        return

PRMGET  movlw 10        ;get eeprom data for kms, time
        movwf LOOPA
        movlw 0
        movwf EEADR
        movlw TRNVAL
        goto GETPRM

PRMPKS: movlw 1         ;get eeprom data for peak speed
        movwf LOOPA
        movlw 10        ;eeprom address
        movwf EEADR
        movlw ANSA1     ;store for peak speed in turns/10 secs
        goto GETPRM

PRMTOT:                 ;get eeprom data for total kilometres
        movlw 3         ;for KMSBN1 to KMSBN3
        movwf LOOPA     ;but excluding TRNVAL & TRNVLM
        movlw 13        ;eeprom start address
        movwf EEADR
        movlw ANSA1     ;put it into ANSA1 address
        goto GETPRM

GETPRM: movwf FSR
GTPRM2: PAGE1           ;get EEPROM data
        BSF EECON1,RD   ;enable read
        PAGE0
        movf EEDATA,W   ;read data
        movwf INDF      ;store data for display
        incf EEADR,F    ;inc eeprom address
        incf FSR,F      ;inc data address
        decfsz LOOPA,F
        GOTO GTPRM2
        return

SETPRM: movlw 10        ;store current data in eeprom
        movwf LOOPA     ;from TRNVAL to time
        movlw 0         ;ie. kms, time and peak speed
        movwf EEADR     ;set eeprom address to 0
        movlw TRNVAL    ;start address
        goto BYTE1

PKSPRM: movlw 1         ;store peak speed in eeprom
        movwf LOOPA
        movlw 10
        movwf EEADR     ;set eeprom address to 10
        movlw TURN10    ;address
        goto BYTE1

TOTPRM: movlw 3         ;store TOTAL kms in eeprom
        movwf LOOPA
        movlw 13
        movwf EEADR     ;set eeprom address to 11
        movlw ANSA1     ;address total kms count after calc
        goto BYTE1

BYTE1:  movwf FSR
BYTE2:  PAGE1
        BSF EECON1,WREN ;enable write
        PAGE0
        movf INDF,W     ;get data value
        movwf EEDATA    ;set data

        PAGE1
        movlw $55       ;manual dictated factors
        movwf EECON2
        movlw $AA
        movwf EECON2
        BSF EECON1,WR

CHKWRT: BTFSS EECON1,4
        GOTO CHKWRT
        BCF EECON1,4    ;clear eeprom interupt flag
        PAGE0
        BCF INTCON,6
        incf FSR,F
        incf EEADR,F
        decfsz LOOPA,F
        GOTO BYTE2

        PAGE1
        BCF EECON1,WREN ;disable write
        BCF EECON1,4
        PAGE0
        BCF INTCON,6
        RETURN

PREPRP: call DECIML     ;convert to decimal. Answer in ANSA
        movlw ANSA1
        call PRPSHW
        goto SCREEN

PRPSHW: movwf FSR       ;prepare dual BCD data for single byte showing
        movwf STORE1
        movlw WORK2
        movwf STORE2

        movf RSTFLG,W
        andlw %11100000
        iorlw 10        ;reset RSTFLG (reset flag)
        movwf RSTFLG
        BSF PORTA,3     ;clear Reset alarm flag
        movlw 3
        movwf LOOPA

PREP2:  movf STORE1,W
        movwf FSR
        movf INDF,W
        movwf STORE3
        movf STORE2,W
        movwf FSR
        movf STORE3,W
        andlw 15
        iorlw 48        ;OR decimal 48
        movwf INDF
        incf STORE2,F
        incf FSR,F
        swapf STORE3,W
        andlw 15
        iorlw 48        ;OR decimal 48
        movwf INDF
        incf STORE2,F
        incf STORE1,F
        decfsz LOOPA,F
        goto PREP2

        movf WORK2,W
        movwf WORK1
        movf WORK3,W
        movwf WORK2
        movlw 46        ;decimal point
        movwf WORK3

PREP3:  movf TSTFLG,W
        andlw 15
        call TABLE2
        movwf STORE1
        btfss STORE1,7  ;is it 128?
        movwf DEC3
        movlw 75        ;kms
        btfsc TSTFLG,4  ;kms or miles?
        movlw 77        ;miles
        movwf DEC4
        return

SCREEN: movlw 8
        movwf LOOPA
        movlw WORK1
        addlw 7
        movwf FSR

SCRN2:  movf INDF,W
        call LCDOUT
        decf FSR,F
        decfsz LOOPA,F
        goto SCRN2

        BCF PORTB,5     ;clear RS line
        BCF TSTFLG,5    ;clear RS flag
        movlw 8
        addwf TSTFLG,W
        andlw 8
        BCF TSTFLG,3
        iorwf TSTFLG,F
        btfss TSTFLG,3  ;is line 2 needed?
        goto SCRN3      ;no
        movlw %11000000 ;set address for line 2 (address 64)
        call LCDOUT
        goto ENDSCR

SCRN3:  movlw %10000000
        call LCDOUT     ;set address for line 1 (address 0)
ENDSCR: return          ;return to INTRPT routine

TOTAL:                  ;copies KMSBN1-KMSBN3 into DEC1-3
                        ;then adds val in DEC1-3
                        ;to val in ANSA1-3
                        ;result goes back into ANSA1-3
        call GETTOT     ;get prev total from eeprom & add current
        btfss TSTFLG,4  ;is flag for kms or miles?
        goto TOTAL2     ;kms
        movlw ANSA1     ;miles
        call COPYFD     ;copy ANSA into DEC
        call MILCNV     ;convert to miles. Source in DEC. A in ANSA

TOTAL2: movlw ANSA1
        goto PREPRP

GETTOT: call CLRANS
        call CLRDEC
        call PRMTOT     ;get total distance from eeprom, store in ANSA
        movlw KMSBN1    ;copy KMS to DEC
        call COPYFD
        BSF TSTFLG,6    ;set flag for Add
        call DIV4       ;add them
        return

COPYFD: movwf FSR       ;copy data at FSR to DEC
        movf INDF,W     ;DO NOT CALL CLRDEC with this
        movwf DEC1      ;for reasons unknown it does not work then!
        incf FSR,F
        movf INDF,W
        movwf DEC2
        incf FSR,F
        movf INDF,W
        movwf DEC3
        clrf DEC4
        clrf DEC5
        return

RESETB: movlw $20
        movwf LOOPA
        movlw $0D
        movwf FSR

RESETC: clrf INDF
        incf FSR,F
        decfsz LOOPA,F
        goto RESETC
        return

STORIT: call SETPRM     ;store kms, time in eeprom at power down
        BCF PORTA,3     ;turn on buzzer
ENDSTR: call NEGTIV     ;at power down this loop is held indefinitely
        goto ENDSTR

RSETIT: movlw 6         ;reset time, dist, speed, peak, average
        movwf STORE1    ;(if time-out count-down reaches zero)
        movf RSTFLG,W
        andlw 15
        iorlw 48
        movwf WORK1
        BCF PORTA,3     ;set Reset alarm output
        movlw WORK2
        movwf FSR
        movlw 7
        movwf LOOPA
STOR3:  movf STORE1,W
        call PWROFF
        movwf INDF
        decf STORE1,F
        incf FSR,F
        decfsz LOOPA,F
        goto STOR3

        movf RSTFLG,W
        andlw 31
        btfss STATUS,Z  ;is reset flag = 0?
        goto SCREEN     ;no, so show Reset count
        BSF RSTFLG,4    ;yes, set flag that reset has been done
        call GETTOT     ;get total, answer is in ANSA
        call TOTPRM     ;store it in eeprom
        movlw 10
        movwf LOOPA
        movlw CLKHRS
        movwf FSR

STOR5:  clrf INDF
        decf FSR,F
        decfsz LOOPA,F
        goto STOR5
        clrf TURN10
        call PKSPRM     ;reset peak ... new at 7JUL96
        clrf TRNSEC
        movlw 25
        movwf CLKCNT
        return          ;(do NOT call SCREEN at Reset - data wrong)

RSTPRM:                 ;total reset eeprom. Switch S4 on at start
        call RESETB     ;reset all working bytes
        call SETPRM     ;reset eeprom kms, time
        call PKSPRM     ;reset peak
        call TOTPRM     ;reset totals

WAITIT: btfss PORTA,4   ;is reset switch still low?
        goto WAITIT     ;yes
        return

TRNSHW:                 ;show turns count
        movf TURNS,W    ;real time turns count
        goto TRS2
TRNSH2: movf TRNSEC,W   ;average turns per 10 secs
TRS2:   call CLRANS
        movwf ANSA1
        movlw ANSA1
        call DECIML
        movlw ANSA1
        call PRPSHW
        movf WORK4,W
        movwf WORK3
        movlw 32
        movwf WORK4
        movwf DEC1
        movwf DEC2
        movlw 84        ;T
        movwf DEC4
        goto SCREEN

NEGTIV: movlw 2         ;toggle PORTB,1 for LCD -VE generation
        addwf PORTB,W
        andlw 2
        BCF PORTB,1
        iorwf PORTB,F
        return

                        ;The following routines are for use when wheel
                        ;diameter is to be programmed externally
                        ;they must be deleted if fixed diameter
                        ;is set at about line 280 - see EPE text
SIZE2:  movlw 20        ;get eeprom data at loc 20 for wheel size MSB
        goto SIZEIT
SIZE1:  movlw 21        ;get eeprom data at loc 21 for wheel size LSB
SIZEIT: movwf EEADR
GTSIZE: PAGE1           ;get EEPROM data
        BSF EECON1,RD   ;enable read
        PAGE0
        movf EEDATA,W   ;read data
        return

WHEEL:  movlw 1         ;store diameter byte in eeprom
        movwf LOOPA
        movf STORE1,W
        addlw 20
        movwf EEADR     ;set eeprom address to 20
        movlw TRNSEC    ;address TURNS average per 10 secs
        goto BYTE1      ;store it (only locs 20/21 will actually be used)

                        ;end of deletion

        .END            ;final line. This line must be retained.
