;MHB
;19-7-02
;18-8-02
;4-10-02
;		list n=90,t=on
;*** FIDO.ASM  *****
;Version1.6
;Author: M H Boyden
;
;                  *****************************************
;		   ****** FIDO - A low cost pedometer ******
;		   *****************************************

;Notes:
;To conserve power, 39ms debounce is used to flash decending bars.
;Learn bleeps reduced to 39ms.
;Vertical position bar is flashed using a 4ms delay and a 255 counter (temp)




;This code, in conjunction with the hardware specified, monitors
;the distance moved by a walker or runner and also calculates average speed.    

;****** SOFTWARE SUMMARY *******
;Two mercury switches are used to detect motion. Backwards motion is
;regarded as invalid. 
;
;The pcb contains two mercury switches and these are arranged to detect
;accelerations every time the leg moves. The leg forward sensor
;is connected to ra0, and the leg backward sensor is connected to ra1. 
;The sensors are active lo. The unit MUST be worn on the LEFT leg, with
;the display facing upwards for viewing.
;
;The truth table for the sensors is as follows:
; 	Rear	   	Front 	 	
; 	Sensor(ra1)   Sensor(ra0) 	Remarks
;
;	0		0		invalid - jumping?
;	1		0		leg moving forward
;	0		1		leg moving backward
;	1		1		leg vertical			
;

;Only the following sequence of detection permits
;a valid 'step' to be recorded.
;
;1. read ra1, ra0 - wait for leg vertical ie 11
;2. wait for leg forward detection ie 10
;3. signal 'leg forward'  ie single flash  top bar of display
;4. wait for leg vertical ie 11
;5. signal 'leg vertical' ie continious flash  middle bar of display
;6. wait for leg back ie 01
;7. signal 'leg back' ie single flash  bottom bar of display
;7. return to 2

;Thus to indicate that the sensors are working correctly, the upper, middle and lower
;bars of the led will flash in turn, indicating that a valid walk cycle is being
;correctly sensed. Adjust the position of the tilt switches until a smooth movment of the
;leds from top to bottom is observed.
  
;Depending upon calibration, the unit can display distance in miles or km
;and average speed in miles per hour or km per hour.
;The selection of the type of distance measurement is defined in the 'Learn Mode'
;and this explained in more detail later. 

;A single digit displays in the format 99.9 units.
;(The decimal point is displayed by the 7 segment middle segment).
;(If the value of tens is zero, then this digit is not displayed.
;  
;The mode is set by sw2 but only when the unit is verticle ie middle bar is flashing. 
;This is a three position switch (one side sprung) 
; a. display (P)rogress (in distance units) and then (A)verage speed (in units per hour).
; b. measure the distance travelled
; c. learn and store the number of strides the user has taken to move 1 mile or km.  

;a. Display Mode (spring loaded to centre return toggle)
;The display will look like: (pause) P3.2 (pause) A4.1 
;The unit cannot measure movement whilst it is in the display mode.

;b. Measure (normally off position)
;With sw1 in the normally off position (sw1b) the unit counts strides, calculates
;distance and average speed. The 'decending bars' shows that valid walking cycles
;are being counted.

;c. Learn Mode (not spring loaded)
;It is necesssary for the unit to know the number of steps that the wearer will
;make to complete a mile or kilometer.   
;When sw1c is enabled, the unit will commence recording the number of steps that the
;wearer is taking to walk either 1 kilometer or 1 mile. If imperial units (miles)
;are required then walk 1 mile, if a metric readout is required, then walk 1 kilometer.
;If Fido is to record walking then the calibration distance should be walked. If
;measurement is to be made for running, then the calibration distance should be run.  
;
;The letter L is displayed after every walkcycle to indicate that the unit is in 
;learn mode, also accompanied by a short tone from the sounder. When 255 steps
;have been recorded each step is accompanied by a double bleep, indicating that 
;the steps are now being recorded. 

;After walking the 'calibration distance', sw1 is turned from sw1c to sw1b and a few 
;more steps are taken.
;The unit will store the stride constant into EEPROM where it is
;stored indefinatley. If the unit is to be recalibrated or is to be used with a new person
;then the procedure outlined above is repeated.

;A default value is included to allow the unit to function for demonstration purposes,
;but the display may be inaccurate. For the author, the approximate number
;of strides to walk 1 mile is 1140D ie constant is 114D or 72H. To reduce the possibility
;that sw1c is selected by accident, the EEPROM data is updated only after 255 strides
;have been recoreded.

;A 7 seg led is attached on port rb0 to rb5 and ra4.
;In-circuit programming is carried out via pins rb6, rb7. A buzzer is attached between
;rb7 and ground, by means of a plug.
;
;The PIC16F84-20 runs at 20MHz.
;
  INCLUDE "p16c84.inc"
		list p=16c84,n=90,t=on
; PIC is a type 16F84-20 running at 20MHz
 __CONFIG H'3FF2'

#define 	page0   bcf status,5
#define 	page1   bsf status,5

;vectors
		org	H'0'	;cold start for 16f84
		goto	start	;vector to start
		org	H'4'	;interrupt vector
		goto	pulse	;service an interval

;equates
porta		equ	H'5'	;port a data reg
portb		equ	H'6'	;port b
z		equ	D'2'	;the status zero flag bit (Z)
c		equ	H'0'	;the carry flag bit (c)
carry		equ	H'0'
status		equ	H'03'	;the status register
intcon		equ	H'0B'	;interrupt resister
gie		equ	H'7'	;gie is bit 7 in the interrupt reg
blank		equ	D'16'	;offset to blank display
option		equ	H'81'	;option register
pc		equ	H'02'	;program counter register
pclath		equ	H'0A'	;high prog ctr byte

EEDATA 		equ H'08'
eecon1 		equ H'08'
EEADR  		equ H'09'
eecon2 		equ H'09'
RD     		equ 0  		;eeprom read enable flag
WR     		equ 1  		;eeprom write initiate flag
WREN   		equ 2  		;eeprom write enable flag
eestart 	equ H'0'        ;start address for eeprom memory

;load EEPROM stride constant default
		org	H'2100'	;eeprom memory 
		de	H'72'	;stride const 0.1mile walking 1140/10=114D(72H) 
		dw	H'0'
		de	"EPE Pedometer V1.6 - M Boyden 2002"

;define storage space
		org	H'0C'	;spaced from working area
temp1		res	1	;delay storage
temp2		res	1	;delay storage
w_sav		res	1	;temp w reg
s_sav		res	1	;temp status 
seconds		res 	1	;seconds counter
minutes		res	1	;minutes counter

tempdis		res	1	;temp display storage

distlo		res	1	;up to 0.6 mile counter lo byte
disthi		res	1	;up tp 0.6 mile counter hi byte
msec		res	1	;milisecond counter

dist_tens	res	1	;distance for display tens
dist_units	res	1	;distance for display units
dist_tenths	res	1	;display for display tenths

spd_tens	res	1	;average speed for display tenths	
spd_units	res	1	;avarage speed for display units
spd_tenths	res	1	;avarage speed for display tenths
Tdisthi		res	1	;Total number of 0.6 miles distance hi byte
Tdistlo		res	1	;Total number of 0.6 miles distance lo byte 
Ttimehi		res	1	;Total number of 0.6 hour hi byte counter
Ttimelo		res	1	;Total number of 0.6 hour lo byte counter 
tempee		res	1	;temporary eeprom storage

Tdisthi1	res	1	;Total distance snapshot hi byte
Tdistlo1	res	1	;Total distance snapshot lo byte 


;Mathematical storage for 16 bit variables  
ACCaLO 		res 	1
ACCaHI  	res    	1
ACCbLO  	res    	1
ACCbHI  	res    	1
ACCcLO  	res     1
ACCcHI  	res     1
ACCdLO  	res     1
ACCdHI  	res     1
ACCaLO1		res	1
ACCaHI1		res	1
ACCbLO1		res	1
ACCbHI1		res	1
temp    	res     1		;maths bit loop counter


;****** START ******
;clear downs, interrupt enables, and port assignments
start
		clrf	msec		;clear 10ms counter
		clrf	porta		;clear for good practice
		clrf	portb		;clear for good practice
		clrf	seconds		;clear elapsed seconds store 
		clrf	minutes		;clear elapsed minutes store

		clrf	distlo		;below .6 mile strides counter hi byte
		clrf	disthi		;below .6 mile strides counter lo byte
		clrf	dist_tens
		clrf	dist_units	;miles travelled units
		clrf	dist_tenths	;miles travelled tenths
		clrf	spd_tens
		clrf	spd_units	;average travel time miles/hr units
		clrf	spd_tenths	;average travel time miles/hr tenths

		clrf	Tdisthi		
		clrf	Tdistlo		
		clrf	Ttimehi		
		clrf	Ttimelo		
		clrf	tempee		


;Clear down the arithmetic storage
		call	clrmaths

;****** SETUP PORTS ******
;set up port a
;ra0=leg forward sensor ip
;ra1=leg back sensor ip
;ra4=display drive, ra3,2=set function buttons
		movlw	B'00001111'	;a0 to a3 set to i/p
		tris	porta		;send it
		movlw	H'ff'		;set display off
		movwf	porta		;

;set up port b,
;b0 to b5=drives 7 segment display
;b7 = buzzer +ve terminal, -ve terminal grounded
;b7,6=are also used for in circuit programing. 
		movlw	B'00000000'	;all o/p
		tris	portb		;send it
		movlw	B'01111111'	;set display and buzzer off
		movwf	portb		;

;****** SETUP TMR0 ******
;setup TMR0 to interrupt every 10ms
;set RTCC to divide by 256, so T(interrupt)=0.2x256xn 
;so for T(interrupt required)=10ms=0.2x256xn, n=195.3125
;so, T(interrupt actual)=195x.2x256=9984us
;so an undertiming of -16us in 10ms gives an error of -5.7s per hour.
 		movlw  	B'00000111'     ;set to internal clk, divide by 256
		option
  		movlw  	.256-.195       ;interrupt every 9984us
 		movwf  	TMR0

;****** SETUP INTERRUPTS ******
;Disable TMR0 and global interrupt flags
			bcf	intcon,0	;reset rb4 to rb7 flag
			bcf	intcon,1	;reset rb0 interrupt flag
			bcf	intcon,2	;reset TMR0 flag
			bcf	intcon,3	;disable rb4 to rb7 interrupt
			bcf	intcon,4	;disable rb0 interrupt function
			bcf	intcon,5	;disable TMR0 interrupt
;EEPROM			bcf	intcon,6	;EEPROM stuff
			bcf	intcon,7	;disable all interrupts

;****** MAINLINE SETUPS *******
;Clear or set variables and ensure that the leg is vertical 
;before starting
mainsetup	call	kill		;blank display

		clrf	distlo		;clear down store
		clrf	disthi
		clrf	Tdistlo
		clrf	Tdisthi
		clrf	Ttimelo
		clrf	Ttimehi

;Check that the leg is vertical ie ra0=1,ra1=1
lop1		btfss	porta,0		;front sensor hi?
		goto	lop1
		call	del39ms		;
		btfss	porta,1		;yes, and back sensor hi?
		goto	lop1		;no, keep trying

		movlw	D'17'		;light middle bar
		call	display	
		call	del117ms	; 
		call	kill

;signal ready to start
		bsf	portb,7		;buzzer on
		call	blipdel
		bcf	portb,7		;buzzer off

;ready to start, so enable 10ms interrupt clock
		bsf	intcon,5	;enable TMRO
		bsf	intcon,7	;enable global

;**********************************
;****** MAINLINE STARTS HERE ******
;**********************************

;
mainline	nop

;Check the mode switch status and divert to the appropriate module
;Bits ra3 (sw1c) and ra2 (sw1a) define the mode
;Learn Mode test SW1c - Learn mode or continue to just measure?
sw1c		btfss	porta,3		;down=1, up=0
		call	modules3	;up - learn mode
		btfss	porta,3		;down - test sw again
		goto	sw1c		;up - learn endlessly
					;down - measure
;Measure strides Mode
;count strides, also check if sw1 is selecting Display Mode and if it is:
;1. calculate distance (constantly done in mainline), display distance
;2. calculate average speed and display
;check for a normal stride cycle and increment stride counter disthi, distlo 
		call	legcycle	;check for a valid stride and bump

;if disthi/lo = 0.6, bump total miles and calculate distance in readiness for
;display - dist_tens,units,tenths
		call	cal_dist	;bump (Tdisthi,Tdistlo) 

		goto	mainline

;********************************
;****** MAINLINE ENDS HERE ******
;********************************


;****** MODULES2 - DISPLAY DISTANCE AND SPEED ******
;software functions triggered by sw2 - display distance and average speed or goto
;sw3 for further functions
modules2	nop			;select the module of code to execute

		movlw	D'22'		;P for Progress
		call	display
		call	longdel	
		call	dispdist	;display distance as 99.9 units
	
		call	kill		;blank
		call	longdel

		movlw	D'21'		;A for Average Speed
		call	display
		call	longdel	
		call	dispavsp	;display average speed as 99.9 units/hr
		call	shortdel	;interval between display cycles
		return

;****** MODULES3 - LEARN AND STORE STRIDES *****
;software functions triggered by sw3 - learn the number of strides for a mile or
;complete a normal stride count for distance and speed function above
modules3	call	legcycle	;check for a valid stride, bump disthi/lo

;show L for learn and buzz after each valid stride
		movlw	D'18'		;L for Learn
		call	display
		bsf	portb,7		;buzz after each step, buzzer on
		call	del39ms
		bcf	portb,7		;buzzer off
		
;Only write to the EEPROM after 256 strides
		movf	disthi,w	;fetch stride counter hi byte
		sublw	D'01'		;1H - W
		btfss	status,c	;disthi< 256D?
		goto	weprom		;no, divide and write to EEPROM
		return			;yes, return and leave EEPROM alone

;more than 256 steps, so compute 1/10 mile stride division factor
;divide the distance by 10 and write result to EEPROM
weprom		nop
		call	del39ms
		bsf	portb,7		;buzz again
		call	del39ms
		bcf	portb,7		;buzzer off

		call	clrmaths
		movf	disthi,w	;load distance into numerator
		movwf	ACCbHI
		movf	distlo,w
		movwf	ACCbLO

		clrf	ACCaHI		;load denominator = 10
		movlw	D'10'
		movwf	ACCaLO

		call	D_div		;result in ACCbLO

		movf	ACCbLO,w
		movwf	tempee		;load eeprom data
		movlw	D'0'		;point to eeprom target address
		call	setmem		;write to eeprom
		return

;*****************************************************
;****** MAINLINE AND MODULE SWITCHING ENDS HERE ******
;*****************************************************



;****** LEGCYCLE ******
;Check that the leg has gone through a valid walking or running cycle
;To check that the unit is correctly positioned and walking cycles are
;being correctly sensed, the top, middle and bottom bars of the 7 segment
;display are lighted to indicate that the motion has been detected ok.
;Check that leg is forward ie ra1=1,ra0=0 ; active=lo
legcycle	nop
lop2		btfsc	porta,0		;front sensor lo?
		goto	lop2
		movlw	D'20'		;light top bar
		call	display		;show a vaild position and,
		call	del39ms		;debounce 
		call	kill		;turn off display
		btfss	porta,1		;yes, and back sensor hi?
		goto	lop2		;no, keep trying
		call	del117ms

;Check that leg is vertical again ie ra0=1,ra1=1
lop3		btfss	porta,0		;front sensor hi?
		goto	lop3
		call	del39ms		;
		btfss	porta,1		;yes, and back sensor hi?
		goto	lop3		;no, keep trying
;		call	del117ms

;whilst vertical, check if display mode is enabled (ra2=0) on sw1a
;MODES SW1a - Display distance and speed or continue to measure
		clrf	temp		;clear the loop counter
sw1a		btfss	porta,2		;enabled=0, disabled=1
		call	modules2	;0 - display mode; distance and speed
		btfss	porta,2		;test sw again
		goto	sw1a		;loop endlessly

;flash middle bar at about 1Hz, whilst waiting for movement or display selection
		incf	temp
		call	del4ms		
		btfss	temp,7
		goto	baron		;keep bar on until temp=D128	
		goto	baroff		;keep bar off until temp=D256
baron		movlw	D'17'		;light middle bar
		call	display	
		goto	barexit
baroff		call	kill 		;turn off middle bar
barexit		nop


;Check that leg is backward ie ra0=1,ra1=0
;lop4		btfss	porta,0		;front sensor hi?
;		goto	lop4		;no, cycle again


;ra0 must still be hi, go and recheck cycle if it is not
		btfss	porta,0
		goto	sw1a
		btfsc	porta,1		;back sensor ra1=0?
		goto	sw1a		;no, cycle again
		movlw	D'19'		;flash bottom bar
		call	display	
		call	del39ms		;
		call	kill
;		call	del117ms


;A vaild walking or running cycle has now been detected
;Increment the 16 bit stride counter
		incf	distlo,f	;bump the stridectr lo byte
		btfsc	status,z	;and the hi byte 
		incf	disthi,f	;on every 256 rollover
		return

;****** DISPLAY ******
;Display the value or sign held in wreg
display		movwf	tempdis		;save the display character
;		bsf	pclath,0	;make sure we return to page 0
		call	l_upb		;get the port b pattern
		movwf	portb		;and send it
		movf	tempdis,w	;fetch the display character
;		bsf	pclath,0	;make sure we return to page 0
		call	l_upa		;get the port a pattern
		movwf	porta		;and send it
		return			


;****** L_UPB, L_UPA - 7 SEGMENT LOOKUP TABLES ******
;look up the 7 segment pattern to display a character.
;The value to be displayed is in the W register, which is added
;to the program counter, to create an offset. This is used to 
;fetch the required segment illumination pattern. 
;A common anode display is used and the segments are mapped below. 
;
;Enter with the wreg containing the value (or sign) to be displayed 

;Look up for the segments connected to port b0 to b5
; - b,c,d,e,f,g (start at bottom segment (a), going anticlock) 
l_upb
		addwf	pc,f		;add W to the value of the
					;program counter 
;0=on,1=off
;			  **543210	;RB port assignment
;	                    defcgb	;common anode segments
		retlw	B'00000010'	;0
		retlw	B'00111010'	;1
		retlw	B'00010001'	;2
		retlw	B'00011000'	;3	
		retlw	B'00101000'	;4
		retlw	B'00001100'	;5
		retlw	B'00000100'	;6
		retlw	B'00011010'	;7
		retlw	B'00000000'	;8
		retlw	B'00001000'	;9

		retlw	B'00000000'	;A
		retlw	B'00100100'	;B
		retlw	B'00000111'	;C
		retlw	B'00110000'	;D	
		retlw	B'00000101'	;E
		retlw	B'00000101'	;F

		retlw	B'00111111'	;16 - turn off display
		retlw	B'00111101'	;17 - middle bar (dash sign)
		retlw	B'00100111'	;18 - L for learn
		retlw	B'00111111'	;19 - lower bar (rear sensor)
		retlw	B'00011111'	;20 - upper bar (forward sensor)
		retlw	B'00000000'	;21 - A for Average speed
		retlw	B'00000001'	;22 - P for Progress 

;Look up for the segment connected to port a4 
; - a (bottom segment) 
l_upa
		addwf	pc,f		;add W to the value of the
					;program counter 
;0=on,1=off
;			  00043210	;RA port assignment
;	                     a		;common anode segments
		retlw	B'00000000'	;0
		retlw	B'00010000'	;1
		retlw	B'00000000'	;2
		retlw	B'00000000'	;3	
		retlw	B'00010000'	;4
		retlw	B'00000000'	;5
		retlw	B'00000000'	;6
		retlw	B'00010000'	;7
		retlw	B'00000000'	;8
		retlw	B'00000000'	;9

		retlw	B'00010000'	;A
		retlw	B'00000000'	;B
		retlw	B'00000000'	;C
		retlw	B'00000000'	;D	
		retlw	B'00000000'	;E
		retlw	B'00010000'	;F

		retlw	B'00010000'	;16 - turn off display
		retlw	B'00010000'	;17 - middle bar (dash sign)
		retlw	B'00000000'	;18 - L for learn
		retlw	B'00000000'	;19 - lower bar (rear sensor)
		retlw	B'00010000'	;20 - upper bar (forward sensor)
		retlw	B'00010000'	;21 - A for Average speed
		retlw	B'00010000'	;22 - P for Progress


;****** PULSE - INTERRUPT SERVICE ROUTINE ******
;interrupt service routine 
;Come here every TMR0 rollover which is every 10ms.
;The interrupt is used as a timer for speed calculations - max timed period is 
;equal to Ttimehi, Ttimelo (max $FFFF, 65535D) * 6 minutes = 6553 hrs = 270 days
;save status information
pulse		bcf	intcon,7	;quickly disable further interrupts
		bcf	intcon,5	;disable TMR0 counter drive 
		movwf	w_sav		;save the w reg
		swapf	status,w	;save zero flag
		movwf	s_sav		;save the status reg

;reset TMR0 counter
		movlw  	.256-.195       ;actual delay is 9984us
 		movwf  	TMR0		;ie 16us too early

		bcf	intcon,2	;reset TMR0 interrupt flag
		bsf	intcon,5	;enable TMR0 counter drive

;increment the 1 sec counter
		incf	msec,f		;bump the ms counter
		movlw	D'100'		;100x10ms in a second
		subwf	msec,w		;subtract 100 from ms counter
		btfss	status,z	;skip if result 0 ie 100x10msecs
		goto	pulse_exit	;not 1 sec yet 

;increment the minutes counter
		clrf	msec		;reset 10ms counter
		incf	seconds,f	;bump the seconds counter
		movlw	D'60'		;60 seconds in a minute
		subwf	seconds,w	;subtract 60 from seconds
		btfss	status,z	;skip if result 0 ie 60 secs
		goto	pulse_exit	;not 60 sec yet

;increment the 6 minutes counter 
;TEST - THIS WILL BUMP THE TOTAL DISTANCE COUNTER every minute
;		incf	Tdistlo,f	;*TESTING ONLY
		clrf	seconds		;reset the seconds counter
		incf	minutes,f	;bump minute counter
		movlw	D'6'		;6 minutes
		subwf	minutes,w
		btfss	status,z
		goto	pulse_exit	;not 6 mins yet

;increment the total elapsed time counter (6mins*65000!)
		clrf	minutes
		incf	Ttimelo,f	;bump the accumulative time lo byte
		btfsc	status,z	;and the hi byte 
		incf	Ttimehi,f	;on every 256 rollover

;take and store a snapshot of the distance for average speed calculation  
		movf	Tdistlo,w
		movwf	Tdistlo1
		movf	Tdisthi,w
		movwf	Tdisthi1

pulse_exit	bsf	intcon,7	;enable interrupts again
;restore register information
		swapf	s_sav,w		;restore status flag
		movwf	status
		swapf	w_sav,f		;restore w register
		swapf	w_sav,w
		retfie			;return 

;****** CAL_DIST ******
;Calculate the Distance travelled.
;A 16 bit distance counter is subtracted from the stride constant held in EEPROM. The stride
;constant is set to either the default (H72), or to a new value following a 'learning'
;walk (or run) of 1 mile. 
;When 0.1 mile is detected, three counters are incremented and used to display distance
;in tens, units and tenths of miles. 
;The end of the routine checks if 99.9 miles have been reached, whereupon
;the counters are reset.
;There are several possible returns from this module

;check for a tenth of a mile
cal_dist	nop
		movlw	H'0'		;point to eeprom address
		call	getmem		;get stride division constant
;		movlw	H'72'		;** test only 0.1m stride const
		subwf	distlo,w	;subtract sconst from distance
		btfss	status,c	;skip if result positive
		return			;not 0.1 mile yet
;bump the accumalative 16 bit (0.1) mile counter
		incf	Tdistlo,f	;lo byte bump
		btfsc	status,z
		incf	Tdisthi,f	;hi byte bump
;increment tenths counter
		clrf	distlo		;clear the distance counter
		incf	dist_tenths,f	;bump the tenths
		movlw	D'10'		;check for >10 tenths
		subwf	dist_tenths,w	;subtract 10 from tenths
		btfss	status,c	;skip if result positive
		return			;not 10 tenths yet
;increment units counter
		clrf	distlo		;reset the distance counter
		clrf	dist_tenths	;reset the tenths counter
		incf	dist_units,f	;bump the units counter

;indicate that another mile or km has been travelled
		bsf	portb,7		;buzzer on
		call	longdel
		bcf	portb,7		;buzzer off

		movlw	D'10'		;check for >10 units
		subwf	dist_units,w	;subtract 10 from tenths
		btfss	status,c	;skip if result positive
		return			;not 10 units yet
;increment tens counter
		clrf	distlo		;reset the distancs counter
		clrf	dist_tenths	;reset the tenths counter
		clrf	dist_units	;reset units counter
		incf	dist_tens,f	;bump the elapsed hours counter
		movlw	D'10'		;check for >10 units
		subwf	dist_tens,w	;subtract 10 from tenths
		btfss	status,c	;skip if result positive
		return			;not 10 tens yet

;reached 99.9 miles so reset pedometer 
rset_dist	clrf	dist_tenths	;reset display distance counters
		clrf	dist_units	;
		clrf	dist_tens	
		clrf	spd_tenths	;reset display time counters
		clrf	spd_units	
		clrf	spd_tens
		clrf	Tdisthi		;reset distance counters
		clrf	Tdistlo
		clrf	Ttimehi
		clrf	Ttimelo
		return	

;****** DISPDIST ******
;Display the distance as 99.9 miles
;Leading zeros are not displayed
dispdist	movf	dist_tens,w	;display tens of miles,
		btfsc	status,z	;if above zero
		goto	dispunits
		call	display		;display digit
		call	shortdel

		call	kill		;blank display
		call	shortdel

dispunits	movf	dist_units,w	;display units of miles,
;		btfsc	status,z	;if above zero
;		goto	disptenths
		call	display		
		call	shortdel

		call	kill
		call	shortdel

disptenths	movlw	D'17'		;decimal point (dash)
		call	display
		call	shortdel
	
		movf	dist_tenths,w	;display tenths of mile
		call	display
		call	shortdel

		call	kill		;pause between display cycles
		call	shortdel
		call	shortdel
		call	shortdel
		return


;****** DISPSPEED ******
;Display the avarage speed 99.9 miles/hr
;Leading zeros are not displayed
dispspeed	movf	spd_tens,w	;display tens of miles,
		btfsc	status,z	;if above zero
		goto	dspunits
		call	display		;display digit
		call	shortdel

		call	kill		;blank display
		call	shortdel

dspunits	movf	spd_units,w	;display units of miles,
;		btfsc	status,z	;if above zero
;		goto	dsptenths
		call	display		
		call	shortdel

		call	kill
		call	shortdel

dsptenths	movlw	D'17'		;decimal point (dash)
		call	display
		call	shortdel
	
		movf	spd_tenths,w	;display tenths of mile
		call	display
		call	shortdel

		call	kill		;pause between display cycles
		call	shortdel
		return

;****** KILL ******
;turn off display
kill		movlw	blank	;set blank offset
		call	display	;send to display
		return

;****** LONGDEL ******
;Delay about 1.56 seconds at 20MHz
longdel		call	shortdel	
		call	shortdel
		call	shortdel
		call	shortdel
		return

;****** SHORTDEL ******
;Delay about .39 sec at 20MHz
shortdel	call	delay
		call	delay
		call	delay
		call	delay
		call	delay
blipdel		call	delay
		call	delay
del117ms	call	delay		;117 ms
		call	delay
del39ms		call	delay
		return
	
;****** DELAY (39ms and 4 ms)******
;Tcycle=0.2us
;delay=3x.2x65536=39ms
;del4ms=26x153=3.9ms
del4ms		movlw	D'26'		;26x153=3.9ms
		movwf	temp1
		goto	del1
delay		movlw	H'ff'
		movwf	temp1
del1		movlw	H'ff'		;3x.2x255=153us
		movwf	temp2
del2		decfsz	temp2,1
		goto	del2
		decfsz	temp1,1
		goto	del1
		return


;****** DISPAVSP ******
;Calculate and then display average speed as 99.9 miles/hour

;The average speed is calculated from a snapshot of the 
;total elapsed distance / total elapsed time.
;The snapshot distance is held in Tdistlhi1 and Tdistlo1 but each step 0.1 mile.
;The accumalative time is held in Ttimelo and Ttimehi but each unit is 6mins.
;The resultant division is therefore in units of miles per hour.
 
;Return result in ACCbLO and remainder in ACCcLO
;	ACCb =Tdisthi,Tdistlo/ACCa=Ttimehi,Ttimelo
;	ACCb=dividend	ACCc=remainder 
;Also, to avoid dealing with infinity retrun with a result of zero for the following
;conditions:
; - if the numerator is not greater than the denominator
; - if the demonator is zero  

dispavsp	clrf	spd_tenths	;clear average speed display
		clrf	spd_units
		clrf	spd_tens

;Load the denominator (total time)	
		movf	Ttimelo,w	;load ACCa with time lo byte
		movwf   ACCaLO
		movf	Ttimehi,w	;load ACCa with time hi byte
		movwf	ACCaHI		;

;Load the numerator (total distance snapshot)
		movf	Tdisthi1,w	;fetch distance hi byte
		movwf	ACCbHI		;and load
		movf	Tdistlo1,w	;fetch distance lo byte
		movwf   ACCbLO		;and load
divide		call	D_div		;clear down if invalid division
					;result in ACCb, remainder in ACCc

;load display variables and display
		movf	ACCbLO,w	;fetch average (units only)
		movwf	spd_units	;load units display

;now calculate tenths - load remainder and multiply by 10D
		movlw	D'00'		;load 10D
		movwf	ACCaHI
		movlw	D'10'
		movwf	ACCaLO

		movlw	D'00'		
		movwf	ACCbHI
		movf	ACCcLO,w	;load remainder (lo byte)
		movwf	ACCbLO

		call	D_mpy		;R*10 result in ACCb,ACCc

;now calculate R*10/Ttimehi,Ttimelo
;load numerator
		movf	ACCcHI,w	;ACCc->ACCb
		movwf	ACCbHI
		movf	ACCcLO,w
		movwf	ACCbLO
;load denominator
		movf	Ttimelo,w
		movwf	ACCaLO
		movf	Ttimehi,w
		movwf	ACCaHI

		call	D_div		;result in ACCbLO=R*10/Ttimehi,Ttimelo

		movf	ACCbLO,w
		movwf	spd_tenths 

		call	dispspeed	;display average speed
		return


;                            ****************************
;                            ****** MATHS ROUTINES ******
;                            ****************************

;****** MULTIPLICATION ******
;ACCb(16 bits) * ACCa(16 bits) -> ACCb,ACCc ( 32 bits )
;ACCaLO & ACCaHI ( 16 bits )
;ACCbLO & ACCbHI ( 16 bits )
;32 bit result is in location ( ACCbHI,ACCbLO,ACCcHI,ACCcLO )
D_mpy	call    setup		;results in ACCb(16 msb's) and ACCc(16 lsb's)
mloop   rrf     ACCdHI,f        ;rotate d right
	rrf     ACCdLO,f
	btfsc   STATUS,carry    ;need to add?
	call    D_add
	rrf     ACCbHI,f
	rrf     ACCbLO,f
	rrf     ACCcHI,f
	rrf     ACCcLO,f
	decfsz  temp,f          ;loop until all bits checked
	goto    mloop
	retlw   0
    
;
setup   movlw   .16             ; for 16 shifts
	movwf   temp
	movf    ACCbHI,w          ;move ACCb to ACCd
	movwf   ACCdHI
	movf    ACCbLO,w
	movwf   ACCdLO
	clrf    ACCbHI
	clrf    ACCbLO
	retlw   0

;****** ADDITION / SUBTRACTION ******
;Double Precision Subtraction ( ACCb - ACCa -> ACCb )
D_sub   call    neg_A           ; At first negate ACCa; Then add
; Double Precision  Addition ( ACCb + ACCa -> ACCb )
D_add   movf    ACCaLO,w
	addwf   ACCbLO,f          ;add lsb
	btfsc   STATUS,carry    ;add in carry
	incf    ACCbHI,f
	movf    ACCaHI,w
	addwf   ACCbHI,f            ;add msb
	retlw   0
;
neg_A   comf    ACCaLO,f          ; negate ACCa ( -ACCa -> ACCa )
	incf    ACCaLO,f
	btfsc   STATUS,z
	decf    ACCaHI,f
	comf    ACCaHI,f
	retlw   0

;****** DIVISION ******
;Division : ACCb(16 bits) / ACCa(16 bits) -> ACCb(16 bits) with
;remainder in ACCc (16 bits)
;Includes the following checks and returns: 
;1. if numerator is less then then demoninator return with zero
;2. if demoninator is zero, then return with zero (should be infinity)

D_div		call	storeAB		;put ACCa and ACCb into temp store

;Load the denominator - clear and return if zero	
divchk1		movf	ACCaLO,w	;load ACCa with time lo byte
		btfss	status,z	;zero?
		goto	divchk2		;no, so do next check
		call	clrmaths	;yes, so clear down and return
		return

hizero		movf	ACCaHI,w	;load ACCa with time hi byte
		btfss	status,z	;also zero?
		goto	divchk2		;no, so do next test 
		call	clrmaths	;yes, so clear down and return
		return
				
;Check that the numerator is greater than or equal to the denominator
;Note - values are destroyed and must be reloaded into accumulators
;Load the numerator



;Use 16 bit subtraction to determine if B>A (valid), or B<A (invalid)
divchk2		call	D_sub		;B-A => B 	;greater than?
		btfss	ACCbHI,7	;bit 7 set = -ve
		goto	loaddiv		;reload A and B and do division
		call	clrmaths	;invalid division
		return

loaddiv		call	loadAB			;reload A and B

;Do 16 bit division
	call    setup
	clrf    ACCcHI
	clrf    ACCcLO
dloop   bcf     STATUS,carry
	rlf     ACCdLO,f
	rlf     ACCdHI,f
	rlf     ACCcLO,f
	rlf     ACCcHI,f
	movf    ACCaHI,w
	subwf   ACCcHI,w          ;check if a>c
	btfss   STATUS,z
	goto    nochk
	movf    ACCaLO,w
	subwf   ACCcLO,w        ;if msb equal then check lsb
nochk   btfss   STATUS,carry    ;carry set if c>a
	goto    nogo
	movf    ACCaLO,w        ;c-a into c
	subwf   ACCcLO,f
	btfss   STATUS,carry
	decf    ACCcHI,f
	movf    ACCaHI,w
	subwf   ACCcHI,f
	bsf     STATUS,carry    ;shift a 1 into b (result)
nogo    rlf     ACCbLO,f
	rlf     ACCbHI,f
	decfsz  temp,f            ;loop untill all bits checked
	goto    dloop
	retlw   0



;Store ACCa and ACCb into temporary store
storeAB	movf	ACCaLO,w
	movwf	ACCaLO1
	movf	ACCaHI,w
	movwf	ACCaHI1

	movf	ACCbLO,w
	movwf	ACCbLO1
	movf	ACCbHI,w
	movwf	ACCbHI1
	return

;Load temporary store into ACCa and ACCb 
loadAB	movf	ACCaLO1,w
	movwf	ACCaLO
	movf	ACCaHI1,w
	movwf	ACCaHI

	movf	ACCbLO1,w
	movwf	ACCbLO
	movf	ACCbHI1,w
	movwf	ACCbHI
	return

;Clear down the arithmetic storage
clrmaths	clrf	ACCaLO 		
		clrf	ACCaHI  	
		clrf	ACCbLO  	
		clrf	ACCbHI  	
		clrf	ACCcLO  
		clrf	ACCcHI  	
		clrf	ACCdLO  	
		clrf	ACCdHI  	
		clrf	temp    	

		return


;****** SETMEM ******
;EEPROM write routine
;Enter with data to be sent already loaded into tempee and with 
;wreg pointing to the desired eeprom address 
setmem  movwf EEADR     ;Now copy W into EEADR to set eeprom address
        page1
        bsf EECON1,WREN ;enable write flag
        page0
        movf tempee,W 	;get data value from DATA and hold in W
        movwf EEDATA    ;copy W into eeprom data byte register

MANUAL  page1           ;these next 12 lines are according to
        movlw H'55'     ;Microchip manual dictated factors
        movwf EECON2    ;they cause the action required by
        movlw H'AA'     ;by the eeprom to store the data in EEDATA
        movwf EECON2    ;at the address held by EEADR.
        bsf EECON1,WR   ;set the ``perform write'' flag

CHKWRT  btfss EECON1,4  ;wait until bit 4 of EECON1 is set
        goto CHKWRT
        bcf EECON1,WREN ;disable write
        bcf EECON1,4    ;clear bit 4 of EECON1
        page0
        bcf INTCON,6    ;clear bit 6 of INTCON
        return          ;and return

;****** GETMEM ****** 
;EEPROM read routine
;This routine is entered with W holding the eeprom byte address to be read.
getmem  movwf EEADR     ;Now copy W into EEADR to set eeprom address
        page1           ;
        BSF EECON1,RD   ;enable read flag
        page0
        movf EEDATA,W   ;read eeprom data now in EEDATA into W
        return          ;and return

		end

; _________ end of program __________



