;*****************************************************************************        
;
;   Module:     bootload.inc
;               
;   Author:     Mike Hibbett 
;                                                                  
;   Version:    0.1 08/06/05                                                  
;
;               The bootloader 'application' for the camerawatch2 project.
;               Note that this is a special bootloader, tuned to the camera
;               watch2 hardware. 
;               It contains *all* the code required to initialise the hardware,
;               display messages on the LCD, detect keypresses, interact
;               with a host over RS232 and reprogram the flash - all within 
;               0x200 bytes, since we will be reprogramming all the other 
;               memory locations. For this reason some of the code may be 
;               duplicated with code in the user app.
;
;
;*****************************************************************************        



;*****************************************************************************        
;
;   Macro :  BLDISPLAY
;            Helper macro to write a string to the display
;
;   Input:   Address of String to display
;
;   Output:  W modified, BLDspStr called
;
;*****************************************************************************        
BLDISPLAY   macro   addr
    movlw   HIGH    addr
    movwf   TBLPTRH
    movlw   LOW    addr
    movwf   TBLPTRL
    call    BLDspStr
            endm



;*****************************************************************************        
;
;   Function :  bootloader

;               This routine is always in low memory immediately after the
;               reset and interrupt vectors (along with the remapped vectors.)
;               It provides initial hardware setup, detection of bootloader 
;               activation request. It will never exceed 0x600-0x1C hex bytes
;
;   Input:      None.
;
;   Output:     Returns if bootloader activation is not required.
;
;*****************************************************************************        
Bootloader
    clrf    PORTB 
    clrf    LATB 
    ;   First, delay for 1/2s for things to settle
    ;   We are running at about 40MHz
    movlw   D'50'
    call    UIWait10ms

    ; do initial bootloader hardware setup ( port directions and levels )
    
    movlw   COMP_OFF_VAL
    movwf   CMCON 
    
    movlw   CVRCON_OFF_VAL
    movwf   CVRCON 
    
    movlw   ADC_OFF_VAL
    movwf   ADCON1 

    ; Port directions and levels
    movlw   PORTA_LEVELS
    movwf   PORTA 
    movlw   PORTA_TRIS_VAL
    movwf   TRISA 
    movlw   PORTB_LEVELS
    movwf   PORTB 
    movlw   PORTB_TRIS_VAL
    movwf   TRISB 
    movlw   PORTC_LEVELS
    movwf   PORTC 
    movlw   PORTC_TRIS_VAL
    movwf   TRISC 
      
    ; Now setup the LCD into 4-bit mode, and clear the display
    call    DspPutCmd4Bit

    movlw   D'4'
    call    UIWait10ms

    movlw   0x28
    call    DspPutCmd       ; 4 bit, 2 line, small font        

    movlw   D'4'
    call    UIWait10ms

    movlw   0x28
    call    DspPutCmd       ; repeat, since first time doesn't always work        

    movlw   0x00C
    call    DspPutCmd       ; disp on, no cursor            
    
    call    DspHome
    call    DspClear

    BLDISPLAY WELCOME_L1
    call    DspLine2    
    BLDISPLAY WELCOME_L2
    
    ; Display logo for 1s  at least
    movlw   D'100'
    call    UIWait10ms

    ; Now look to see if a button is pressed
    movf    PORTA, W
    andlw   0x3C
    sublw   0x3C
    btfsc   STATUS, Z
    return  
    
    ; Debounce period
    movlw   D'5'
    call    UIWait10ms
    
    ; Now look again to see if a button is still pressed
    movf    PORTA, W
    andlw   0x3C
    sublw   0x3C
    btfsc   STATUS, Z
    return  

    ; A key was pressed. Update display... 
    call    DspHome
    call    DspClear

    BLDISPLAY   DOWNLOAD_L1
    call    DspLine2    
    BLDISPLAY   DOWNLOAD_L2

    ; ... Wait for key to be released
BL001
    movf    PORTA, W
    andlw   0x3C
    sublw   0x3C
    btfss   STATUS, Z
    bra     BL001  

    ; OK, execute the RS232 Download routine.
    ; We never return from this code.
    
    ; Sync with the PC by exchanging 'X' bytes, followed by a G for Go!
BL002
    movlw   0x01
    call    UIWait10ms
    movlw   'X'
    call    RS232TxByte
    call    RS232RxByte
    movlw   'X'
    subwf   hostByte, W
    btfss   STATUS, Z
    bra     BL002
       
    movlw   0x01
    call    UIWait10ms
    movlw   'G'
    call    RS232TxByte

    ; Then wait for a flash programming packet. This is 
    ; AAD.....DC
    ;  AA == 16 bit address to write to 
    ;  D..D 64 data bytes
    ;  C  check byte
    ; PIC returns Y for ok, C for checkdigit error, F for flash error
    ;  Z for exiting ok
    ; Send address of 0x0000 ( and no data or check byte ) to end transfer
    ; Only addresses above bootloader are accepted. TODO - validate addr

BLProg
    call    RxFlashBlock
    
    ; Termination packet received?
    movlw   'Z'
    subwf   progRxResult, W
    btfsc   STATUS, Z
    bra     BLEnd
    
    ; Error in received packet?
    movlw   'C'
    subwf   progRxResult, W
    btfsc   STATUS, Z
    bra     BLError
    
    call    ProgramFlashBlock
    
    ; Error in received packet?
    movlw   'F'
    subwf   progRxResult, W
    btfsc   STATUS, Z
    bra     BLError2


    ; Echo OK to Host
    movlw   0x01
    call    UIWait10ms
    movlw   'Y'
    call    RS232TxByte

    bra     BLProg
    
BLEnd
    ; Echo exiting to Host
    movlw   0x01
    call    UIWait10ms
    movlw   'Z'
    call    RS232TxByte

    call    DspHome
    call    DspClear

    BLDISPLAY   REBOOT_L1
    call    DspLine2    
    BLDISPLAY   REBOOT_L2
    movlw   D'200'
    call    UIWait10ms
    reset

BLError
    ; Echo Error to Host
    movlw   0x01
    call    UIWait10ms
    movlw   'C'
    call    RS232TxByte
    bra     BLProg

BLError2
    ; Echo Error to Host
    movlw   0x01
    call    UIWait10ms
    movlw   'F'
    call    RS232TxByte
    bra     BLProg
    
    
    

;*****************************************************************************        
;
;   Function :  LCDWriteDelay
;               Provides a small delay when driving the LCD pins
;              
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
LCDWriteDelay
    nop
    nop
    return  
    
    
    
;*****************************************************************************        
;
;   Function :  dspPutCmd4Bit
;               Puts the LCD into 4 bit mode from 8 bit mode
;              
;   Input:      Command to write is in W
;
;   Output:     None
;
;*****************************************************************************        
DspPutCmd4Bit
    bcf     CONTROL, LCD_RW 
    call    LCDWriteDelay
    bcf     CONTROL, LCD_RS 
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E 
    movlw   0x0F
    andwf   CONTROL, F    
    movlw   0x20
    iorwf   CONTROL, F    
    call    LCDWriteDelay
    bcf     CONTROL, LCD_E             
    call    LCDWriteDelay
    return  



;*****************************************************************************        
;
;   Function :  LCDBusy
;               This call blocks until the busy flag in the LCD is clear
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
LCDBusy
    movlw   DATA_BUS_INPUT
    movwf   DATA_BUS_TRIS    
    bcf     CONTROL, LCD_RS      ; Set LCD for command mode
    call    LCDWriteDelay
    bsf     CONTROL, LCD_RW      ; Setup to read busy flag
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E       ; LCD E-line High
    call    LCDWriteDelay
    movf    PORTB, W             ; Read busy flag + DDram address
    andlw   0xF0
    movwf   lcdTmp2    
    bcf     CONTROL, LCD_E       ; LCD E-line Low
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E       ; LCD E-line High
    call    LCDWriteDelay
    swapf   PORTB, W             ; Read busy flag + DDram address
    andlw   0x0F
    addwf   lcdTmp2, F    
    bcf     CONTROL, LCD_E       ; LCD E-line Low    
    movf    lcdTmp2, W 
    andlw   0x80                    ; Check Busy flag, High = Busy
    btfss   STATUS, Z 
    bra     LCDBusy    
    bcf     CONTROL, LCD_RW 
    call    LCDWriteDelay
    movlw   DATA_BUS_OUTPUT
    movwf   DATA_BUS_TRIS 
    return
       
       

;*****************************************************************************        
;
;   Function :  DspPutChar
;               This call writes a character to the current cursor position.
;
;   Input:      Character in W
;
;   Output:     None
;
;*****************************************************************************        
DspPutChar
    movwf   lcdTmp               ; Character to be sent is in W
    call    LCDBusy                 ; Wait for LCD to be ready
    bcf     CONTROL, LCD_RW      ; Set LCD in read mode
    call    LCDWriteDelay
    bsf     CONTROL, LCD_RS      ; Set LCD in data mode
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E       ; LCD E-line High
    call    LCDWriteDelay
    movlw   0x0F
    andwf   CONTROL, F    
    movf    lcdTmp, W 
    andlw   0xF0
    iorwf   CONTROL, F           ; Send data to LCD
    call    LCDWriteDelay
    bcf     CONTROL, LCD_E       ; LCD E-line Low
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E       ; LCD E-line High
    call    LCDWriteDelay
    movlw   0x0F
    andwf   CONTROL, F    
    swapf   lcdTmp, W 
    andlw   0xF0
    iorwf   CONTROL, F           ; Send data to LCD
    call    LCDWriteDelay
    bcf     CONTROL, LCD_E       ; LCD E-line Low
    call    LCDWriteDelay
    return
  


;*****************************************************************************        
;
;   Function :  DspPutCmd
;               Sends a command to the LCD. 
;              
;   Input:      Command to write is in W
;
;   Output:     None
;
;*****************************************************************************        
DspPutCmd
    movwf   lcdTmp              
    call    LCDBusy               
    bcf     CONTROL, LCD_RW 
    call    LCDWriteDelay
    bcf     CONTROL, LCD_RS 
    call    LCDWriteDelay
    bsf     CONTROL, LCD_E
    movlw   0x0F
    andwf   CONTROL, F    
    movf    lcdTmp, W            ; Retrieve command byte, send first nibble
    andlw   0xF0
    iorwf   CONTROL, F 
    call    LCDWriteDelay
    bcf     CONTROL, LCD_E    
    movlw   0x0F
    call    LCDWriteDelay
    andwf   CONTROL, F    
    bsf     CONTROL, LCD_E 
    swapf   lcdTmp, W            ; Retrieve command byte, send second nibble
    andlw   0xF0
    iorwf   CONTROL, F 
    call    LCDWriteDelay
    bcf     CONTROL, LCD_E    
    call    LCDWriteDelay
    return  



;*****************************************************************************        
;
;   Function :  DspClear
;               This call clears the display and returns the cursor position
;               to top left.
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
DspClear
    movlw   0x001
    call    DspPutCmd
    return
    
    

;*****************************************************************************        
;
;   Function :  DspHome
;               This call returns the cursor position to top left.
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
DspHome
    movlw   0x002
    call    DspPutCmd
    return
        


;*****************************************************************************        
;
;   Function :  DspLine2
;               This call positions the cursor at the beginning of the second
;               line
;
;   Input:      None
;
;   Output:     None
;
;*****************************************************************************        
DspLine2
    movlw   0x0c0               ; pos 40h = 01000000
    call    DspPutCmd
    return



;*****************************************************************************        
;
;   Function : UIWait10ms
;              delays for a multiple of 10ms
;
;   Input:     multiple in W
;
;*****************************************************************************        
UIWait10ms
    movwf   delay3 
    
d1ms001
    movlw   D'200'
    call    UIWait50us
    decfsz  delay3, F 
    bra     d1ms001
    return



;*****************************************************************************        
;
;   Function :  waitBitTime
;               Waits for 1 bit time
;               Baud rate is 115200 @ 40MHZ. 1 bit time = 8.68us, say 87 cycles
;
;   Input:      None
;
;   Output:     None
;  
;
;*****************************************************************************        
waitBitTime
    ; 2 + 2 + 2 + (d-1)*(3) + 2
    
    movlw   D'27'           ; 1
    movwf   bitDelay        ; 1

wbt001    
    decfsz  bitDelay,F      ; 1/2
    bra    wbt001           ; 2

    nop    
    return



;*****************************************************************************        
;
;   Function :  waitTxBitTime
;               Waits for 1 bit time, less a constant time of 8 cycles
;               because the loop overhead is 8 cycles
;               Baud rate is 115200 @ 40MHZ 1 bit time = 8.68us, say 87 cycles
;               Therefore we want, including call/return overhead, 79 cycles
;
;   Input:      None
;
;   Output:     None
;  
;
;*****************************************************************************        
waitTxBitTime
    ; 2 + 2 + 2 + (d-1)*(3) + 2
    
    movlw   D'24'           ; 1
    movwf   bitDelay        ; 1
    
wtbt001    
    decfsz  bitDelay,F      ; 1/2
    bra     wtbt001         ; 2
    
    nop
    nop    
    return



;*****************************************************************************        
;
;   Function :  waitHalfBitTime
;               Waits for 1 bit time
;               Baud rate is 38400 @ 20MHz
;
;   Input:      None
;
;   Output:     None
;  
;
;*****************************************************************************        
waitHalfBitTime
;               Waits for 1/2 bit time
;               Baud rate is 115200 @ 40MHZ. 1 bit time = 8.68us, say 87 cycles
;               Therefoe we want, including call/return overhead, 43 cycles
; 2 + 2 + 2 + (d-1)*(3) + 2
    
    movlw   D'12'           ; 1
    movwf   bitDelay        ; 1
    
whbt001    
    decfsz  bitDelay,F      ; 1/2
    bra    whbt001          ; 2
    
    nop
    nop    
    return



;*****************************************************************************        
;
;   Function : UIWait50us
;              delays for a multiple of 50us
;
;   Input:     multiple in W
;              cycles are 2 + 1 + (delay2)*( 1+1 + ((d-1)*3 + 2) + 3)  + 3
;              Need 500 cycles at 40MHz
;                         
;
;*****************************************************************************        
UIWait50us             ; 2     ( the call itself )
    movwf   delay2      ; 1  

d1us002
    movlw   D'165'      ; 1          
    movwf   delay1      ; 1          
    
d1us001      
    decfsz  delay1, F   ; 1 or 2        
    bra     d1us001     ; 2               
    
    decfsz  delay2, F   ; 1 or 2
    bra     d1us002     ; 2
    return              ; 2        



;*****************************************************************************        
;
;   Function :  RS232TxByte
;               Sends byte in w over the rs232 interface
;               Baud rate is 115200 @ 40MHz
;   Input:      byte in W
;
;   Output:     None
;
;*****************************************************************************        
RS232TxByte
    movwf   hostByte 
    
    movlw   0x08
    movwf   bitCount 
    
    ; start bit
    bcf     LATA, TX_DATA 

rt001    
    call    waitTxBitTime             ; wait 8.7us - 8 cycles
    
    rrcf    hostByte,F 
    btfss   STATUS, C 
    bcf     LATA, TX_DATA 
    btfsc   STATUS,C 
    bsf     LATA, TX_DATA 
    decfsz  bitCount,F 
    bra     rt001
    
    call    waitTxBitTime             ; wait 8.7us - 8 cycles
    nop
    nop
    nop
    nop
    nop
    nop
    
    ; stop bit
    bsf     LATA, TX_DATA
    call    waitTxBitTime             ; wait 8.7us - 8 cycles
    nop
    nop
    nop
    nop
    nop
    nop    
    return
    


;*****************************************************************************        
;
;   Function :  RS232RxByte
;               Receives a byte over the rs232 interface
;               Baud rate is 115200 @ 40MHz
;   Input:      None
;
;   Output:     byte in hostByte
;
;               NOTE: This is a blocking call
;
;*****************************************************************************        
RS232RxByte
    movlw   0x08
    movwf   bitCount 
    clrf    hostByte 
    
re002
    btfsc   PORTA, RX_DATA 
    bra     re002
   
    call    waitBitTime         ; Past the start bit
    call    waitHalfBitTime     ; to middle of first bit
   
re003
    bsf     STATUS,C 
    btfss   PORTA, RX_DATA 
    bcf     STATUS,C 
    rrcf    hostByte, F 
    call    waitTxBitTime       ; wait 1 bit time - 8 cycles
    decfsz  bitCount,F 
    bra     re003        
    return
 


;*****************************************************************************        
;
;   Function :  RxFlashBlock
;               Receives a 64 byte data block over RS232
;               Baud rate is 115200 @ 40MHz
;
;   Input:      None. 
;
;   Output:     progAddHigh, progAddLow, progRxResult, prog buffer modified
;
;               NOTE: Returns result in progRxResult:
;               'Y' Block received ok
;               'C' Check byte error
;               'Z' Termination requested
;
;               Received data is not echoed ( a result byte is echoed later)
;
;*****************************************************************************        
RxFlashBlock
    call    RS232RxByte
    movf    hostByte, W
    movwf   progRxCheck     ; Init Check digit
    movwf   progAddHigh   
    call    RS232RxByte
    movf    hostByte, W
    iorwf   progRxCheck,W   ; Zero address sent?
    btfsc   STATUS,Z
    bra     RXF001          ; Yes - exit now, no more data is sent
    movf    hostByte, W
    movwf   progAddLow
    addwf   progRxCheck, F
    
    ; setup pointer to where received data will go
    movlw   0x40
    movwf   FSR0L
    clrf    FSR0H

RXF000
    call    RS232RxByte
    movf    hostByte, W
    movwf   POSTINC0
    addwf   progRxCheck, F
    btfss   FSR0L, 7
    bra     RXF000
    
    ; All bytes received. Now get Check byte, and compare with ours
    call    RS232RxByte
    movf    hostByte, W
    subwf   progRxCheck, W
    btfss   STATUS, Z
    bra     RXF002          ; CheckSum Error 

    ; No errors    
    movlw   'Y'
    movwf   progRxResult
    return
    
    ; Checksum Error
RXF002
    movlw   'C'
    movwf   progRxResult
    return
           
    
RXF001
    movlw   'Z'
    movwf   progRxResult
    return



;*****************************************************************************        
;
;   Function :  ProgramFlashBlock
;               Programs a 64 byte data block into the flash
;               
;   Input:      None
;
;   Output:     progAddHigh, progAddLow, progRxResult, prog buffer modified
;
;               NOTE: Returns result in progRxResult:
;               'Y' Programmed OK
;               'F' Flash programming error
;
;               A result byte is echoed to the host later on.
;
;*****************************************************************************        
ProgramFlashBlock

    ; First, erase the block requested
    clrf    TBLPTRU
    movf    progAddHigh, W
    movwf   TBLPTRH
    movf    progAddLow, W
    movwf   TBLPTRL
    
    bsf     EECON1, EEPGD
    bcf     EECON1, CFGS
    bsf     EECON1, WREN
    bsf     EECON1, FREE
    bcf     INTCON, GIE
    
    movlw   0x55
    movwf   EECON2
    movlw   0xAA
    movwf   EECON2
    bsf     EECON1, WR              ; Start erase; CPU stalls until done
    
    ; Now program the erased block. We do it in 2, 32byte writes
    
    tblrd*-                         ; Dummy read decrement
    
    movlw   0x40
    movwf   FSR0L
    clrf    FSR0H
    
    movlw   D'32'
    movwf   progCounter

PFB001
    movf    POSTINC0, W
    movwf   TABLAT
    tblwt+*                         ; inc address then write
    decfsz  progCounter, F
    bra     PFB001
    
    bsf     EECON1, EEPGD
    bcf     EECON1, CFGS
    bsf     EECON1, WREN

    movlw   0x55
    movwf   EECON2
    movlw   0xAA
    movwf   EECON2
    bsf     EECON1, WR              ; Start erase; CPU stalls until done
    bcf     EECON1, WREN
    
    ; Now do next 32 bytes
    movlw   0x60
    movwf   FSR0L
    clrf    FSR0H
    
    movlw   D'32'
    movwf   progCounter

PFB002
    movf    POSTINC0, W
    movwf   TABLAT
    tblwt+*                         ; inc address then write
    decfsz  progCounter, F
    bra     PFB002
    
    bsf     EECON1, EEPGD
    bcf     EECON1, CFGS
    bsf     EECON1, WREN

    movlw   0x55
    movwf   EECON2
    movlw   0xAA
    movwf   EECON2
    bsf     EECON1, WR              ; Start erase; CPU stalls until done
    bcf     EECON1, WREN
    
    movlw   'Y'
    movwf   progRxResult
    
    return


;*****************************************************************************        
;
;   Function :  BLDspStr
;               Write a string to the display at the current cursor position.
;               This does not handle line wrapping
;
;   Input:      TBLPTRH/L setup with string address.
;
;   Output:     None
;
;*****************************************************************************        
BLDspStr
    clrf    TBLPTRU
    
BLD001
    tblrd*+
    movf    TABLAT, W
    bz      BLD000
    call    DspPutChar
    bra     BLD001    
    
BLD000
    return



WELCOME_L1
    DB      "EPE CameraWatch2",0
WELCOME_L2
    DB      "By Mike Hibbett",0
DOWNLOAD_L1
    DB      "Bootloader is",0
DOWNLOAD_L2
    DB      "ready for PC",0
REBOOT_L1
    DB      "Finished loading",0
REBOOT_L2
    DB      "SW resetting...",0
