#include <stdio.h>
#include <pic.h>
#include "usart.h"

// configuration bit settings
__CONFIG(INTIO & WDTDIS & PWRTEN & MCLREN & UNPROTECT & BORDIS & IESODIS & FCMEN);

volatile unsigned long actRPM;
volatile unsigned int tmrStall = 0;
volatile unsigned long prevTime = 0;

void main(void){
	long cmdSpeed, lRPM, lErr;
	int cmdSignal;

	PIE1	= 0b00000000;
	INTCON	= 0b00000000;
	
	// set up the serial port for RS-232 comms so that we get
	// feedback from the motor control when it is running
	OSCCON	= 0b01110000;	// change to 8MHz operation
	ANSEL	= 0b00000001;	// make AN0 analog input
	TXSTA 	= 0b00100100;   // High speed, Async mode, TX enabled, 8-bit
	RCSTA 	= 0b10010000;	// RX enabled, Enable module
	SPBRG 	= 51;			// 9600 baud

	// set up the PWM output module for single direction speed control
	PR2 = 99;				// 20kHz PWM frequency 
	CCP2CON = 0b00001100;	// single output PWM mode
	CCPR2L = 48;			// set for 50% duty cycle (PR / 2)
	TMR2ON = 1;				// turn on the timer and start the PWM
	TRISD2 = 0;				// set pin as output	
	ADCON0 = 0b00000001;	// Analog input on AN0
	
	// set up CCP1 to time the duration of the dark pulse from the opto
	CCP1CON = 0b00000101;	// capture every rising edge
	CCP1IE = 1;
	PEIE = 1;				// enable CCP and peripheral interrupts
	OPTION = 0b00000111;	// T0 prescale 256
	T0IE = 1;				// enable T0
	T1CON = 0b00110001;		// configure T1 for prescale 8
	GIE = 1;				// enable interrupts
	
	TRISB0 = 0;
	cmdSignal = 0;
	
	// main loop reads the ADC value and sets the duty cycle
	while (1) {
		GODONE = 1; 			// start an ADC Conversion
		while (GODONE);			// wait for it to finish
		cmdSpeed = ADRESH;		// read the result
		// scale the result into the range 0-100
		cmdSpeed = (cmdSpeed * 99) / 255;
		printf("%6d,", cmdSpeed);

		// convert counts into percentage of full scale RPM
		// range is ~ 0 to 100
		lRPM = 6010 / actRPM;
		lErr = cmdSpeed - lRPM;
		// add a proportion of the error to the command signal
		cmdSignal += lErr / 2;
		
		// limit the command signal
		if (cmdSignal < 0)
			cmdSignal = 0;
		if (cmdSignal > 100)
			cmdSignal = 100;
		
		// handle stall conditions at slow speed	
		if ((tmrStall > 20) && ( cmdSpeed > 0)) {
			tmrStall = 0;
			cmdSignal = 35;
		}	
			
		printf(" %6d,", lRPM);
		printf(" %6d,", lErr);
		printf(" %6d\r\n", cmdSignal);
		CCPR2L = cmdSignal;
	}
}

// read the capture port and store as a 10-bit result the time for the
// last half revolution
static void interrupt isr(void)
{
	unsigned long newTime;
	
	if (CCP1IF) {
		newTime = (CCPR1H << 2) | (CCPR2L >> 6);
		if (TMR1IF)
			actRPM = newTime + (1024 - prevTime);
		else
			actRPM = newTime - prevTime;
		prevTime = newTime;
		CCP1IF = 0;
		TMR1IF = 0;
		tmrStall = 0;
	}
	
	if (T0IF) {
		RB0 = !RB0;
		tmrStall++;
		T0IF = 0;
	}	
}
