//By Colin Meikle September 1997
//Rev A.0
//Email colin.meikle@virgin.net
//Based on Original code from atmel app note (www.atmel.com)


// AT89C1051 code for/C1051/C1052 programmer.
// Uses fixed delays to determine end of write cycles.
// Delays are independent of system speed.

//compiled with Borland C 4.5

#include <stdio.h>
#include <string.h>
#include <dos.h>
#include "tctimer.h"

#define FALSE	0
#define TRUE	-1

//#define PBASE	0x378			// LPT1 I/O addresses
//#define PBASE	0x278			// LPT2 I/O addresses
//#define PDATA	(PBASE+0)
//#define PSTAT	(PBASE+1)
//#define PCTRL	(PBASE+2)

#define MAXSIZE	2048		// AT89C2051 FLASH size

#define TCVT	(1.19318)		// timer conversion constant
					// frequency over 1E6

// Four-bit function codes (corresponding to P3.6:P3.5:P3.4:P3.3).

#define	WRITE_DATA	0xe
#define	READ_DATA	0xc
#define	WRITE_LOCK_1	0xf
#define	WRITE_LOCK_2	0x3
#define	CHIP_ERASE	0x1
#define	READ_SIGNATURE	0x0

typedef unsigned char     BYTE;
typedef unsigned int      WORD;
typedef unsigned int      BOOLEAN;
typedef unsigned long int BIGINT;

BOOLEAN load_data( char *, BYTE *, int * );
BOOLEAN save_data( char *, BYTE *, int );
void erase( BYTE * );
void program( BYTE *, BYTE *, int );
void xread( BYTE *, BYTE *, int );
BOOLEAN verify( BYTE *, BYTE *, int );
BOOLEAN blank( BYTE * );
void signature( BYTE *);
void lock( BYTE *, int );
void reset( BYTE * );
void set_function( BYTE );
void set_data( BYTE );
BYTE get_data( void );
void enable_address( BYTE * );
void disable_address( BYTE * );
void enable_data( BYTE * );
void disable_data( BYTE * );
void enable_RST( BYTE * );
void disable_RST( BYTE * );
void pulse_RST( BYTE*, int );
void pulse_XTAL1( BYTE*, int );
void pulse( BYTE *, int );
void xdelay( long );


extern void tinit( void );
extern void tend( void );
extern void tread( void );
extern void disable_traps( void );
extern void enable_traps( void );

WORD pctrl, pdata;		// LPTx control and data port addresses
long tcount = 0;		// timer count
int CHIPSIZE=2048;

main(argc, argv)
int argc;
char *argv[];
{
 //	FILE *fptr;
	int fsize;
	int error=0;

	double *time_ptr;
	BYTE pgmdata[MAXSIZE], control = 0;
	char *pch, fname[20];
	WORD far *p_lpt1 = (WORD far *)0x00400008;
	WORD far *p_lpt2 = (WORD far *)0x0040000a;

	if ((argc > 1) && ((pch = strpbrk( argv[1], "12" )) != NULL)) {
		switch (*pch) {
		case '1':			// LPT1
			pdata = *p_lpt1;
			pctrl = *p_lpt1 + 2;
			break;
		case '2':			// LPT2
			pdata = *p_lpt2;
			pctrl = *p_lpt2 + 2;
			break;
		}
		if (pdata == 0) {		// port undefined
			puts( "Specified Parallel Port is not implemented." );
			return( 255 );
		}
	} else {
		puts( "Parallel Port 1 or 2 must be specified on command line." );
//		printf( "Usage: %s <LPT1 | LPT2>\n", argv[0] );
		puts( "Usage: <fname> <LPT1 | LPT2>" );
		return( 255 );
	}


	time_ptr=(double *)malloc(sizeof(double));
	initializetimer();
	 puts("RESET PARRALLEL PORT . READY TO PROGRAM \n \n");
	 reset( &control );


	while (TRUE) {
		//clear screen


		if(CHIPSIZE==2048)
		 puts( "\nChange Device to 1051   1\n");
		else
		 puts( "\nChange Device to 2051   2\n");
		puts ( "Chip Erase\t\tE" );
		puts( "Program Device\t\tP" );
		puts( "Verify against File\tV" );
		puts( "Save to File\t\tS" );
		puts( "Blank Check\t\tB" );
		puts( "Read Signature\t\tR" );
		puts( "Write Lock Bit 1\tL" );
		puts( "Write Lock Bit 2\tM\n" );
		puts( "Exit\t\t\tX\n\n" );

		printf( "Enter selection: " );
		gets( pch );
		*pch |= 0x20;		// convert first char to lower case

		switch (*pch) {

			case '1':
			  CHIPSIZE=1024;
			  break;
			case '2':
			  CHIPSIZE=2048;
			  break;

			case 'e':	// chip erase
				erase( &control );
				break;
			case 'p':	// write chip from file

				printf( "Enter file name: " );
				gets( fname );
				printf("\Enter file format b=bin i=intel-hex ");
				gets(pch);
				*pch |= 0x20;
				error=TRUE;
				if(*pch=='b'){
					fsize = CHIPSIZE;
					if (load_data( fname, pgmdata, &fsize )){
					 puts("ERASING DEVICE");
					 erase( &control ); //erase before program
					 puts("PROGRAMMING DEVICE");
					 program( &control, pgmdata, fsize );
					 error=FALSE;
					}
					else
					{
					 error=TRUE;
					}
				 }
				 if(*pch=='i'){
					fsize = CHIPSIZE;
					 if (load_intel( fname, pgmdata, fsize )){
					 puts("ERASING DEVICE");
					 erase( &control ); //erase before program
					 puts("PROGRAMMING DEVICE");
					  program( &control, pgmdata, fsize );
					  error=FALSE;
					 }
					 else{
					  error=TRUE;
					}
				 }
				if(error==TRUE) {
				  //	_clearscreen( _GCLEARSCREEN );
					puts( "Error opening or reading input data file." );
					puts( "\nPress Enter to continue..." );
					gets( pch );
				}
				break;
			case 'v':	// compare chip contents to file
				printf( "Enter file name: " );
				gets( fname );
				printf("\Enter file format b=bin i=intel-hex ");
				gets(pch);
				*pch |= 0x20;
				error=TRUE;
				if(*pch=='b'){
					fsize = CHIPSIZE;
					if (load_data( fname, pgmdata, &fsize )){
					 if (!verify( &control, pgmdata, fsize )) {
						puts( "\nPress Enter to continue..." );
						gets( pch );
					 }
					 error=FALSE;
					}
					else
					{
					 error=TRUE;
					}
				 }
				 if(*pch=='i'){
					fsize = CHIPSIZE;
					 if (load_intel( fname, pgmdata, fsize )){
					  if (!verify( &control, pgmdata, fsize )) {
						puts( "\nPress Enter to continue..." );
						gets( pch );
					  }
					  error=FALSE;
					 }
					 else{
					  error=TRUE;
					}
				 }
				if(error==TRUE){
				  //	_clearscreen( _GCLEARSCREEN );
					puts( "Error opening or reading input data file." );
					puts( "\nPress Enter to continue..." );
					gets( pch );
				}
				break;
			case 's':	// save chip contents to file
				printf( "Enter file name: " );
				gets( fname );
				xread( &control, pgmdata, CHIPSIZE );
				if (!save_data( fname, pgmdata, CHIPSIZE )) {
				  //	_clearscreen( _GCLEARSCREEN );
					puts( "Error opening or reading output data file." );
					puts( "\nPress Enter to continue..." );
					gets( pch );
				}
				break;
			case 'b':	// verify blank chip
			  //	_clearscreen( _GCLEARSCREEN );
				if (blank( &control ))
					puts( "Device is blank" );
				else
					puts( "Device is not blank" );
				puts( "\nPress Enter to continue..." );
				gets( pch );
				break;
			case 'r':	// read signature bytes
				signature( &control);
				puts( "\nPress Enter to continue..." );
				gets( pch );
				break;
			case 'l':	// write lock bit 1
				lock( &control, 1 );
				break;
			case 'm':	// write lock bit 2
				lock( &control, 2 );
				break;
  //			case 'x':	// exit program
			default:
				restoretimer();
				return( 0 );
		}
	}
}


//	Read data from indicated file into specified array.
//	If file is smaller than specified byte count, read entire file
//	and change byte count to reflect smaller file size. Encountering
//	end-of-file before the count is satisfied is not an error.
//	If file is larger than specified byte count, read only the
//	specified number of bytes.

BOOLEAN load_data( fname, store, sptr )
char fname[];
BYTE store[];
int *sptr;
{
	FILE *fptr;
	int nbytes;

	if ((fptr = fopen( fname, "rb" )) == NULL)
		return( FALSE );		// file open failed
	nbytes = fread( store, 1, *sptr, fptr );
	if (ferror( fptr ))
		return( FALSE );		// file read failed
	if (feof( fptr ))			// check for end-of-file
		*sptr = nbytes;			// reduce byte count
	fclose( fptr );
	return( TRUE );
}


//	Write data from specified array into indicated file.
//	Returns a boolean value indicating success or failure.

BOOLEAN save_data( fname, store, bcount )
char fname[];
BYTE store[];
int bcount;
{
	FILE *fptr;

	if ((fptr = fopen( fname, "wb" )) == NULL)
		return( FALSE );		// file open failed
	if (fwrite( store, 1, bcount, fptr ) != bcount)
		return( FALSE );		// file write failed
	fclose( fptr );
	return( TRUE );
}


//	Clear the chip memory to all ones. Suggested before programming.

void erase( cptr )
BYTE *cptr;
{
	reset( cptr );				// restore defaults
	set_function( CHIP_ERASE );		// select function
	enable_address( cptr );			// enable func, PROG*
	xdelay(10);		// delay 10 uS
	enable_RST( cptr );			// RST=12v
	xdelay(30000);	// delay 30 mS RST rise->PROG*
	pulse( cptr, 10000 );			// apply 10 mS PROG* pulse
	xdelay(10);		// delay PROG*->adr/data
	reset( cptr );				// restore defaults
	xdelay(30000);	// delay 30 mS for RST fall
}


//	Program the chip with the contents of the specified data array.
//	The specified byte count may be less than the number of bytes in
//	the array or the chip. Programming always begins at address zero.
//	The maximum programming time is allotted to each byte write.
//	No attempt is made to determine if a byte programs correctly or
//	if it is already programmed with the desired value.

void program( cptr, data, count )
BYTE *cptr, data[];
int count;
{
	WORD addr;
	int dot;

	printf("\nProgramming %d bytes...\n",count);
	reset( cptr );				// restore defaults
	pulse_RST( cptr, 10 );			// reset address counter
	set_function( WRITE_DATA );		// select function
	enable_address( cptr );			// enable function, PROG*
	enable_data( cptr );			// enable bus before write
	xdelay(10);		// delay function->RST
	enable_RST( cptr );			// RST=12v
	xdelay(30000);	// delay 15 mS RST rise->PROG*

	for (addr=0; addr<count; addr++) {
		if(dot==64){ //print a dot
		printf(".");
		dot=0;
		}
		dot++;
		set_data( data[addr] );		// apply data
		xdelay(10);	// delay data->PROG*
		pulse( cptr, 100 );		// apply 100 uS PROG* pulse
		xdelay(1500);	// delay 1.5 mS for write
		pulse_XTAL1( cptr, 10 );	// advance address counter
	}

	reset( cptr );				// restore defaults
	xdelay(15000);	// delay 15 mS for RST fall
}


//	Read the contents of the chip into the specified data array.
//	The specified byte count may be less than the number of bytes in
//	the chip. Reading always begins at address zero.

void xread( cptr, data, count )
BYTE *cptr, data[];
int count;
{
	WORD addr;

	reset( cptr );				// restore defaults
	pulse_RST( cptr, 10 );			// reset address counter
	set_function( READ_DATA );		// select function
	enable_address( cptr );			// enable function, PROG*

	for (addr=0; addr<count; addr++) {
		xdelay(10);	// delay address->data
		data[addr] = get_data();
		pulse_XTAL1( cptr, 10 );	// advance address counter
	}

	reset( cptr );				// restore defaults
}


//	Compare the contents of the chip to the specified data array.
//	The specified byte count may be less than the number of bytes in
//	the chip. Verification always begins at address zero. Failures are
//	displayed by address, with actual contents and expected contents.
//	A boolean value is returned indicating verify pass/fail.

BOOLEAN verify( cptr, data, count )
BYTE *cptr, data[];
int count;
{
	BYTE tmp;
	BOOLEAN flag=TRUE;
	WORD addr;

	reset( cptr );				// restore defaults
	pulse_RST( cptr, 10 );			// reset address counter
	set_function( READ_DATA );		// select function
	enable_address( cptr );			// enable function, PROG*

	for (addr=0; addr<count; addr++) {
		xdelay(10);	// delay address->data
		if ((tmp = get_data()) != data[addr]) {
			if (flag) {		// first time only
			//clear screan
			}
			printf( "verify fail at %.4X  is %.2X sb %.2X\n", addr, tmp, data[addr] );
			flag = FALSE;
		}
		pulse_XTAL1( cptr, 10 );	// advance address counter
	}

	reset( cptr );				// restore defaults
	return( flag );
}


//	Determine if the chip is erased. Locations of failures are not
//	determined. A boolean value is returned indicating blank/not blank.

BOOLEAN blank( cptr )
BYTE *cptr;
{
	BYTE tmp;
	BOOLEAN flag = TRUE;			// default is blank
	WORD addr;

	reset( cptr );				// restore defaults
	pulse_RST( cptr, 10 );			// reset address counter
	set_function( READ_DATA );		// select function
	enable_address( cptr );			// enable function, PROG*

	for (addr=0; addr<CHIPSIZE; addr++) {
		xdelay(10);	// delay address->data
		if (get_data() != 0xff)		// compare to erased value
			flag = FALSE;		// not blank
		pulse_XTAL1( cptr, 10 );	// advance address counter
	}

	reset( cptr );				// restore defaults
	return( flag );
}


//	Read signature bytes.
//	The first byte is at address zero, the second at one. When set to
//	1Eh and 11h, respectively, they identify an AT89C1051. The third
//	byte is at address two and indicates 12 volt programming when set
//	to FFh.

void signature( cptr )
BYTE *cptr;
{
	BYTE tmp1, tmp2, tmp3;

	reset( cptr );				// restore defaults
	pulse_RST( cptr, 10 );			// reset address counter
	set_function( READ_SIGNATURE );		// select function
	enable_address( cptr );			// enable func, PROG*
	xdelay(10);		// delay address->data
	tmp1 = get_data();			// read first byte

	pulse_XTAL1( cptr, 10 );		// advance address counter
	xdelay(10);		// delay address->data
	tmp2 = get_data();			// read second byte

	pulse_XTAL1( cptr, 10 );		// advance address counter
	xdelay(10);		// delay address->data
	tmp3 = get_data();			// read second byte

	printf( "signature byte 1:  %.2X\n", tmp1 );
	printf( "signature byte 2:  %.2X\n", tmp2 );
	printf( "signature byte 3:  %.2X\n", tmp3 );
	if(tmp2==0x21){
	 printf("Device is a AT2051\n");
	 CHIPSIZE=2048;
	}
	if(tmp2==0x11){
	 printf("Device is a AT1051\n");
	 CHIPSIZE=1024;
	}
	 if(tmp2==0xff){
	 printf("No Device. Check cables and power\n");
	}
	reset( cptr );				// restore defaults
}


//	Write specified lock bit.

void lock( cptr, lbit )
BYTE *cptr;
int lbit;
{
	reset( cptr );				// restore defaults

	switch (lbit) {				// select function
		case 1:
			set_function( WRITE_LOCK_1 );
			break;
		case 2:
			set_function( WRITE_LOCK_2 );
			break;
	}

	enable_address( cptr );			// enable function, PROG*
	enable_RST( cptr );			// RST=12v
	xdelay(15000);	// delay 15 mS RST rise->PROG*
	pulse( cptr, 100 );			// apply 100 uS PROG* pulse
	xdelay(10);		// delay PROG*->adr/data
	reset( cptr );				// restore defaults
	xdelay(15000);	// delay 15 mS for RST fall
}


//	Return programmer hardware to the passive state.
//	Writes zeros into the programmer control latch, which floats PROG*
//	and the address and data busses and puts 5 volts on RST. Note that
//	XTAL1 is always enabled, but is cleared to zero. The programmer
//	address latches are loaded with zeros, the data latch with ones.
//	NOTE: The programmer will not power up correctly with ones left
//	on the LPTx data bus. Also, latches must be addressed before
//	the strobe is generated to avoid glitches.

void reset( cptr )
BYTE *cptr;
{
	outp( pdata, 0 );		// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	outp( pctrl, 0x0c );		// select low address latch
	outp( pctrl, 0x0d );		// latch data
	outp( pctrl, 0x0c );

	outp( pctrl, 0x00 );		// select high address latch
	outp( pctrl, 0x01 );		// latch data
	outp( pctrl, 0x00 );

	outp( pdata, 0xff );		// set up data

	outp( pctrl, 0x04 );		// select data latch
	outp( pctrl, 0x05 );		// latch data
	outp( pctrl, 0x04 );

//	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data

	*cptr = 0;			// save control latch value
}


//	Write specified value to the programmer function latch.
//	The function latch is shared with the high order address lines,
//	which are not used and are written to ones. Only the three
//	least significant bits of the function are written.

void set_function( func )
BYTE func;
{
	outp( pdata, ((func << 5) | 0x1f) );
					// set up low three bits of function
	outp( pctrl, 0x00 );		// select high address latch
	outp( pctrl, 0x01 );		// latch data
	outp( pctrl, 0x00 );

	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data
}


//	Write specified value into the programmer data latch.

void set_data( outdata )
BYTE outdata;
{
	outp( pdata, outdata );		// set up output data

	outp( pctrl, 0x04 );		// select data latch
	outp( pctrl, 0x05 );		// latch data
	outp( pctrl, 0x04 );

//	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data
}


//	Return data present on programmer data lines.
//	Does not first disable programmer output data latch.
//	Some parallel interface cards allow the output data buffer
//	to be disabled by setting high bit five of the control register.
//	If the card does not support this feature, the interface and
//	programmer data buffers must duke it out.

BYTE get_data( void )
{
	BYTE tmp;

	outp( pdata, 0xff );		// set LPTx port data high
	outp( pctrl, 0x24 );		// disable LPTx port data driver
	outp( pctrl, 0x26 );		// enable read data buffer

	xdelay(10);	// delay 10 uS
	tmp = inp( pdata );		// get data

	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data

	return( tmp );
}


//	Enable outputs of programmer address and function latches.
//	Note that PROG* (P3.2) is also enabled.

void enable_address( cptr )
BYTE *cptr;
{
	outp( pdata, (*cptr |= 0x10) );	// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data
}


//	Disable outputs of programmer address latches.
//	Note that PROG* (P3.2) is also disabled.

void disable_address( cptr )
BYTE *cptr;
{
	outp( pdata, (*cptr &= ~0x10) );	// set up data
	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );
	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data
}


//	Enable output of programmer data latch.

void enable_data( cptr )
BYTE *cptr;
{
	outp( pdata, (*cptr |= 0x20) );	// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data
}


//	Disable output of programmer data latch.

void disable_data( cptr )
BYTE *cptr;
{
	outp( pdata, (*cptr &= ~0x20) );	// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data
}


//	Enable 12 volts on RST.
//	Note that RST will not immediately reach the specified value.

void enable_RST( cptr )
BYTE *cptr;
{
	outp( pdata, (*cptr |= 0x80) );	// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data
}


//	Return RST to 5 volts.
//	Note that RST will not immediately reach the specified value.

void disable_RST( cptr )
BYTE *cptr;
{
	outp( pdata, (*cptr &= ~0x80) );	// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data
}


//	Generate low-going pulse on RST of specified duration.
//	Time should be specified in microseconds.

void pulse_RST( cptr, time )
BYTE *cptr;
int time;
{
	outp( pdata, (*cptr |= 0x40) );		// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	xdelay(time);	// wait specified count

	outp( pdata, (*cptr &= ~0x40) );	// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data
}


//	Generate high-going pulse on XTAL1 of specified duration.
//	Time should be specified in microseconds.

void pulse_XTAL1( cptr, time )
BYTE *cptr;
int time;
{
	outp( pdata, (*cptr |= 0x02) );		// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	xdelay(time);	// wait specified count

	outp( pdata, (*cptr &= ~0x02) );	// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data
}


//	Generate low-going pulse on PROG* of specified duration.
//	Time should be specified in microseconds.

void pulse( cptr, time )
BYTE *cptr;
int time;
{
	outp( pdata, (*cptr |= 0x04) );		// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	xdelay(time);	// wait specified count

	outp( pdata, (*cptr &= ~0x04) );	// set up data

	outp( pctrl, 0x08 );		// select control latch
	outp( pctrl, 0x09 );		// latch data
	outp( pctrl, 0x08 );

	outp( pctrl, 0x04 );		// control signals inactive
	outp( pdata, 0 );		// clear data
}

void xdelay( del )
long del;
{
long start_time;

	del=del*1.2;   //timer resolution

	start_time=readtimer();
	while((readtimer()-start_time)<del){
	//printf(".");
	}
}

